(function() {
    const { createElement: e, Fragment } = wp.element;
    const { Button, Dropdown, MenuGroup, MenuItem } = wp.components;
    const { __ } = wp.i18n;
    const { withSelect, withDispatch } = wp.data;
    const { compose } = wp.compose;
    
    // Create the AI Enhancer Button component
    const AIEnhancerButton = compose([
        withSelect((select) => {
            const { getBlocks } = select('core/block-editor');
            const { getCurrentPostId } = select('core/editor');
            
            return {
                blocks: getBlocks(),
                postId: getCurrentPostId()
            };
        }),
        withDispatch((dispatch) => {
            const { replaceBlocks } = dispatch('core/block-editor');
            const { createNotice } = dispatch('core/notices');
            
            return {
                replaceBlocks,
                createNotice
            };
        })
    ])((props) => {
        const { blocks, postId, replaceBlocks, createNotice } = props;
        
        const handleEnhancement = (type) => {
            // Get current content from all blocks
            const content = wp.blocks.serialize(blocks);
            
            if (!content || content.trim() === '') {
                createNotice('error', 'Please add some content to your post before enhancing it.');
                return;
            }
            
            // Show loading notice
            const loadingNotice = createNotice('info', 'Enhancing content with AI...', {
                isDismissible: false
            });
            
            // Make AJAX request
            wp.apiFetch({
                path: '/wp-admin/admin-ajax.php',
                method: 'POST',
                data: {
                    action: 'ai_enhance_content',
                    nonce: aiEnhancer.nonce,
                    post_id: postId,
                    content: content,
                    enhancement_type: type
                }
            }).then((response) => {
                // Remove loading notice
                dispatch('core/notices').removeNotice(loadingNotice.notice.id);
                
                if (response.success) {
                    // Parse enhanced content back into blocks
                    const enhancedBlocks = wp.blocks.parse(response.data.content);
                    
                    // Replace all blocks with enhanced ones
                    replaceBlocks(blocks.map(block => block.clientId), enhancedBlocks);
                    
                    // Show success notice
                    createNotice('success', response.data.message);
                } else {
                    createNotice('error', 'Error: ' + (response.data || 'Unknown error occurred'));
                }
            }).catch((error) => {
                // Remove loading notice
                dispatch('core/notices').removeNotice(loadingNotice.notice.id);
                
                console.error('AI Enhancement Error:', error);
                createNotice('error', 'Enhancement failed. Please try again.');
            });
        };
        
        return e(Dropdown, {
            className: 'ai-blogger-tools-dropdown',
            contentClassName: 'ai-blogger-tools-dropdown__content',
            position: 'bottom left',
            renderToggle: ({ isOpen, onToggle }) =>
                e(Button, {
                    onClick: onToggle,
                    'aria-expanded': isOpen,
                    className: 'ai-blogger-tools-button',
                    style: {
                        background: '#0073aa',
                        color: 'white',
                        border: 'none',
                        borderRadius: '3px',
                        padding: '6px 12px',
                        marginLeft: '8px'
                    }
                }, [
                    e('span', { 
                        className: 'dashicons dashicons-admin-generic',
                        style: { marginRight: '5px' }
                    }),
                    'Content AI Tools',
                    e('span', { 
                        className: 'dashicons dashicons-arrow-down-alt2',
                        style: { marginLeft: '5px', fontSize: '12px' }
                    })
                ]),
            renderContent: () =>
                e(MenuGroup, null, [
                    e(MenuItem, {
                        key: 'enrich',
                        icon: 'edit',
                        onClick: () => handleEnhancement('enrich')
                    }, 'Expand Content Only'),
                    e(MenuItem, {
                        key: 'faq',
                        icon: 'editor-help',
                        onClick: () => handleEnhancement('faq')
                    }, 'Add FAQ Section'),
                    e(MenuItem, {
                        key: 'seo',
                        icon: 'search',
                        onClick: () => handleEnhancement('seo')
                    }, 'Improve SEO')
                ])
        });
    });
    
    // Add the button to the editor header
    const addAIEnhancerButton = () => {
        const editorHeaderToolbar = document.querySelector('.edit-post-header-toolbar');
        if (editorHeaderToolbar && !document.querySelector('.ai-blogger-tools-wrapper-gutenberg')) {
            const buttonContainer = document.createElement('div');
            buttonContainer.className = 'ai-blogger-tools-wrapper-gutenberg';
            buttonContainer.style.display = 'inline-flex';
            buttonContainer.style.alignItems = 'center';
            buttonContainer.style.marginLeft = '8px';
            
            editorHeaderToolbar.appendChild(buttonContainer);
            
            // Render the React component
            wp.element.render(e(AIEnhancerButton), buttonContainer);
        }
    };
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', addAIEnhancerButton);
    } else {
        addAIEnhancerButton();
    }
    
    // Also try to add button when editor is ready
    wp.domReady(addAIEnhancerButton);
    
    // Fallback: try again after a short delay
    setTimeout(addAIEnhancerButton, 1000);
})(); 