<?php
/**
 * AI Content Generation
 * 
 * Handles blog content generation using OpenAI API and related functionality
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Generate blog post once with text and image (AJAX handler)
 */
function my_auto_blog_generate_single_blog() {
    // Security: Check user capabilities
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }

    // Security: Verify nonce
    if (!check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce', false)) {
        wp_send_json_error('Security check failed');
        return;
    }

    // Check subscription status before generating
    if (!sica_can_generate_post()) {
        $subscription_status = sica_check_subscription_status();
        
        if (isset($subscription_status['upgrade_required']) && $subscription_status['upgrade_required']) {
            wp_send_json_error('Monthly limit reached (' . ($subscription_status['posts_limit'] ?? 'N/A') . ' posts). Please upgrade your plan to continue generating blog posts.');
        } elseif (isset($subscription_status['overage_available']) && $subscription_status['overage_available']) {
            wp_send_json_error([
                'message' => 'Monthly limit reached. Purchase additional posts for $' . ($subscription_status['overage_price'] ?? '0') . ' each?',
                'overage_available' => true,
                'overage_price' => $subscription_status['overage_price'] ?? 0
            ]);
        } else {
            wp_send_json_error('Generation limit reached. Please check your subscription status.');
        }
        return;
    }

    // Disable WordPress's default timeout for this AJAX request
    @ini_set('max_execution_time', 0); // No time limit for this specific request
    
    // Set longer timeouts for HTTP requests
    add_filter('http_request_timeout', function($timeout) {
        return 300; // 5 minutes for HTTP requests
    });
    
    if (isset($_POST['single_blog_title'])) {
        $title = sanitize_text_field($_POST['single_blog_title']);
        
        // Generate the blog post (tracking handled within generation)
        generate_and_create_blog_post($title);
    } else {
        wp_send_json_error('No blog title provided');
    }
}
add_action('wp_ajax_generate_single_blog', 'my_auto_blog_generate_single_blog');

/**
 * Create post from content planner
 */
function create_post_from_content_planner() {
    $publish_immediately = get_option('publish_immediately', 0);
    $titles = get_option('my_auto_blog_title');
    $default_author_id = get_option('my_auto_blog_author', 1);

    // Check and notify if no titles are available before attempting to process
    if (empty($titles)) {
        send_notification_email('titles_empty');
        sica_log('No titles available to create a post.', 'warning');
        return;  // Exit the function as there's nothing to process
    }

    // Convert the titles string into an array, splitting by newline
    $titles = explode("\n", $titles);

    if (!empty($titles)) {
        $title = array_shift($titles);  // Get the first title and remove it from the list
        $generated_content = generate_blog_content_with_openai($title);
        $image_url = generate_image_with_dalle($title);
        $post_status = $publish_immediately ? 'publish' : 'draft';

        $new_post = array(
            'post_title'    => wp_strip_all_tags($title),
            'post_content'  => $generated_content,
            'post_status'   => $post_status,
            'post_author'   => $default_author_id,
        );

        $post_id = wp_insert_post($new_post);
        if (!is_wp_error($post_id)) {
            // Attempt to attach the featured image
            if ($image_url) {
                $image_id = media_sideload_image($image_url, $post_id, '', 'id');
                if (!is_wp_error($image_id)) {
                    set_post_thumbnail($post_id, $image_id);
                } else {
                    sica_log('Error attaching image to post: ' . $image_id->get_error_message(), 'error');
                }
            } else {
                sica_log('Error: Image URL not found.', 'warning');
            }

            // Update the title list in the database (title already removed by array_shift)
            update_option('my_auto_blog_title', implode("\n", $titles));
            
            // Auto-generate more titles if queue is running low
            auto_generate_titles_if_needed();

            // Send an email notification about the created post
            sica_log('[SICA Manual] Sending post creation notification via manual generation', 'info');
            send_notification_email('post_created', [
                'post_url' => get_permalink($post_id),
                'post_title' => get_the_title($post_id),
                'post_id' => $post_id,
                'mode' => 'manual' // Manual content generation
            ]);
        } else {
            sica_log('Error creating post: ' . $post_id->get_error_message(), 'error');
        }
    } else {
        // This block might be redundant now but let's ensure all bases are covered
        send_notification_email('titles_empty');
        sica_log('No titles available after attempting to process a post.', 'warning');
    }
}

/**
 * Main function to generate and create a complete blog post
 * Now uses server-side async processing to bypass WordPress hosting timeout limits
 */
function generate_and_create_blog_post($title, $is_cron = false) {
    // Set extended execution time for complex blog generation
    @ini_set('max_execution_time', 900); // 15 minutes (for polling)
    @ini_set('memory_limit', '512M'); // Increase memory for image processing

    sica_log('Starting blog generation for title: ' . $title, 'info');

    try {
        // Reserve/track generation on server (source of truth)
        $generation_id = null;
        $using_fallback = false;

        sica_log('Step 1: Calling sica_track_post_generation', 'info', array('title' => $title));

        try {
            $generation_id = sica_track_post_generation($title, false, null);

            sica_log('Step 1 complete: sica_track_post_generation returned', 'info', array(
                'generation_id' => $generation_id ?? 'null',
                'type' => gettype($generation_id)
            ));

            // Check if we got a fallback response
            if ($generation_id === 'fallback_local') {
                $using_fallback = true;
                $generation_id = null;
                sica_log('Using local fallback due to server unavailability - customer benefit approach', 'warning');
            }
        } catch (Exception $e) {
            sica_log('Step 1 FAILED: Usage reservation error', 'error', array(
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ));
            // On tracking errors, allow generation (customer benefit)
            $using_fallback = true;
            sica_log('Server tracking failed, proceeding with local fallback', 'warning');
        }

        // Get business details
        $business_name = get_bloginfo('name');
        $business_category = get_option('my_auto_blog_business_category', '');
        $service_area = get_option('my_auto_blog_service_area', '');
        $products_services = get_option('my_auto_blog_products_services', '');

        // =============================================================================
        // TRY ASYNC MODE FIRST (bypasses WordPress hosting timeout limits)
        // =============================================================================
        if (function_exists('sica_start_async_generation') && function_exists('sica_wait_for_job_completion')) {
            sica_log('Attempting server-side async generation', 'info');

            // Build content prompt
            $content_prompt = build_content_prompt($title, $business_name, $business_category, $service_area, $products_services);

            // Build image prompt
            $image_prompt = build_image_prompt($title, $business_category, $service_area);
            $image_model = get_option('sica_image_model', 'gpt-image-1');

            // Start async job
            $async_result = sica_start_async_generation($content_prompt, $title, $image_prompt, $image_model, $generation_id);

            if (!is_wp_error($async_result) && !empty($async_result['job_id'])) {
                sica_log('Async job started, polling for completion', 'info', array(
                    'job_id' => $async_result['job_id']
                ));

                // Wait for job to complete (server processes in background, we just poll)
                $job_result = sica_wait_for_job_completion($async_result['job_id'], 600, 5);

                if ($job_result['ok'] && $job_result['status'] === 'completed') {
                    sica_log('Async job completed successfully', 'info');

                    // Extract content from job result
                    $content = '';
                    if (!empty($job_result['content_result']['choices'][0]['message']['content'])) {
                        $content = $job_result['content_result']['choices'][0]['message']['content'];
                    }

                    // Extract image URL from job result
                    $image_url = null;
                    if (!empty($job_result['image_result'])) {
                        $image_data = $job_result['image_result'];
                        if (!empty($image_data['data'][0]['url'])) {
                            $image_url = $image_data['data'][0]['url'];
                        } elseif (!empty($image_data['data'][0]['b64_json'])) {
                            // Convert base64 to WordPress URL
                            if (function_exists('save_base64_image_to_media_library')) {
                                $image_url = save_base64_image_to_media_library($image_data['data'][0]['b64_json'], $title);
                            }
                        }
                    }

                    // Now create the WordPress post (fast local operation)
                    return create_post_from_async_results($title, $content, $image_url, $is_cron, $generation_id, $using_fallback);
                } else {
                    // Async job failed - log and fall through to sync mode
                    $error_msg = $job_result['error_message'] ?? $job_result['error'] ?? 'Unknown async error';
                    sica_log('Async job failed, falling back to sync mode', 'warning', array(
                        'error' => $error_msg,
                        'status' => $job_result['status'] ?? 'unknown'
                    ));
                }
            } else {
                // Async start failed - fall through to sync mode
                $error_msg = is_wp_error($async_result) ? $async_result->get_error_message() : 'Failed to start async job';
                sica_log('Async start failed, falling back to sync mode', 'warning', array('error' => $error_msg));
            }
        }

        // =============================================================================
        // FALLBACK: Synchronous mode (original behavior, may timeout on some hosts)
        // =============================================================================
        sica_log('Using synchronous generation (fallback)', 'info');

        sica_log('About to call generate_blog_content_with_openai', 'info', array(
            'title' => $title,
            'generation_id' => $generation_id,
            'using_fallback' => $using_fallback ? 'yes' : 'no',
            'business_name' => $business_name
        ));

        // Generate initial content with all required parameters (including generation_id for validation skip)
        $content = generate_blog_content_with_openai(
            $title,
            $business_name,
            $business_category,
            $service_area,
            $products_services,
            $generation_id  // Pass generation_id so generate.php can skip redundant validation
        );

        sica_log('generate_blog_content_with_openai returned', 'info', array(
            'content_length' => strlen($content),
            'content_empty' => empty($content) ? 'YES' : 'no'
        ));

        if (empty($content)) {
            throw new Exception('No content was generated');
        }

        // Add contextual internal links - temporarily disabled
        $content_with_links = $content;

        // Generate image
        $image_url = generate_image_with_dalle($title);

        // Create post
        $post_data = array(
            'post_title'    => wp_strip_all_tags($title),
            'post_content'  => $content_with_links,
            'post_status'   => (int) get_option('publish_immediately', 0) === 1 ? 'publish' : 'draft',
            'post_author'   => get_option('my_auto_blog_author', 1),
        );

        sica_log('Creating post with content length: ' . strlen($content_with_links), 'info');
        $post_id = wp_insert_post($post_data);

        if (is_wp_error($post_id)) {
            throw new Exception('Failed to create post: ' . $post_id->get_error_message());
        }

        // Assign appropriate category
        $category_id = assign_intelligent_category($title, $content_with_links, $business_category);
        if ($category_id) {
            wp_set_post_categories($post_id, array($category_id));
            sica_log('Assigned category ID ' . $category_id . ' to post', 'info');
        }

        // Generate and store schema markup as post meta (not in content)
        $schema_data = generate_article_schema($title, $content_with_links, $business_name, $post_id);
        update_post_meta($post_id, '_ai_blogger_schema', $schema_data);
        
        // Generate and store meta description
        $meta_description = generate_meta_description($content_with_links);
        update_post_meta($post_id, '_ai_blogger_meta_description', $meta_description);

        // Handle image attachment
        if ($image_url) {
            // Ensure we have media functions available
            if (!function_exists('media_sideload_image')) {
                require_once(ABSPATH . 'wp-admin/includes/media.php');
                require_once(ABSPATH . 'wp-admin/includes/file.php');
                require_once(ABSPATH . 'wp-admin/includes/image.php');
            }
            
            $image_id = media_sideload_image($image_url, $post_id, $title, 'id');
            if (!is_wp_error($image_id)) {
                set_post_thumbnail($post_id, $image_id);
                sica_log('Image attached successfully with ID: ' . $image_id, 'info');
            } else {
                sica_log('Error attaching image to post: ' . $image_id->get_error_message(), 'error');
            }
        }

        // Resolve thumbnail URL for UI (small preview)
        $thumb_url = get_the_post_thumbnail_url($post_id, 'thumbnail');

        // On successful creation, remove this title from the backlog queue if present
        $queued_titles_raw = get_option('my_auto_blog_title', '');
        if (!empty($queued_titles_raw)) {
            $queued_titles = array_values(array_filter(array_map('trim', explode("\n", (string) $queued_titles_raw))));
            $idx = array_search($title, $queued_titles, true);
            if ($idx !== false) {
                unset($queued_titles[$idx]);
                update_option('my_auto_blog_title', implode("\n", $queued_titles));
            }
        }

        // Mark completion of generation on server (confirms and finalizes usage)
        try {
            if (!empty($generation_id)) {
                sica_complete_generation($generation_id, true, $post_id);
            } elseif ($using_fallback) {
                // For fallback generations, track locally only
                sica_track_post_generation_local(false);
                sica_log('Fallback generation completed - tracked locally only', 'info');
            }
        } catch (Exception $e) {
            sica_log('Usage completion error: ' . $e->getMessage(), 'error');
        }

        // Mark post as AI-generated for analytics (no personal data stored)
        update_post_meta($post_id, '_ai_blogger_generated', '1');

        // Send notification only if NOT called from cron (scheduler handles its own notifications)
        if (!$is_cron) {
            sica_log('[SICA Manual] Sending post creation notification via manual content creation', 'info');
            send_notification_email('post_created', array(
                'post_url' => get_permalink($post_id),
                'post_title' => get_the_title($post_id),
                'post_id' => $post_id,
                'mode' => 'manual' // Manual content generation
            ));
        } else {
            sica_log('[SICA Scheduler] Skipping manual notification - scheduler will handle its own notification', 'info');
        }

        // Send privacy-compliant analytics ping (if function exists)
        if (function_exists('send_analytics_ping')) {
            send_analytics_ping('blog_created', array(
                'method' => $is_cron ? 'cron' : 'manual',
                'has_image' => !empty($image_url)
            ));
        }

        // Add to unified activity log
        if (function_exists('sica_add_scheduler_log_entry')) {
            sica_add_scheduler_log_entry(array(
                'status' => 'success',
                'message' => 'Post created successfully',
                'source' => $is_cron ? 'scheduled' : 'manual',
                'post_title' => get_the_title($post_id),
                'post_url' => get_permalink($post_id),
                'post_id' => $post_id
            ));
        }

        if (!$is_cron) {
            wp_send_json_success(array(
                'post_id' => $post_id,
                'post_url' => get_permalink($post_id),
                'post_title' => get_the_title($post_id),
                'thumb_url' => $thumb_url
            ));
        } else {
            // For cron calls, return success with post details for logging
            return array(
                'success' => true,
                'post_id' => $post_id,
                'post_url' => get_permalink($post_id),
                'post_title' => get_the_title($post_id)
            );
        }
        return true;

    } catch (Exception $e) {
        sica_log_exception($e, 'Error in generate_and_create_blog_post');
        sica_log('Error details: ' . $e->getMessage(), 'error', array(
            'title' => $title,
            'is_cron' => $is_cron,
            'trace' => $e->getTraceAsString()
        ));
        
        // Provide user-friendly error messages for common issues
        $user_message = $e->getMessage();
        if (strpos($e->getMessage(), 'No valid API token') !== false || strpos($e->getMessage(), 'unauthorized') !== false) {
            $user_message = 'Authentication failed. Please check your license key in Account Settings and try again.';
        } elseif (strpos($e->getMessage(), 'timeout') !== false || strpos($e->getMessage(), 'cURL error 28') !== false) {
            $user_message = 'The blog generation process timed out due to high server load or API delays. Please try again in a few minutes.';
        } elseif (strpos($e->getMessage(), 'memory') !== false) {
            $user_message = 'The server ran out of memory during blog generation. Please contact your hosting provider to increase memory limits.';
        } elseif (strpos($e->getMessage(), 'API') !== false) {
            $user_message = 'There was an issue connecting to the AI service. Please check your API key and try again.';
        }
        
        // On failure, notify server to release any reservation
        try {
            if (!empty($generation_id)) {
                sica_complete_generation($generation_id, false, null, $user_message);
            }
        } catch (Exception $ce) {
            sica_log('Usage completion (failure) error: ' . $ce->getMessage(), 'error');
        }

        // Add failure to unified activity log
        if (function_exists('sica_add_scheduler_log_entry')) {
            sica_add_scheduler_log_entry(array(
                'status' => 'failed',
                'message' => 'Failed to create blog post',
                'source' => $is_cron ? 'scheduled' : 'manual',
                'post_title' => $title,
                'error' => $user_message,
                'error_details' => substr($e->getMessage(), 0, 200)
            ));
        }

        if (!$is_cron) {
            wp_send_json_error($user_message);
        } else {
            // For cron calls, return error details so scheduler can log them
            return array(
                'success' => false,
                'error' => $user_message,
                'exception' => $e->getMessage(),
                'trace' => substr($e->getTraceAsString(), 0, 500) // Limit trace length
            );
        }
        return false;
    }
}

/**
 * Call OpenAI to get written blog content using gpt-4o-search-preview with live web search
 * Enhanced with real-time web search for current, locally-relevant information
 */
function generate_blog_content_with_openai($title, $business_name, $business_category, $business_service_area, $products_services, $generation_id = null) {
    $keywords = get_option('my_auto_blog_keywords', '');

    // Format keywords for better prompt context
    $keywords_arr = explode("\n", $keywords);
    $keywords_str = implode(", ", array_filter(array_map('trim', $keywords_arr)));

    // Use custom master prompt if available, otherwise use default
    $master_prompt_template = get_option('my_auto_blog_master_prompt', '');
    if (empty($master_prompt_template)) {
        $master_prompt_template = get_default_blog_prompt();
    }
    
    // Safe variable replacement - only replace supported variables
    $supported_variables = [
        '{business_category}' => $business_category,
        '{business_name}' => $business_name,
        '{service_area}' => $business_service_area,
        '{title}' => $title,
        '{products_services}' => $products_services,
        '{keywords_str}' => $keywords_str
    ];
    
    $prompt = str_replace(array_keys($supported_variables), array_values($supported_variables), $master_prompt_template);
    
    // Log warning if unsupported variables are found
    if (preg_match_all('/\{([^}]+)\}/', $prompt, $matches)) {
        $remaining_vars = array_unique($matches[0]);
        if (!empty($remaining_vars)) {
            // error_log('AI Blogger: Unsupported variables found in prompt: ' . implode(', ', $remaining_vars));
        }
    }

        // Use SaaS proxy for content generation
    try {
        $customer_email = get_option('sica_customer_email', '');
        $api_token = get_option('sica_api_token', '');

        // CRITICAL: Log to Activity Log as well for visibility on live sites
        if (function_exists('sica_add_scheduler_log_entry')) {
            sica_add_scheduler_log_entry(array(
                'status' => 'pending',
                'message' => 'Calling content generation API...',
                'source' => 'debug',
                'post_title' => $title
            ));
        }

        sica_log('Starting SaaS proxy content generation', 'info', array(
            'title' => $title,
            'customer_email' => !empty($customer_email) ? substr($customer_email, 0, 5) . '***' : 'EMPTY',
            'has_token' => !empty($api_token) ? 'yes (len=' . strlen($api_token) . ')' : 'NO TOKEN',
            'generation_id' => $generation_id ?? 'none'
        ));

        // Use SaaS proxy for content generation (pass generation_id so server trusts reserved usage)
        $generation_result = sica_generate_content_with_subscription($prompt, $title, false, null, $generation_id);

        sica_log('SaaS proxy returned', 'info', array(
            'has_content' => isset($generation_result['content']) && !empty($generation_result['content']) ? 'yes' : 'NO',
            'result_keys' => is_array($generation_result) ? implode(',', array_keys($generation_result)) : 'not_array'
        ));

        // Extract content directly from result
        $content = $generation_result['content'] ?? '';

        if (empty($content)) {
            sica_log('No content returned from SaaS proxy', 'error', array(
                'title' => $title,
                'full_result' => json_encode($generation_result)
            ));
            return '';
        }

        sica_log('Blog content generated successfully using SaaS proxy', 'info', array('content_length' => strlen($content)));
        
        // Clean + normalize content for WordPress
        // 1) Remove stray wrappers / code fences
        $content = str_replace(['<body>', '</body>', '```html', '```'], '', $content);

        // 2) Convert Markdown-style links into minimal superscript references and build a References list
        $cleanUrl = function ($url) {
            $parts = wp_parse_url($url);
            if (!$parts) { return esc_url_raw($url); }
            $scheme   = isset($parts['scheme']) ? $parts['scheme'] . '://' : '';
            $host     = $parts['host'] ?? '';
            $port     = isset($parts['port']) ? ':' . $parts['port'] : '';
            $path     = $parts['path'] ?? '';
            $query    = '';
            if (!empty($parts['query'])) {
                parse_str($parts['query'], $qs);
                // Remove all utm_* parameters and common tracking
                foreach ($qs as $k => $v) {
                    if (preg_match('/^utm_/i', $k) || in_array(strtolower($k), ['utm', 'ref', 'source'])) {
                        unset($qs[$k]);
                    }
                }
                if (!empty($qs)) {
                    $query = '?' . http_build_query($qs);
                }
            }
            $frag     = isset($parts['fragment']) ? '#' . $parts['fragment'] : '';
            return esc_url_raw($scheme . $host . $port . $path . $query . $frag);
        };

        // Collect references in order of appearance
        $references = [];
        $refIndexByUrl = [];
        $content = preg_replace_callback('/\[([^\]]+)\]\((https?:\/\/[^)\s]+)\)/', function ($m) use (&$references, &$refIndexByUrl, $cleanUrl) {
            $url  = $cleanUrl(trim($m[2]));
            if (!isset($refIndexByUrl[$url])) {
                $references[] = ['url' => $url];
                $refIndexByUrl[$url] = count($references); // 1-based index
            }
            $n = $refIndexByUrl[$url];
            // Determine rel: nofollow for commercial; do-follow for gov/edu or same host
            $host = parse_url($url, PHP_URL_HOST);
            $site_host = parse_url(home_url(), PHP_URL_HOST);
            $is_gov_edu = preg_match('/\.(gov|edu)$/i', (string)$host);
            $is_same_host = $host && $site_host && strcasecmp($host, $site_host) === 0;
            $rel = $is_gov_edu || $is_same_host ? 'noopener' : 'nofollow noopener';
            return '<sup id="n' . $n . '"><a href="' . esc_url($url) . '" target="_blank" rel="' . $rel . '" role="doc-noteref" aria-label="Reference ' . $n . '">[' . $n . ']</a></sup>';
        }, $content);

        // Also handle patterns like "Source: https://..." or "(Source: https://...)"
        $content = preg_replace_callback('/\(??\bSource\s*:\s*(https?:\/\/[^)\s]+)\)?/i', function($m) use (&$references, &$refIndexByUrl, $cleanUrl) {
            $url = $cleanUrl(trim($m[1]));
            if (!isset($refIndexByUrl[$url])) {
                $references[] = ['url' => $url];
                $refIndexByUrl[$url] = count($references);
            }
            $n = $refIndexByUrl[$url];
            // Determine rel
            $host = parse_url($url, PHP_URL_HOST);
            $site_host = parse_url(home_url(), PHP_URL_HOST);
            $is_gov_edu = preg_match('/\.(gov|edu)$/i', (string)$host);
            $is_same_host = $host && $site_host && strcasecmp($host, $site_host) === 0;
            $rel = $is_gov_edu || $is_same_host ? 'noopener' : 'nofollow noopener';
            return '<sup id="n' . $n . '"><a href="' . esc_url($url) . '" target="_blank" rel="' . $rel . '" role="doc-noteref" aria-label="Reference ' . $n . '">[' . $n . ']</a></sup>';
        }, $content);

        // Convert bare URLs into superscript references, preserving trailing punctuation
        $content = preg_replace_callback('/(?<!href=")((https?:\/\/[^\s<>"\)\]]+))(\)?|[\.,;:])?/i', function($m) use (&$references, &$refIndexByUrl, $cleanUrl) {
            $raw = $m[2];
            $trail = '';
            // If there is a trailing punctuation captured in group 3, keep it after the superscript
            if (isset($m[3]) && $m[3]) {
                $trail = $m[3];
            }
            $url = $cleanUrl($raw);
            if (!isset($refIndexByUrl[$url])) {
                $references[] = ['url' => $url];
                $refIndexByUrl[$url] = count($references);
            }
            $n = $refIndexByUrl[$url];
            $host = parse_url($url, PHP_URL_HOST);
            $site_host = parse_url(home_url(), PHP_URL_HOST);
            $is_gov_edu = preg_match('/\.(gov|edu)$/i', (string)$host);
            $is_same_host = $host && $site_host && strcasecmp($host, $site_host) === 0;
            $rel = $is_gov_edu || $is_same_host ? 'noopener' : 'nofollow noopener';
            return '<sup id="n' . $n . '"><a href="' . esc_url($url) . '" target="_blank" rel="' . $rel . '" role="doc-noteref" aria-label="Reference ' . $n . '">[' . $n . ']</a></sup>' . $trail;
        }, $content);

        // Remove any empty parentheses left from patterns like ([text](url))
        $content = preg_replace('/\(\s*\)/', '', $content);

        // Remove wrapping parentheses around single superscript references like ([1]) → just <sup>[1]</sup>
        $content = preg_replace(
            '/\(\s*(<sup[^>]*>\s*<a[^>]*role=\"doc-noteref\"[^>]*>\s*\[[0-9]+\]\s*<\/a>\s*<\/sup>)\s*\)/i',
            '$1',
            $content
        );

        // Convert lone [Source] tokens (no URL) into a minimal symbol † to avoid ugly brackets
        $content = preg_replace('/\[\s*Source\s*\]/i', '<sup title="Source">&dagger;</sup>', $content);

        // Instead of visible endnotes, attach hidden citations container for schema extraction
        if (!empty($references)) {
            $citation_urls = array_map(function($r){ return $r['url']; }, $references);
            $citations_attr = esc_attr(implode('|', $citation_urls));
            $content .= "\n\n<div class=\"ai-citations\" data-citations=\"{$citations_attr}\" style=\"display:none\"></div>";
        }

        // Final sanitize for allowed post HTML
        $content = wp_kses_post($content);
        
        // Add the signature at the end if not already present
        $signature = get_option('my_auto_blog_signature', '');
        if (!empty($signature) && strpos($content, $signature) === false) {
            $content .= "\n\n" . $signature;
        }
        
        return $content;
        
    } catch (Exception $e) {
        $msg = $e->getMessage();
        sica_log('SaaS proxy EXCEPTION caught', 'error', array(
            'message' => $msg,
            'title' => $title,
            'trace' => $e->getTraceAsString()
        ));

        // CRITICAL: Log exception to Activity Log for visibility on live sites
        if (function_exists('sica_add_scheduler_log_entry')) {
            sica_add_scheduler_log_entry(array(
                'status' => 'failed',
                'message' => 'Content API exception: ' . substr($msg, 0, 100),
                'source' => 'debug',
                'post_title' => $title,
                'error' => $msg
            ));
        }

        // If server says usage exceeded but we have a reserved generation id, try a one-time non-charging retry
        if (stripos($msg, 'Usage limit exceeded') !== false && !empty($generation_id)) {
            try {
                sica_log('Attempting retry with is_try_again flag', 'info');
                $retry = sica_generate_content_with_subscription($prompt, $title, true, null, $generation_id);
                $content = $retry['content'] ?? '';
                if (!empty($content)) {
                    sica_log('Retry with is_try_again succeeded after usage limit error', 'info');
                    // proceed to cleanup/format below (duplicate of success path)
                } else {
                    sica_log('Retry returned empty content', 'error');
                    return '';
                }
            } catch (Exception $e2) {
                sica_log('Retry after usage limit failed: ' . $e2->getMessage(), 'error');
                return '';
            }
        } else {
            sica_log('Not retrying - returning empty', 'error', array('reason' => 'not_usage_limit_error'));
            return '';
        }
    }
    if ($response_code !== 200) {
        sica_log('Search-preview model returned error code ' . $response_code, 'error', array(
            'error_details' => substr($response_body, 0, 500),
            'title' => $title
        ));
        sica_log('SaaS proxy failed - no direct OpenAI fallbacks allowed', 'error');
        return '';
    }

    $data = json_decode($response_body, true);
    if (isset($data['choices'][0]['message']['content'])) {
        $content = $data['choices'][0]['message']['content'];
        
        // error_log('✅ Blog content generated successfully using gpt-4o-search-preview with web search');
        
        // Clean up any HTML artifacts
        $content = wp_kses_post($content);
        $content = str_replace(['<body>', '</body>', '```html', '```'], '', $content);
        
        // Convert markdown citations to proper WordPress format
        $content = preg_replace(
            '/\(\[([^\]]+)\]\(([^)]+)\)\)/',
            '<sup><a href="$2" target="_blank" rel="noopener" class="source-link">[$1]</a></sup>',
            $content
        );
        
        // Clean up UTM parameters from citations
        $content = preg_replace('/(\?|\&)utm_source=[^&\s"\']*/', '', $content);
        
        // Wrap in article tags for semantic HTML (meta description and schema will be added separately)
        $content = "<article class=\"h-entry\">\n" . $content . "\n</article>";
        
        return $content;
    }

    // error_log('No content generated from OpenAI - check API response structure');
    
    // No direct OpenAI fallbacks allowed - all requests must go through SaaS proxy
    // error_log('SaaS proxy content generation failed - no direct OpenAI fallbacks permitted');
    return '';
}

/**
 * Build the content prompt for generation
 */
function build_content_prompt($title, $business_name, $business_category, $service_area, $products_services) {
    $keywords = get_option('my_auto_blog_keywords', '');
    $keywords_arr = explode("\n", $keywords);
    $keywords_str = implode(", ", array_filter(array_map('trim', $keywords_arr)));

    // Use custom master prompt if available, otherwise use default
    $master_prompt_template = get_option('my_auto_blog_master_prompt', '');
    if (empty($master_prompt_template)) {
        $master_prompt_template = get_default_blog_prompt();
    }

    // Safe variable replacement
    $supported_variables = [
        '{business_category}' => $business_category,
        '{business_name}' => $business_name,
        '{service_area}' => $service_area,
        '{title}' => $title,
        '{products_services}' => $products_services,
        '{keywords_str}' => $keywords_str
    ];

    return str_replace(array_keys($supported_variables), array_values($supported_variables), $master_prompt_template);
}

/**
 * Build the image prompt for generation
 */
function build_image_prompt($title, $business_category, $service_area) {
    // Check if user has custom image prompt override
    $custom_image_prompt = get_option('my_auto_blog_image_style_prompt');

    if ($custom_image_prompt !== false && !empty(trim($custom_image_prompt))) {
        return replace_image_prompt_variables($custom_image_prompt, $title, $business_category, $service_area);
    } else {
        return generate_dynamic_image_prompt($title, $business_category, $service_area);
    }
}

/**
 * Create WordPress post from async generation results
 */
function create_post_from_async_results($title, $content, $image_url, $is_cron, $generation_id, $using_fallback) {
    if (empty($content)) {
        sica_log('create_post_from_async_results: Content is empty', 'error');
        throw new Exception('No content returned from async generation');
    }

    // Clean and process content (same as sync flow)
    $content = process_generated_content($content);

    // Get business details for schema
    $business_name = get_bloginfo('name');
    $business_category = get_option('my_auto_blog_business_category', '');

    // Create post
    $post_status = (int) get_option('publish_immediately', 0) === 1 ? 'publish' : 'draft';
    $post_data = array(
        'post_title'    => wp_strip_all_tags($title),
        'post_content'  => $content,
        'post_status'   => $post_status,
        'post_author'   => get_option('my_auto_blog_author', 1),
    );

    $post_id = wp_insert_post($post_data);

    if (is_wp_error($post_id)) {
        sica_log('create_post_from_async_results: wp_insert_post failed: ' . $post_id->get_error_message(), 'error');
        throw new Exception('Failed to create post: ' . $post_id->get_error_message());
    }

    // Assign appropriate category
    $category_id = assign_intelligent_category($title, $content, $business_category);
    if ($category_id) {
        wp_set_post_categories($post_id, array($category_id));
    }

    // Generate and store schema markup
    $schema_data = generate_article_schema($title, $content, $business_name, $post_id);
    update_post_meta($post_id, '_ai_blogger_schema', $schema_data);

    // Generate and store meta description
    $meta_description = generate_meta_description($content);
    update_post_meta($post_id, '_ai_blogger_meta_description', $meta_description);

    // Handle image attachment
    $image_id = null;
    if ($image_url) {
        // Check if this is already a local WordPress URL (already saved to media library)
        $site_url = get_site_url();
        $upload_dir = wp_upload_dir();
        $is_local_url = (strpos($image_url, $site_url) === 0 || strpos($image_url, $upload_dir['baseurl']) === 0);

        if ($is_local_url) {
            // Image already in media library - find the attachment ID
            $image_id = attachment_url_to_postid($image_url);
            if ($image_id) {
                // Update attachment to associate with this post
                wp_update_post(array('ID' => $image_id, 'post_parent' => $post_id));
                set_post_thumbnail($post_id, $image_id);
                sica_log('Used existing attachment ID: ' . $image_id, 'info');
            } else {
                sica_log('Could not find attachment for local URL: ' . $image_url, 'warning');
            }
        } else {
            // External URL - need to sideload
            if (!function_exists('media_sideload_image')) {
                require_once(ABSPATH . 'wp-admin/includes/media.php');
                require_once(ABSPATH . 'wp-admin/includes/file.php');
                require_once(ABSPATH . 'wp-admin/includes/image.php');
            }

            $image_id = media_sideload_image($image_url, $post_id, $title, 'id');
            if (!is_wp_error($image_id)) {
                set_post_thumbnail($post_id, $image_id);
            } else {
                sica_log('Image sideload failed: ' . $image_id->get_error_message(), 'warning');
                $image_id = null;
            }
        }
    }

    // Get thumbnail URL for UI
    $thumb_url = get_the_post_thumbnail_url($post_id, 'thumbnail');

    // Remove title from backlog queue if present
    $queued_titles_raw = get_option('my_auto_blog_title', '');
    if (!empty($queued_titles_raw)) {
        $queued_titles = array_values(array_filter(array_map('trim', explode("\n", (string) $queued_titles_raw))));
        $idx = array_search($title, $queued_titles, true);
        if ($idx !== false) {
            unset($queued_titles[$idx]);
            update_option('my_auto_blog_title', implode("\n", $queued_titles));
        }
    }

    // Mark completion on server
    try {
        if (!empty($generation_id)) {
            sica_complete_generation($generation_id, true, $post_id);
        } elseif ($using_fallback) {
            sica_track_post_generation_local(false);
        }
    } catch (Exception $e) {
        sica_log('Usage completion error: ' . $e->getMessage(), 'warning');
    }

    // Mark post as AI-generated
    update_post_meta($post_id, '_ai_blogger_generated', '1');

    // Send notification
    if (!$is_cron) {
        send_notification_email('post_created', array(
            'post_url' => get_permalink($post_id),
            'post_title' => get_the_title($post_id),
            'post_id' => $post_id,
            'mode' => 'manual'
        ));
    }

    // Send analytics ping
    if (function_exists('send_analytics_ping')) {
        send_analytics_ping('blog_created', array(
            'method' => $is_cron ? 'cron' : 'manual',
            'has_image' => !empty($image_url),
            'async' => true
        ));
    }

    // Add to activity log
    if (function_exists('sica_add_scheduler_log_entry')) {
        sica_add_scheduler_log_entry(array(
            'status' => 'success',
            'message' => 'Post created successfully (async)',
            'source' => $is_cron ? 'scheduled' : 'manual',
            'post_title' => get_the_title($post_id),
            'post_url' => get_permalink($post_id),
            'post_id' => $post_id
        ));
    }

    if (!$is_cron) {
        wp_send_json_success(array(
            'post_id' => $post_id,
            'post_url' => get_permalink($post_id),
            'post_title' => get_the_title($post_id),
            'thumb_url' => $thumb_url
        ));
    }

    return array(
        'success' => true,
        'post_id' => $post_id,
        'post_url' => get_permalink($post_id),
        'post_title' => get_the_title($post_id)
    );
}

/**
 * Process and clean generated content
 */
function process_generated_content($content) {
    // Clean up HTML artifacts
    $content = str_replace(['<body>', '</body>', '```html', '```'], '', $content);

    // URL cleaner function
    $cleanUrl = function ($url) {
        $parts = wp_parse_url($url);
        if (!$parts) { return esc_url_raw($url); }
        $scheme   = isset($parts['scheme']) ? $parts['scheme'] . '://' : '';
        $host     = $parts['host'] ?? '';
        $port     = isset($parts['port']) ? ':' . $parts['port'] : '';
        $path     = $parts['path'] ?? '';
        $query    = '';
        if (!empty($parts['query'])) {
            parse_str($parts['query'], $qs);
            foreach ($qs as $k => $v) {
                if (preg_match('/^utm_/i', $k) || in_array(strtolower($k), ['utm', 'ref', 'source'])) {
                    unset($qs[$k]);
                }
            }
            if (!empty($qs)) {
                $query = '?' . http_build_query($qs);
            }
        }
        $frag     = isset($parts['fragment']) ? '#' . $parts['fragment'] : '';
        return esc_url_raw($scheme . $host . $port . $path . $query . $frag);
    };

    // Convert markdown links to superscript references
    $references = [];
    $refIndexByUrl = [];
    $content = preg_replace_callback('/\[([^\]]+)\]\((https?:\/\/[^)\s]+)\)/', function ($m) use (&$references, &$refIndexByUrl, $cleanUrl) {
        $url  = $cleanUrl(trim($m[2]));
        if (!isset($refIndexByUrl[$url])) {
            $references[] = ['url' => $url];
            $refIndexByUrl[$url] = count($references);
        }
        $n = $refIndexByUrl[$url];
        $host = parse_url($url, PHP_URL_HOST);
        $site_host = parse_url(home_url(), PHP_URL_HOST);
        $is_gov_edu = preg_match('/\.(gov|edu)$/i', (string)$host);
        $is_same_host = $host && $site_host && strcasecmp($host, $site_host) === 0;
        $rel = $is_gov_edu || $is_same_host ? 'noopener' : 'nofollow noopener';
        return '<sup id="n' . $n . '"><a href="' . esc_url($url) . '" target="_blank" rel="' . $rel . '" role="doc-noteref" aria-label="Reference ' . $n . '">[' . $n . ']</a></sup>';
    }, $content);

    // Handle Source: patterns
    $content = preg_replace_callback('/\(??\bSource\s*:\s*(https?:\/\/[^)\s]+)\)?/i', function($m) use (&$references, &$refIndexByUrl, $cleanUrl) {
        $url = $cleanUrl(trim($m[1]));
        if (!isset($refIndexByUrl[$url])) {
            $references[] = ['url' => $url];
            $refIndexByUrl[$url] = count($references);
        }
        $n = $refIndexByUrl[$url];
        $host = parse_url($url, PHP_URL_HOST);
        $site_host = parse_url(home_url(), PHP_URL_HOST);
        $is_gov_edu = preg_match('/\.(gov|edu)$/i', (string)$host);
        $is_same_host = $host && $site_host && strcasecmp($host, $site_host) === 0;
        $rel = $is_gov_edu || $is_same_host ? 'noopener' : 'nofollow noopener';
        return '<sup id="n' . $n . '"><a href="' . esc_url($url) . '" target="_blank" rel="' . $rel . '" role="doc-noteref" aria-label="Reference ' . $n . '">[' . $n . ']</a></sup>';
    }, $content);

    // Convert bare URLs
    $content = preg_replace_callback('/(?<!href=")((https?:\/\/[^\s<>"\)\]]+))(\)?|[\.,;:])?/i', function($m) use (&$references, &$refIndexByUrl, $cleanUrl) {
        $raw = $m[2];
        $trail = isset($m[3]) && $m[3] ? $m[3] : '';
        $url = $cleanUrl($raw);
        if (!isset($refIndexByUrl[$url])) {
            $references[] = ['url' => $url];
            $refIndexByUrl[$url] = count($references);
        }
        $n = $refIndexByUrl[$url];
        $host = parse_url($url, PHP_URL_HOST);
        $site_host = parse_url(home_url(), PHP_URL_HOST);
        $is_gov_edu = preg_match('/\.(gov|edu)$/i', (string)$host);
        $is_same_host = $host && $site_host && strcasecmp($host, $site_host) === 0;
        $rel = $is_gov_edu || $is_same_host ? 'noopener' : 'nofollow noopener';
        return '<sup id="n' . $n . '"><a href="' . esc_url($url) . '" target="_blank" rel="' . $rel . '" role="doc-noteref" aria-label="Reference ' . $n . '">[' . $n . ']</a></sup>' . $trail;
    }, $content);

    // Cleanup
    $content = preg_replace('/\(\s*\)/', '', $content);
    $content = preg_replace(
        '/\(\s*(<sup[^>]*>\s*<a[^>]*role=\"doc-noteref\"[^>]*>\s*\[[0-9]+\]\s*<\/a>\s*<\/sup>)\s*\)/i',
        '$1',
        $content
    );
    $content = preg_replace('/\[\s*Source\s*\]/i', '<sup title="Source">&dagger;</sup>', $content);

    // Add hidden citations container
    if (!empty($references)) {
        $citation_urls = array_map(function($r){ return $r['url']; }, $references);
        $citations_attr = esc_attr(implode('|', $citation_urls));
        $content .= "\n\n<div class=\"ai-citations\" data-citations=\"{$citations_attr}\" style=\"display:none\"></div>";
    }

    // Final sanitize
    $content = wp_kses_post($content);

    // Add signature
    $signature = get_option('my_auto_blog_signature', '');
    if (!empty($signature) && strpos($content, $signature) === false) {
        $content .= "\n\n" . $signature;
    }

    return $content;
}

/**
 * Validate API key format
 */
function my_auto_blog_validate_api_key($input) {
    // Simple validation or sanitization
    return sanitize_text_field($input);
}

/**
 * Validate and clean titles input
 */
function my_auto_blog_validate_titles($input) {
    // Split the input into lines
    $lines = explode("\n", $input);
    $output = array();
    foreach ($lines as $line) {
        if (!empty($line)) {
            $output[] = sanitize_text_field($line);
        }
    }
    // Return the array back as a newline-separated string
    return implode("\n", $output);
}