<?php
/**
 * Analytics and Notification Utilities
 * 
 * Handles usage tracking, notifications, and API key management
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Track content generation usage
 */
function track_content_generation($type = 'post', $success = true, $details = array()) {
    if (!function_exists('wp_remote_post')) {
        return;
    }
    
    $site_url = home_url();
    $tracking_data = array(
        'action' => 'track_usage',
        'type' => $type,
        'success' => $success,
        'site_url' => $site_url,
        'details' => $details
    );
    
    // Non-blocking request
    wp_remote_post('https://api.sica.ai/track_usage.php', array(
        'timeout' => 0.01,
        'blocking' => false,
        'headers' => array('Content-Type' => 'application/json'),
        'body' => json_encode($tracking_data)
    ));
}

/**
 * Send notification emails for various events
 */
function send_notification_email($type, $details = []) {
    $mode = isset($details['mode']) ? $details['mode'] : 'unknown';
    sica_log('Email notification starting', 'info', array('type' => $type, 'mode' => $mode));

    $notify_emails = get_option('my_auto_blog_notify_emails', '');
    if (empty($notify_emails)) {
        sica_log('Email: No notification emails configured', 'warning');
        return false;
    }
    
    $emails = array_filter(array_map('trim', explode(',', $notify_emails)));
    if (empty($emails)) {
        // error_log('[SICA Email] No valid emails after filtering');
        return false;
    }
    
    $site_name = get_bloginfo('name');
    $site_url = home_url();
    
    // Prepare email content based on notification type
    $subject = '';
    $message = '';
    
    switch ($type) {
        case 'post_created':
            // Use improved email system with proper subject generation
            $args = [
                'post_id' => isset($details['post_id']) ? $details['post_id'] : 0,
                'title' => isset($details['post_title']) ? $details['post_title'] : 'Untitled',
                'permalink' => isset($details['post_url']) ? $details['post_url'] : '',
                'mode' => isset($details['mode']) ? $details['mode'] : 'auto',
                'status' => 'published',
                'site_url' => $site_url,
                'recipient' => ''
            ];

            try {
                $normalized = sica_email_normalize_args($args);
                sica_log('Email: normalized args', 'info', array('post_id' => $normalized['post']['id']));

                $payload = sica_email_build_payload($normalized);
                sica_log('Email: payload built', 'info', array('headline' => $payload['headline']));

                $subject = sica_email_subject($payload);
                sica_log('Email: subject generated', 'info', array('subject' => $subject));

                $message = sica_email_html_template($payload);
                sica_log('Email: HTML template generated', 'info', array('length' => strlen($message)));
            } catch (Exception $e) {
                sica_log('Email: Exception during template generation', 'error', array('error' => $e->getMessage()));
                return false;
            } catch (Error $e) {
                sica_log('Email: Error during template generation', 'error', array('error' => $e->getMessage()));
                return false;
            }
            break;
            
        case 'titles_empty':
            $args = [
                'post_id' => 0,
                'title' => 'Title Queue Empty',
                'mode' => 'auto',
                'status' => 'failed',
                'site_url' => $site_url,
                'recipient' => ''
            ];
            $normalized = sica_email_normalize_args($args);
            $payload = sica_email_build_payload($normalized);
            $payload['headline'] = 'Automated posting paused: Title queue empty';
            $subject = '[Queue Empty] ' . $site_name . ' — Automated posting paused';
            $message = sica_build_titles_empty_email($site_name, $site_url);
            break;
            
        default:
            // error_log('[SICA Email] Unknown notification type: ' . $type);
            return false;
    }
    
    // Multi-tier delivery strategy for maximum reliability
    sica_log('Email: Starting multi-tier delivery', 'info', array('recipients' => count($emails), 'subject' => $subject));
    $result = sica_send_email_multi_tier($emails, $subject, $message, $site_name, $site_url, $details);
    sica_log('Email: Multi-tier delivery result', 'info', array('success' => $result ? 'yes' : 'no'));
    return $result;
}

/**
 * Multi-tier email delivery system for maximum deliverability
 * Tier 1: Server-based email (professional infrastructure)
 * Tier 2: Local SMTP (if properly configured)
 * Tier 3: WordPress wp_mail (with diagnostics)
 */
function sica_send_email_multi_tier($emails, $subject, $message, $site_name, $site_url, $details = []) {
    // error_log('[SICA Email] Using multi-tier delivery strategy');
    
    // Check user preference for email delivery method
    $preferred_method = get_option('sica_email_delivery_method', 'auto');
    // error_log('[SICA Email] Preferred delivery method: ' . $preferred_method);
    
    $delivery_successful = false;
    $last_error = '';
    
    // TIER 1: Try server-based email (if not disabled by user)
    if ($preferred_method === 'auto' || $preferred_method === 'server') {
        // error_log('[SICA Email] TIER 1: Attempting server-based email delivery');
        $server_result = sica_send_via_server($emails, $subject, $message, $site_name, $site_url, $details);
        
        if ($server_result['success']) {
            // error_log('[SICA Email] ✅ TIER 1 SUCCESS: Server-based delivery completed');
            return true;
        } else {
            // error_log('[SICA Email] ❌ TIER 1 FAILED: ' . $server_result['error']);
            $last_error = $server_result['error'];
            
            // If user specifically wants server-only, stop here
            if ($preferred_method === 'server') {
                // error_log('[SICA Email] User preference is server-only, not falling back');
                return false;
            }
        }
    }
    
    // TIER 2: Try local SMTP (if properly configured)
    if ($preferred_method === 'auto' || $preferred_method === 'local') {
        // error_log('[SICA Email] TIER 2: Checking for local SMTP configuration');
        $smtp_configured = sica_detect_smtp_configuration();
        
        if ($smtp_configured) {
            // error_log('[SICA Email] ✅ SMTP detected, attempting local delivery');
            $local_result = sica_send_via_local_smtp($emails, $subject, $message, $site_name, $site_url);
            
            if ($local_result['success']) {
                // error_log('[SICA Email] ✅ TIER 2 SUCCESS: Local SMTP delivery completed');
                return true;
            } else {
                // error_log('[SICA Email] ❌ TIER 2 FAILED: ' . $local_result['error']);
                $last_error = $local_result['error'];
            }
        } else {
            // error_log('[SICA Email] ⚠️ TIER 2 SKIPPED: No SMTP configuration detected');
        }
        
        // If user specifically wants local-only, stop here
        if ($preferred_method === 'local') {
            // error_log('[SICA Email] User preference is local-only, not falling back');
            return false;
        }
    }
    
    // TIER 3: Fallback to WordPress wp_mail with enhanced diagnostics
    if ($preferred_method === 'auto' || $preferred_method === 'wordpress') {
        // error_log('[SICA Email] TIER 3: Fallback to WordPress wp_mail');
        $wp_result = sica_send_via_wordpress_mail($emails, $subject, $message, $site_name, $site_url);
        
        if ($wp_result['success']) {
            // error_log('[SICA Email] ⚠️ TIER 3 SUCCESS: WordPress mail reports success (delivery not guaranteed)');
            return true;
        } else {
            // error_log('[SICA Email] ❌ TIER 3 FAILED: ' . $wp_result['error']);
            $last_error = $wp_result['error'];
        }
    }
    
    // All tiers failed
    // error_log('[SICA Email] ❌ ALL TIERS FAILED: Email delivery unsuccessful');
    // error_log('[SICA Email] Last error: ' . $last_error);
    
    // Run comprehensive diagnosis to help user
    sica_diagnose_email_issues();
    
    return false;
}

/**
 * Capture PHPMailer configuration details for debugging
 */
function sica_capture_phpmailer_details($phpmailer) {
    // error_log('[SICA Email Debug] PHPMailer Configuration:');
    // error_log('[SICA Email Debug] Mailer: ' . $phpmailer->Mailer);
    // error_log('[SICA Email Debug] Host: ' . $phpmailer->Host);
    // error_log('[SICA Email Debug] Port: ' . $phpmailer->Port);
    // error_log('[SICA Email Debug] SMTPAuth: ' . ($phpmailer->SMTPAuth ? 'true' : 'false'));
    // error_log('[SICA Email Debug] SMTPSecure: ' . $phpmailer->SMTPSecure);
    // error_log('[SICA Email Debug] Username: ' . $phpmailer->Username);
    // error_log('[SICA Email Debug] From: ' . $phpmailer->From);
    // error_log('[SICA Email Debug] FromName: ' . $phpmailer->FromName);
}

/**
 * TIER 1: Send email via Sica server infrastructure (most reliable)
 */
function sica_send_via_server($emails, $subject, $message, $site_name, $site_url, $details = []) {
    // error_log('[SICA Email Server] Attempting server-based email delivery');
    
    // Prepare notification data for server API
    $notification_data = array(
        'action' => 'send_notification',
        'type' => 'post_created', // Server handles all notification types
        'site_name' => $site_name,
        'site_url' => $site_url,
        'emails' => $emails,
        'subject' => $subject,
        'message' => $message, // Send HTML directly - no encoding needed
        'details' => $details,
        'timestamp' => time(),
        'plugin_version' => '1.0' // Add version for server-side tracking
    );
    
    // Get API token for authentication
    $api_token = get_option('sica_api_token', '');
    
    $response = wp_remote_post('https://api.sica.ai/notifications.php', array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'X-Sica-Source' => 'plugin-notifications',
            'X-Sica-Token' => $api_token
        ),
        'body' => json_encode($notification_data),
        'timeout' => 15 // Longer timeout for reliability
    ));
    
    if (is_wp_error($response)) {
        return array(
            'success' => false, 
            'error' => 'Server connection failed: ' . $response->get_error_message()
        );
    }
    
    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    $data = json_decode($response_body, true);
    
    if ($response_code === 200 && isset($data['ok']) && $data['ok']) {
        sica_log('Email: Server delivery successful', 'info', array('recipients' => count($emails)));
        return array('success' => true);
    } else {
        $error_msg = "Server returned error (Code: {$response_code})";
        if (isset($data['message'])) {
            $error_msg .= ': ' . $data['message'];
        } elseif (isset($data['error'])) {
            $error_msg .= ': ' . $data['error'];
        }
        sica_log('Email: Server delivery failed', 'error', array('error' => $error_msg, 'response' => substr($response_body, 0, 500)));
        return array('success' => false, 'error' => $error_msg);
    }
}

/**
 * TIER 2: Send email via local SMTP (if configured)
 */
function sica_send_via_local_smtp($emails, $subject, $message, $site_name, $site_url) {
    // error_log('[SICA Email SMTP] Attempting local SMTP delivery');
    
    // Optimize email headers for SMTP delivery
    $headers = sica_get_optimized_email_headers($site_name, $site_url);
    
    $success_count = 0;
    $failed_emails = array();

    foreach ($emails as $email) {
        if (!is_email($email)) {
            $failed_emails[] = $email . ' (invalid)';
            continue;
        }
        
        // Add recipient-specific content
        $personalized_message = sica_personalize_email_message($message, $email);
        
        // Hook to capture detailed SMTP information
        add_action('phpmailer_init', 'sica_capture_smtp_details');
        
        $success = wp_mail($email, $subject, $personalized_message, $headers);
        
        remove_action('phpmailer_init', 'sica_capture_smtp_details');
        
        if ($success) {
            $success_count++;
            // error_log('[SICA Email SMTP] Successfully sent via SMTP to: ' . $email);
        } else {
            $failed_emails[] = $email;
            
            // Get detailed SMTP error
            global $phpmailer;
            if (isset($phpmailer) && $phpmailer->ErrorInfo) {
                // error_log('[SICA Email SMTP] SMTP Error: ' . $phpmailer->ErrorInfo);
            }
        }
    }
    
    if ($success_count > 0) {
        // error_log("[SICA Email SMTP] SMTP delivery completed: {$success_count}/" . count($emails) . " emails sent");
        return array('success' => true);
    } else {
        return array(
            'success' => false, 
            'error' => 'SMTP delivery failed for all recipients: ' . implode(', ', $failed_emails)
        );
    }
}

/**
 * TIER 3: Send email via WordPress wp_mail (fallback)
 */
function sica_send_via_wordpress_mail($emails, $subject, $message, $site_name, $site_url) {
    // error_log('[SICA Email WP] Attempting WordPress wp_mail delivery');
    
    // Optimize email headers
    $headers = sica_get_optimized_email_headers($site_name, $site_url);
    
    $success_count = 0;
    $failed_emails = array();
    
    foreach ($emails as $email) {
        if (!is_email($email)) {
            $failed_emails[] = $email . ' (invalid)';
            continue;
        }
        
        // Add recipient-specific content
        $personalized_message = sica_personalize_email_message($message, $email);
        
        // Log detailed information for debugging
        // error_log('[SICA Email WP] Sending to: ' . $email);
        // error_log('[SICA Email WP] Subject: ' . $subject);
        
        $success = wp_mail($email, $subject, $personalized_message, $headers);
        
        if ($success) {
            $success_count++;
            // error_log('[SICA Email WP] wp_mail() returned TRUE for: ' . $email);
        } else {
            $failed_emails[] = $email;
            // error_log('[SICA Email WP] wp_mail() returned FALSE for: ' . $email);
        }
    }
    
    if ($success_count > 0) {
        // error_log("[SICA Email WP] WordPress mail completed: {$success_count}/" . count($emails) . " emails reported as sent");
        // error_log('[SICA Email WP] WARNING: wp_mail success does not guarantee actual delivery');
        return array('success' => true);
    } else {
        return array(
            'success' => false, 
            'error' => 'WordPress wp_mail failed for all recipients: ' . implode(', ', $failed_emails)
        );
    }
}

/**
 * Detect if SMTP is properly configured
 */
function sica_detect_smtp_configuration() {
    // Check for common SMTP plugins
    $smtp_plugins = array(
        'wp-mail-smtp/wp_mail_smtp.php' => 'WP Mail SMTP',
        'easy-wp-smtp/easy-wp-smtp.php' => 'Easy WP SMTP',
        'post-smtp/postman-smtp.php' => 'Post SMTP',
        'smtp-mailer/main.php' => 'SMTP Mailer'
    );
    
    foreach ($smtp_plugins as $plugin_path => $plugin_name) {
        if (is_plugin_active($plugin_path)) {
            // error_log("[SICA Email Detection] SMTP plugin detected: {$plugin_name}");
            return true;
        }
    }
    
    // Check if PHPMailer is configured for SMTP in wp-config or functions.php
    // This is a basic check - more sophisticated detection could be added
    if (defined('SMTP_HOST') || defined('WPMS_SMTP_HOST')) {
        // error_log('[SICA Email Detection] SMTP constants detected in wp-config');
        return true;
    }
    
    return false;
}

/**
 * Capture detailed SMTP configuration for debugging
 */
function sica_capture_smtp_details($phpmailer) {
    // error_log('[SICA Email SMTP Debug] Detailed SMTP Configuration:');
    // error_log('[SICA Email SMTP Debug] Mailer Type: ' . $phpmailer->Mailer);
    // error_log('[SICA Email SMTP Debug] SMTP Host: ' . $phpmailer->Host);
    // error_log('[SICA Email SMTP Debug] SMTP Port: ' . $phpmailer->Port);
    // error_log('[SICA Email SMTP Debug] SMTP Auth: ' . ($phpmailer->SMTPAuth ? 'enabled' : 'disabled'));
    // error_log('[SICA Email SMTP Debug] SMTP Secure: ' . $phpmailer->SMTPSecure);
    // error_log('[SICA Email SMTP Debug] SMTP Username: ' . $phpmailer->Username);
    // error_log('[SICA Email SMTP Debug] From Address: ' . $phpmailer->From);
}

/**
 * IMPROVED TRANSACTIONAL EMAIL SYSTEM
 * Professional, neutral, and informative email notifications
 */

/**
 * Main function to send post notifications with proper timezone and professional formatting
 */
function sica_send_post_notice($args) {
    $normalized = sica_email_normalize_args($args);
    $payload = sica_email_build_payload($normalized);
    
    // Build email content
    $subject = sica_email_subject($payload);
    $html_message = sica_email_html_template($payload);
    $text_message = sica_email_text_template($payload);
    
    // Send via multi-tier system
    $emails = is_array($args['recipient']) ? $args['recipient'] : [$args['recipient']];
    
    return sica_send_email_multi_tier(
        $emails,
        $subject, 
        $html_message, // Use HTML version for rich formatting
        $normalized['site']['name'],
        $normalized['site']['url'],
        $payload
    );
}

/**
 * Normalize and prepare email arguments with proper timezone handling
 */
function sica_email_normalize_args($args) {
    $site_url = isset($args['site_url']) ? $args['site_url'] : (function_exists('home_url') ? home_url('/') : '');
    $site_name = function_exists('get_bloginfo') ? get_bloginfo('name') : parse_url($site_url, PHP_URL_HOST);
    
    // Get WordPress timezone and format time correctly
    $tz_string = function_exists('wp_timezone_string') ? wp_timezone_string() : 'UTC';
    $tz = new DateTimeZone($tz_string);
    $now = new DateTime('now', $tz);
    
    $post_id = intval(isset($args['post_id']) ? $args['post_id'] : 0);
    $post = ($post_id && function_exists('get_post')) ? get_post($post_id) : null;
    
    $permalink = ($post && function_exists('get_permalink')) ? get_permalink($post) : (isset($args['permalink']) ? $args['permalink'] : '');
    $title = $post ? get_the_title($post) : (isset($args['title']) ? $args['title'] : 'Untitled');
    $status = isset($args['status']) ? $args['status'] : 'published';
    $mode = isset($args['mode']) ? $args['mode'] : 'auto'; // auto|manual|test
    
    $edit_url = ($post && function_exists('get_edit_post_link')) ? get_edit_post_link($post_id, '') : '';
    $settings_url = function_exists('admin_url') ? admin_url('admin.php?page=ai-blogger&tab=scheduler') : ($site_url . '/wp-admin/admin.php?page=ai-blogger&tab=scheduler');
    $dashboard_url = function_exists('admin_url') ? admin_url('admin.php?page=ai-blogger') : ($site_url . '/wp-admin/admin.php?page=ai-blogger');
    $notifications_url = function_exists('admin_url') ? admin_url('admin.php?page=ai-blogger&tab=notifications') : ($site_url . '/wp-admin/admin.php?page=ai-blogger&tab=notifications');

    // Get thumbnail, excerpt, and word count for the new template
    $thumbnail_url = '';
    $excerpt = '';
    $word_count = 0;

    if ($post) {
        // Get featured image
        if (function_exists('get_the_post_thumbnail_url')) {
            $thumbnail_url = get_the_post_thumbnail_url($post_id, 'medium_large') ?: '';
        }

        // Get excerpt
        if (function_exists('get_the_excerpt')) {
            $excerpt = get_the_excerpt($post);
            if (empty($excerpt) && function_exists('wp_trim_words')) {
                $content = $post->post_content;
                $excerpt = wp_trim_words(strip_tags($content), 30, '...');
            }
        }

        // Get word count
        $content = $post->post_content;
        $word_count = str_word_count(strip_tags($content));
    }

    // Extract domain from site URL
    $site_domain = parse_url($site_url, PHP_URL_HOST) ?: '';

    return [
        'site' => [
            'name' => $site_name ?: 'WordPress Site',
            'url'  => rtrim($site_url, '/') ?: '',
            'domain' => $site_domain,
            'timezone' => $tz,
        ],
        'post' => [
            'id'    => $post_id,
            'title' => $title,
            'url'   => $permalink,
            'edit'  => $edit_url,
            'status'=> $status,
            'thumbnail' => $thumbnail_url,
            'excerpt' => $excerpt,
            'word_count' => $word_count,
        ],
        'mode' => $mode,
        'timestamp' => $now,
        'recipient' => trim(isset($args['recipient']) ? $args['recipient'] : ''),
        'support_email' => 'team@sicamarketing.com',
        'send_from' => ['email' => 'team@sicamarketing.com', 'name' => 'Sica Content AI'],
        'settings_url' => $settings_url,
        'dashboard_url' => $dashboard_url,
        'notifications_url' => $notifications_url,
    ];
}

/**
 * Build payload data for email templates
 */
function sica_email_build_payload($p) {
    $is_test = ($p['mode'] === 'test');
    $is_auto = ($p['mode'] === 'auto');
    $is_manual = ($p['mode'] === 'manual');
    
    // Handle different post statuses
    if ($p['post']['status'] === 'failed') {
        $what = 'Post generation failed';
    } elseif ($p['post']['status'] === 'draft') {
        $what = 'New draft created';
    } else {
        $what = 'A new post was published!';
    }
    
    if ($is_test) { 
        $what = 'Test notification email'; 
    }
    
    $when = $p['timestamp']->format('Y-m-d H:i:s T');
    
    // Create more descriptive mode labels
    if ($is_test) {
        $mode_label = 'test email';
    } elseif ($is_auto) {
        $mode_label = 'scheduled automation';
    } elseif ($is_manual) {
        $mode_label = 'manual generation';
    } else {
        $mode_label = 'content generation (' . $p['mode'] . ')';
    }
    
    // Calculate read time from word count (average 238 words per minute)
    $word_count = isset($p['post']['word_count']) ? intval($p['post']['word_count']) : 0;
    $read_time = max(1, round($word_count / 238));

    return [
        'headline' => $what,
        'mode_label' => $mode_label,
        'when' => $when,
        'site' => $p['site'],
        'post' => $p['post'],
        'read_time' => $read_time,
        'settings_url' => $p['settings_url'],
        'dashboard_url' => $p['dashboard_url'],
        'notifications_url' => isset($p['notifications_url']) ? $p['notifications_url'] : $p['settings_url'],
        'support_email' => $p['support_email'],
    ];
}

/**
 * Generate professional, scannable email subject
 * Format: "✓ New post: {title}" for instant recognition
 */
function sica_email_subject($d) {
    $raw_title = isset($d['post']['title']) ? $d['post']['title'] : 'Untitled';
    // Decode HTML entities (e.g., &#8220; → ") for clean display in email subject
    $title = html_entity_decode(sanitize_text_field($raw_title), ENT_QUOTES | ENT_HTML5, 'UTF-8');
    $status = isset($d['post']['status']) ? $d['post']['status'] : 'published';

    // Test emails get clear test prefix
    if (strpos($d['mode_label'], 'test') !== false) {
        return '[Test] Post notification: ' . $title;
    }

    // Status-based prefixes for instant recognition
    if ($status === 'failed') {
        return '⚠ Post failed: ' . $title;
    } elseif ($status === 'draft') {
        return '📝 New draft: ' . $title;
    } else {
        return '✓ New post: ' . $title;
    }
}

/**
 * Generate professional HTML email template
 * Modern, mobile-first design optimized for SMB owners
 */
function sica_email_html_template($d) {
    // Helper functions for PHP 7.0+ compatibility
    $esc = function($s) { return esc_html($s ? $s : ''); };
    $url = function($s) { return esc_url($s ? $s : ''); };

    // Extract all data with safe defaults
    $post_url = isset($d['post']['url']) ? $d['post']['url'] : '';
    $edit_url = isset($d['post']['edit']) ? $d['post']['edit'] : '';
    // Decode HTML entities (e.g., &#8220; → ") for clean display
    $raw_title = isset($d['post']['title']) ? $d['post']['title'] : 'Untitled';
    $title = html_entity_decode($raw_title, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    $excerpt = isset($d['post']['excerpt']) ? $d['post']['excerpt'] : '';
    $thumbnail = isset($d['post']['thumbnail']) ? $d['post']['thumbnail'] : '';
    $word_count = isset($d['post']['word_count']) ? intval($d['post']['word_count']) : 0;
    $read_time = isset($d['read_time']) ? intval($d['read_time']) : 1;
    $status = isset($d['post']['status']) ? $d['post']['status'] : 'published';

    $site_name = isset($d['site']['name']) ? $d['site']['name'] : 'Site';
    $site_domain = isset($d['site']['domain']) ? $d['site']['domain'] : '';
    $mode_label = isset($d['mode_label']) ? $d['mode_label'] : '';
    $when = isset($d['when']) ? $d['when'] : '';
    $notifications_url = isset($d['notifications_url']) ? $d['notifications_url'] : '';

    // Format timestamp for display (e.g., "Jan 24, 2026 at 8:16 PM")
    $timestamp_display = $when;
    if (!empty($when)) {
        $dt = DateTime::createFromFormat('Y-m-d H:i:s T', $when);
        if ($dt) {
            $timestamp_display = $dt->format('M j, Y \a\t g:i A');
        }
    }

    // Format mode label for display
    $mode_display = ucfirst(str_replace('_', ' ', $mode_label));

    // Status badge styling
    if ($status === 'failed') {
        $badge_bg = '#fef2f2';
        $badge_color = '#dc2626';
        $badge_text = '⚠ Post generation failed';
    } elseif ($status === 'draft') {
        $badge_bg = '#fef3c7';
        $badge_color = '#d97706';
        $badge_text = '📝 New draft created on ' . $esc($site_domain);
    } else {
        $badge_bg = '#ecfdf5';
        $badge_color = '#059669';
        $badge_text = '✓ New article published on ' . $esc($site_domain);
    }

    // Build social share URLs - decode HTML entities first for clean sharing
    $clean_title = html_entity_decode($title, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    $encoded_url = rawurlencode($post_url);
    $encoded_title = rawurlencode($clean_title);
    $share_twitter = 'https://twitter.com/intent/tweet?text=' . $encoded_title . '&url=' . $encoded_url;
    $share_linkedin = 'https://www.linkedin.com/shareArticle?mini=true&url=' . $encoded_url;
    $share_facebook = 'https://www.facebook.com/sharer/sharer.php?u=' . $encoded_url;

    // Hidden preview text for email clients (appears in inbox preview)
    $preview_text = $esc($site_name) . ' — ' . number_format($word_count) . ' words, ' . $read_time . ' min read. View post or share on social.';

    // Build thumbnail HTML if available
    $thumbnail_html = '';
    if (!empty($thumbnail)) {
        $thumbnail_html = '<a href="' . $url($post_url) . '"><img src="' . $url($thumbnail) . '" alt="' . $esc($title) . '" style="width:100%;height:auto;display:block"></a>';
    }

    // Build the HTML email
    $html = '<!doctype html>
<html>
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
</head>
<body style="font-family:-apple-system,BlinkMacSystemFont,\'Segoe UI\',Roboto,Helvetica,Arial,sans-serif;color:#374151;margin:0;padding:0;background:#f5f5f5;line-height:1.5">
  <div style="padding:20px 12px">

    <!-- Hidden preview text for email clients -->
    <div style="display:none;max-height:0;overflow:hidden">
      ' . $preview_text . '
      ' . str_repeat('&#8204; ', 30) . '
    </div>

    <div style="max-width:540px;margin:0 auto;background:#ffffff;border-radius:8px;overflow:hidden;border:1px solid #e5e7eb">

      <!-- Simple header -->
      <div style="background:#f9fafb;border-bottom:1px solid #e5e7eb;padding:16px 20px;text-align:center">
        <span style="font-size:13px;font-weight:600;color:#6b7280;text-transform:uppercase;letter-spacing:0.03em">Sica Content AI</span>
      </div>

      <!-- Main content area -->
      <div style="padding:20px">

        <!-- Status badge with site identifier -->
        <div style="text-align:center;margin-bottom:16px">
          <span style="display:inline-block;background:' . $badge_bg . ';color:' . $badge_color . ';font-size:12px;font-weight:600;padding:6px 12px;border-radius:20px">
            ' . $badge_text . '
          </span>
        </div>

        <!-- Primary CTA - ABOVE THE FOLD -->';

    if (!empty($post_url)) {
        $html .= '
        <div style="text-align:center;margin-bottom:20px">
          <a href="' . $url($post_url) . '" style="display:inline-block;background:#2563eb;color:#ffffff;padding:12px 28px;text-decoration:none;border-radius:6px;font-size:14px;font-weight:600">
            View Post →
          </a>';
        if (!empty($edit_url)) {
            $html .= '
          <a href="' . $url($edit_url) . '" style="display:inline-block;color:#6b7280;padding:12px 16px;text-decoration:none;font-size:14px;font-weight:500">
            Edit
          </a>';
        }
        $html .= '
        </div>';
    }

    // Post preview card (image, title, excerpt)
    $html .= '
        <!-- Post preview card -->
        <div style="border:1px solid #e5e7eb;border-radius:8px;overflow:hidden;margin-bottom:20px">';

    if (!empty($thumbnail_html)) {
        $html .= '
          <!-- Featured image -->
          ' . $thumbnail_html;
    }

    $html .= '
          <div style="padding:16px">
            <!-- Post title -->
            <h1 style="margin:0 0 12px 0;font-size:18px;font-weight:600;color:#111827;line-height:1.4">
              <a href="' . $url($post_url) . '" style="color:#111827;text-decoration:none">' . $esc($title) . '</a>
            </h1>';

    if (!empty($excerpt)) {
        $html .= '
            <!-- Content excerpt -->
            <p style="margin:0;font-size:14px;color:#4b5563;line-height:1.6">
              ' . $esc($excerpt) . '
            </p>';
    }

    $html .= '
          </div>
        </div>';

    // Quick stats (only show if we have word count)
    if ($word_count > 0) {
        $html .= '
        <!-- Quick stats -->
        <div style="font-size:13px;color:#6b7280;margin-bottom:20px;text-align:center">
          <span style="margin-right:16px">📝 ' . number_format($word_count) . ' words</span>
          <span style="margin-right:16px">⏱ ' . $read_time . ' min read</span>
          <span>✓ SEO optimized</span>
        </div>';
    }

    // Share section (only for published posts with URL)
    if (!empty($post_url) && $status !== 'failed') {
        $html .= '
        <!-- Share section -->
        <div style="background:#f9fafb;border-radius:6px;padding:14px;margin-bottom:20px;text-align:center">
          <div style="font-size:12px;font-weight:500;color:#6b7280;margin-bottom:10px">Share this post</div>
          <a href="' . $url($share_twitter) . '" style="display:inline-block;background:#000000;color:#fff;padding:6px 12px;border-radius:4px;text-decoration:none;font-size:12px;font-weight:500;margin-right:6px">X</a>
          <a href="' . $url($share_linkedin) . '" style="display:inline-block;background:#0077b5;color:#fff;padding:6px 12px;border-radius:4px;text-decoration:none;font-size:12px;font-weight:500;margin-right:6px">LinkedIn</a>
          <a href="' . $url($share_facebook) . '" style="display:inline-block;background:#1877f2;color:#fff;padding:6px 12px;border-radius:4px;text-decoration:none;font-size:12px;font-weight:500">Facebook</a>
        </div>';
    }

    // Details line
    $html .= '
        <!-- Details -->
        <div style="border-top:1px solid #e5e7eb;padding-top:16px;font-size:13px;color:#6b7280;text-align:center">
          ' . $esc($timestamp_display) . ' · ' . $esc($mode_display) . '
        </div>

      </div>

    </div>

    <!-- Compliance footer (outside main card) -->
    <div style="max-width:540px;margin:16px auto 0 auto;text-align:center;font-size:11px;color:#9ca3af;line-height:1.6">
      You received this because post notifications are enabled.';

    if (!empty($notifications_url)) {
        $html .= '
      <a href="' . $url($notifications_url) . '" style="color:#6b7280;text-decoration:underline">Manage settings</a> ·';
    }

    $html .= '
      <a href="https://sica.ai" style="color:#6b7280;text-decoration:underline">Help</a>
    </div>

  </div>
</body>
</html>';

    return $html;
}

/**
 * Generate plain text email template
 * Matching the new HTML template structure
 */
function sica_email_text_template($d) {
    $site_name = trim(isset($d['site']['name']) ? $d['site']['name'] : '');
    $site_domain = isset($d['site']['domain']) ? $d['site']['domain'] : '';
    $title = trim(isset($d['post']['title']) ? $d['post']['title'] : 'Untitled');
    $url = isset($d['post']['url']) ? $d['post']['url'] : '';
    $edit = isset($d['post']['edit']) ? $d['post']['edit'] : '';
    $excerpt = isset($d['post']['excerpt']) ? $d['post']['excerpt'] : '';
    $word_count = isset($d['post']['word_count']) ? intval($d['post']['word_count']) : 0;
    $read_time = isset($d['read_time']) ? intval($d['read_time']) : 1;
    $mode = isset($d['mode_label']) ? ucfirst(str_replace('_', ' ', $d['mode_label'])) : '';
    $when = isset($d['when']) ? $d['when'] : '';
    $status = isset($d['post']['status']) ? $d['post']['status'] : 'published';
    $notifications_url = isset($d['notifications_url']) ? $d['notifications_url'] : '';

    // Format timestamp
    $timestamp_display = $when;
    if (!empty($when)) {
        $dt = DateTime::createFromFormat('Y-m-d H:i:s T', $when);
        if ($dt) {
            $timestamp_display = $dt->format('M j, Y \a\t g:i A');
        }
    }

    // Status-based header
    if ($status === 'failed') {
        $header = "⚠ Post generation failed on $site_domain";
    } elseif ($status === 'draft') {
        $header = "📝 New draft created on $site_domain";
    } else {
        $header = "✓ New article published on $site_domain";
    }

    // Build share URLs - decode HTML entities first for clean sharing
    $clean_title = html_entity_decode($title, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    $encoded_url = rawurlencode($url);
    $encoded_title = rawurlencode($clean_title);
    $share_twitter = 'https://twitter.com/intent/tweet?text=' . $encoded_title . '&url=' . $encoded_url;
    $share_linkedin = 'https://www.linkedin.com/shareArticle?mini=true&url=' . $encoded_url;

    $lines = [
        "SICA CONTENT AI",
        "",
        $header,
        "",
        "─────────────────────────────────────",
        "",
        $title,
        "",
    ];

    if (!empty($excerpt)) {
        $lines[] = $excerpt;
        $lines[] = "";
    }

    if ($word_count > 0) {
        $lines[] = "📝 " . number_format($word_count) . " words · ⏱ " . $read_time . " min read · ✓ SEO optimized";
        $lines[] = "";
    }

    $lines[] = "─────────────────────────────────────";
    $lines[] = "";

    if (!empty($url)) {
        $lines[] = "VIEW POST: $url";
        $lines[] = "";
    }

    if (!empty($edit)) {
        $lines[] = "Edit in WordPress: $edit";
        $lines[] = "";
    }

    if (!empty($url) && $status !== 'failed') {
        $lines[] = "SHARE THIS POST:";
        $lines[] = "X/Twitter: $share_twitter";
        $lines[] = "LinkedIn: $share_linkedin";
        $lines[] = "";
    }

    $lines[] = "─────────────────────────────────────";
    $lines[] = "";
    $lines[] = "$timestamp_display · $mode";
    $lines[] = "Site: $site_name";
    $lines[] = "";
    $lines[] = "─────────────────────────────────────";
    $lines[] = "";
    $lines[] = "You received this because post notifications are enabled.";

    if (!empty($notifications_url)) {
        $lines[] = "Manage settings: $notifications_url";
    }

    $lines[] = "Help: https://sica.ai";

    return implode("\n", $lines);
}

/**
 * Legacy function - now uses improved system
 */
function sica_build_post_created_email($details, $site_name, $site_url) {
    // Convert to new system format and return HTML
    $args = [
        'post_id' => isset($details['post_id']) ? $details['post_id'] : 0,
        'title' => isset($details['post_title']) ? $details['post_title'] : 'Untitled',
        'permalink' => isset($details['post_url']) ? $details['post_url'] : '',
        'mode' => 'auto',
        'status' => 'published',
        'site_url' => $site_url,
        'recipient' => ''
    ];
    
    $normalized = sica_email_normalize_args($args);
    $payload = sica_email_build_payload($normalized);
    
    return sica_email_html_template($payload);
}

/**
 * Build email content for titles empty notification
 */
function sica_build_titles_empty_email($site_name, $site_url) {
    $args = [
        'post_id' => 0,
        'title' => 'Title Queue Empty',
        'mode' => 'auto',
        'status' => 'failed',
        'site_url' => $site_url,
        'recipient' => ''
    ];
    
    $normalized = sica_email_normalize_args($args);
    $payload = sica_email_build_payload($normalized);
    $payload['headline'] = 'Automated posting paused: Title queue empty';
    
    // Custom template for empty queue  
    $esc = function($s) { return esc_html($s ? $s : ''); };
    $url = function($s) { return esc_url($s ? $s : ''); };
    
    $site = $esc($payload['site']['name']);
    $when = $esc($payload['when']);
    $dash_link = $payload['dashboard_url'] ? '<a href="'.$url($payload['dashboard_url']).'" style="color:#175cd3;text-decoration:none;">Plugin Dashboard</a>' : '';
    $sched_link = $payload['settings_url'] ? '<a href="'.$url($payload['settings_url']).'" style="color:#175cd3;text-decoration:none;">Scheduler Settings</a>' : '';
    $planner_link = function_exists('admin_url') ? admin_url('admin.php?page=ai-blogger&tab=planner') : ($site_url . '/wp-admin/admin.php?page=ai-blogger&tab=planner');
    $planner_link = '<a href="'.$url($planner_link).'" style="color:#175cd3;text-decoration:none;">Content Planner</a>';
    
    $headline_escaped = $esc($payload['headline']);
    $support_email_escaped = $esc($payload['support_email']);
    
    return <<<HTML
<!doctype html>
<html>
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <style>
    body{font-family:-apple-system,BlinkMacSystemFont,'Segoe UI',Helvetica,Arial,sans-serif;color:#24292f;margin:0;padding:0;background:#f6f8fa;line-height:1.5}
    .email-container{max-width:600px;margin:0 auto;background:#ffffff;border-radius:12px;overflow:hidden;box-shadow:0 4px 12px rgba(0,0,0,0.1)}
    .header{background:linear-gradient(135deg,#f59e0b 0%,#d97706 100%);color:#ffffff;text-align:center;padding:32px 24px}
    .header h1{margin:0;font-size:24px;font-weight:600;letter-spacing:-0.02em}
    .content{padding:32px 24px}
    .alert-info{background:#fef3c7;border:1px solid #f59e0b;border-radius:8px;padding:20px;margin:20px 0}
    .info-row{display:flex;align-items:center;margin:8px 0;font-size:14px}
    .info-label{font-weight:600;color:#92400e;margin-right:8px;min-width:80px}
    .info-value{color:#92400e}
    .actions{margin:24px 0;padding:20px;background:#e0f2fe;border-radius:8px;border:1px solid #0284c7}
    .action-btn{display:inline-block;background:#0969da;color:#ffffff;padding:8px 16px;text-decoration:none;border-radius:6px;font-size:14px;font-weight:500;margin:4px 8px 4px 0}
    .action-btn:hover{background:#0550ae;color:#ffffff}
    .learn-more{margin:24px 0;padding:16px;background:#dbeafe;border-radius:8px;font-size:14px;color:#1e40af}
    .learn-more a{color:#1e40af;font-weight:500}
    .footer{margin-top:24px;padding:20px;border-top:1px solid #e1e8ed;font-size:13px;color:#656d76;line-height:1.6}
    .footer a{color:#0969da}
    @media (max-width: 600px) {
      .email-container{margin:0;border-radius:0}
      .content{padding:20px 16px}
      .header{padding:24px 16px}
    }
  </style>
</head>
<body>
  <div style="padding:20px 0">
    <div class="email-container">
      <div class="header">
        <h1>{$headline_escaped}</h1>
      </div>
      
      <div class="content">
        <div class="alert-info">
          <div class="info-row">
            <span class="info-label">Site:</span>
            <span class="info-value">{$site}</span>
          </div>
          <div class="info-row">
            <span class="info-label">Status:</span>
            <span class="info-value">Automated posting paused</span>
          </div>
          <div class="info-row">
            <span class="info-label">Reason:</span>
            <span class="info-value">No titles available for content generation</span>
          </div>
          <div class="info-row">
            <span class="info-label">Time:</span>
            <span class="info-value">{$when}</span>
          </div>
        </div>

        <div class="actions">
          <strong>Next Steps:</strong><br><br>
          1. Add new titles to your {$planner_link}<br>
          2. The scheduler will automatically resume posting<br><br>
          <strong>Quick Actions:</strong><br>
          {$dash_link} • {$sched_link}
        </div>

        <div class="learn-more">
          <strong>💡 Tip:</strong> Keep 5-10 titles in your queue to ensure uninterrupted posting. 
          <a href="https://sica.ai">Learn more about content planning →</a>
        </div>

        <div class="footer">
          You received this notification because the Sica Content AI plugin's title queue is empty and automated posting has been paused.
          <br><br>
          <strong>Support:</strong> <a href="mailto:{$support_email_escaped}">{$support_email_escaped}</a>
        </div>
      </div>
    </div>
  </div>
</body>
</html>
HTML;
}

/**
 * Get optimized email headers for maximum deliverability
 */
function sica_get_optimized_email_headers($site_name, $site_url) {
	$headers = array();
	
	// Content type (send HTML to preserve formatting)
	$headers[] = 'Content-Type: text/html; charset=UTF-8';
	
	// From header with proper domain
	$domain = wp_parse_url($site_url, PHP_URL_HOST);
	if (!empty($domain)) {
		// Remove www prefix for cleaner sender domain
		$clean_domain = preg_replace('/^www\./i', '', $domain);
		$from_email = 'notifications@' . $clean_domain;
		$from_name = wp_specialchars_decode($site_name, ENT_QUOTES);
		
		$headers[] = 'From: ' . $from_name . ' <' . sanitize_email($from_email) . '>';
		$headers[] = 'Reply-To: ' . $from_name . ' <' . sanitize_email($from_email) . '>';
	}
	
	// Additional headers for better deliverability
	$headers[] = 'X-Mailer: Sica Content AI Plugin';
	$headers[] = 'X-Priority: 3'; // Normal priority
	$headers[] = 'Message-ID: <' . uniqid() . '@' . ($clean_domain ? $clean_domain : 'sica.ai') . '>';
	$headers[] = 'Auto-Submitted: auto-generated';
	$headers[] = 'X-Auto-Response-Suppress: All';
	
	// Optional: Add List-Unsubscribe header (good practice)
	$unsubscribe_url = add_query_arg(array(
		'sica_action' => 'unsubscribe',
		'nonce' => wp_create_nonce('sica_unsubscribe')
	), $site_url);
	$headers[] = 'List-Unsubscribe: <' . $unsubscribe_url . '>';
	
	return $headers;
}

/**
 * Personalize email message for recipient
 */
function sica_personalize_email_message($message, $email) {
    // Add recipient-specific greeting if email looks like a name
    $email_parts = explode('@', $email);
    $username = $email_parts[0];
    
    // Simple personalization for common name patterns
    if (preg_match('/^[a-z]+\.[a-z]+$/i', $username)) {
        $name_parts = explode('.', $username);
        $first_name = ucfirst($name_parts[0]);
        $personalized = "Hi {$first_name},\n\n" . $message;
        return $personalized;
    }
    
    return $message;
}

/**
 * Diagnose common email configuration issues
 */
function sica_diagnose_email_issues() {
    // error_log('[SICA Email Diagnosis] ═══════════════════════════════════');
    // error_log('[SICA Email Diagnosis] EMAIL DELIVERY TROUBLESHOOTING REPORT');
    // error_log('[SICA Email Diagnosis] ═══════════════════════════════════');
    
    // Check hosting environment
    $server_software = isset($_SERVER['SERVER_SOFTWARE']) ? $_SERVER['SERVER_SOFTWARE'] : 'Unknown';
    // error_log('[SICA Email Diagnosis] Server: ' . $server_software);
    
    // Check if we're on common development environments
    $is_local = (
        strpos(home_url(), 'localhost') !== false ||
        strpos(home_url(), '127.0.0.1') !== false ||
        strpos(home_url(), '.local') !== false ||
        strpos(home_url(), '.test') !== false ||
        strpos(home_url(), '.dev') !== false
    );
    
    if ($is_local) {
        // error_log('[SICA Email Diagnosis] ⚠️ LOCAL DEVELOPMENT DETECTED');
        // error_log('[SICA Email Diagnosis] Local environments often cannot send emails');
        // error_log('[SICA Email Diagnosis] Solution: Use an SMTP plugin with external email service');
    }
    
    // Check for common SMTP plugins
    $smtp_plugins = array(
        'wp-mail-smtp/wp_mail_smtp.php' => 'WP Mail SMTP',
        'easy-wp-smtp/easy-wp-smtp.php' => 'Easy WP SMTP',
        'post-smtp/postman-smtp.php' => 'Post SMTP',
        'smtp-mailer/main.php' => 'SMTP Mailer'
    );
    
    $smtp_detected = false;
    foreach ($smtp_plugins as $plugin_path => $plugin_name) {
        if (is_plugin_active($plugin_path)) {
            // error_log("[SICA Email Diagnosis] ✅ SMTP plugin active: {$plugin_name}");
            $smtp_detected = true;
            break;
        }
    }
    
    if (!$smtp_detected) {
        // error_log('[SICA Email Diagnosis] ❌ NO SMTP PLUGIN DETECTED');
        // error_log('[SICA Email Diagnosis] This is likely why emails are not being delivered');
        // error_log('[SICA Email Diagnosis] WordPress default mail() function is unreliable');
    }
    
    // Check server mail function
    if (!function_exists('mail')) {
        // error_log('[SICA Email Diagnosis] ❌ PHP mail() function not available');
    } else {
        // error_log('[SICA Email Diagnosis] ✅ PHP mail() function available');
    }
    
    // Test basic WordPress email to admin
    // error_log('[SICA Email Diagnosis] Testing WordPress email to admin...');
    $test_result = wp_mail(get_option('admin_email'), 'WordPress Email Test', 'Test from email diagnosis');
    
    if (!$test_result) {
        // error_log('[SICA Email Diagnosis] ❌ WordPress email test FAILED');
    } else {
        // error_log('[SICA Email Diagnosis] ✅ WordPress email test returned TRUE');
        // error_log('[SICA Email Diagnosis] NOTE: TRUE does not guarantee actual delivery');
    }
    
    // Check domain configuration
    $domain = wp_parse_url(home_url(), PHP_URL_HOST);
    $from_email = 'notifications@' . preg_replace('/^www\./i', '', $domain);
    // error_log('[SICA Email Diagnosis] From email being used: ' . $from_email);
    
    // Recommendations
    // error_log('[SICA Email Diagnosis] ═══════════════════════════════════');
    // error_log('[SICA Email Diagnosis] RECOMMENDATIONS:');
    
    if (!$smtp_detected) {
        // error_log('[SICA Email Diagnosis] 1. Install WP Mail SMTP plugin');
        // error_log('[SICA Email Diagnosis] 2. Configure with Gmail, SendGrid, or Mailgun');
        // error_log('[SICA Email Diagnosis] 3. Use your hosting provider\'s SMTP settings');
    }
    
    if ($is_local) {
        // error_log('[SICA Email Diagnosis] 4. For local dev: Use Mailtrap or similar service');
    }
    
    // error_log('[SICA Email Diagnosis] 5. Check spam/junk folders');
    // error_log('[SICA Email Diagnosis] 6. Verify domain DNS/SPF records');
    // error_log('[SICA Email Diagnosis] ═══════════════════════════════════');
}

/**
 * Save and get API key functions
 */
function save_api_key($api_key) {
    if (empty($api_key)) {
        delete_option('my_auto_blog_api_key');
    } else {
        update_option('my_auto_blog_api_key', $api_key);
    }
}

function get_api_key() {
    // Check if API key is defined in wp-config.php (recommended for security)
    if (defined('OPENAI_API_KEY') && !empty(OPENAI_API_KEY)) {
        return OPENAI_API_KEY;
    }
    
    // Fall back to database option
    return get_option('my_auto_blog_api_key', '');
}

/**
 * Plugin activation tasks (called from main activation hook)
 */
if (!function_exists('my_auto_blog_run_activation_tasks')) {
function my_auto_blog_run_activation_tasks() {
    // Initialize options with defaults if they don't exist
    if (false === get_option('my_auto_blog_notify_emails')) {
        add_option('my_auto_blog_notify_emails', '');
    }
    
    // Other activation tasks...
}
}