<?php
/**
 * Content Planner Management
 * 
 * Handles business details, keywords, titles, and content planning functionality
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Render the content planner tab
 */
function my_auto_blog_content_planner_tab() {
    // Display the success message if the transient is set
    if (get_transient('my_auto_blog_content_planner_updated')) {
        ?>
        <div class="notice notice-success is-dismissible">
            <p>Content planner settings updated successfully.</p>
        </div>
        <?php
        delete_transient('my_auto_blog_content_planner_updated');
    }

    // Retrieve existing data
    $titles = get_option('my_auto_blog_title');
    if (!is_array($titles)) {
        $titles = explode("\n", $titles);
    }

    $business_name = get_bloginfo('name');
    $business_category = get_option('my_auto_blog_business_category', '');
    $business_service_area = get_option('my_auto_blog_service_area', '');
    $products_services = get_option('my_auto_blog_products_services', '');
    $all_keywords = get_option('my_auto_blog_keywords', '');

    ?>
    <div>
        <h3>Blog Content Planner</h3>
        <p>Fill in your business details and keywords to generate relevant blog title ideas.</p>
        
        <?php 
        // Add category fix button if uncategorized posts exist
        add_category_fix_button(); 
        add_category_fix_js();
        ?>
        <form id="content-planner-form" method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
            <input type="hidden" name="action" value="save_content_planner">
            <?php wp_nonce_field('save_content_planner_action', 'save_content_planner_nonce'); ?>
            
            <!-- Business Details Section -->
            <div style="margin-bottom: 20px;">
                <h3>Business Details</h3>
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="business_name">Business Name:</label></th>
                        <td>
                            <input type="text" id="business_name" name="business_name" value="<?php echo esc_attr($business_name); ?>" style="width: 300px;" readonly>
                            <p class="description">This is your site name. To change it, go to <a href="<?php echo esc_url(admin_url('options-general.php')); ?>">Settings > General</a>.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="business_category">Business Category:</label></th>
                        <td>
                            <input type="text" id="business_category" name="business_category" value="<?php echo esc_attr($business_category); ?>" style="width: 300px;">
                            <p class="description">E.g., Plumbing, Real Estate, Digital Marketing</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="business_service_area">Service Area:</label></th>
                        <td>
                            <input type="text" id="business_service_area" name="business_service_area" value="<?php echo esc_attr($business_service_area); ?>" style="width: 300px;">
                            <p class="description">E.g., New York City, Southern California, Nationwide</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="products_services">Target Products/Services:</label></th>
                        <td>
                            <textarea id="products_services" name="products_services" style="width: 300px; height: 100px;"><?php echo esc_textarea($products_services); ?></textarea>
                            <p class="description">List your main products or services, one per line</p>
                        </td>
                    </tr>
                </table>
            </div>

            <!-- SEO Keywords and Blog Titles Section -->
            <div>
                <h3>SEO Keywords and Blog Titles</h3>
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="my_auto_blog_keywords">Target SEO keywords:</label></th>
                        <td>
                            <textarea id="my_auto_blog_keywords" name="my_auto_blog_keywords" style="width: 100%; max-width: 500px; height: 140px;"><?php echo esc_textarea($all_keywords); ?></textarea>
                            <p class="description">Enter one keyword per line, 10-20 keywords is best.</p>
                            <div class="keyword-buttons" style="margin-top: 10px;">
                                <button type="button" id="generate_keywords_content" class="button">Generate keyword ideas</button>
                            </div>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="my_auto_blog_title">Upcoming Blog Titles:</label></th>
                        <td>
                            <p class="description">Enter one title per line. Top will be used next.</p>
                            <textarea id="my_auto_blog_title" name="my_auto_blog_title" style="width: 100%; max-width: 800px; height: 200px;"><?php echo esc_textarea(implode("\n", $titles)); ?></textarea>
                            <div>
                                <button type="button" id="generate_titles_button" class="button">
                                    <span id="button_text">Generate more titles using keywords</span>
                                    <span id="loading_indicator" style="display: none;">Loading...</span>
                                </button>
                            </div>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Save Content Settings'); ?>
            </div>
        </form>

        <!-- Generate Blog Now section removed per request -->
    </div>
    <?php
}

/**
 * Save content planner settings
 */
function save_content_planner_settings() {
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized user');
    }

    // Verify nonce
    check_admin_referer('save_content_planner_action', 'save_content_planner_nonce');

    // Save the settings
    $titles = isset($_POST['my_auto_blog_title']) ? sanitize_textarea_field($_POST['my_auto_blog_title']) : '';
    $keywords = isset($_POST['my_auto_blog_keywords']) ? sanitize_textarea_field($_POST['my_auto_blog_keywords']) : '';
    $business_category = isset($_POST['business_category']) ? sanitize_text_field($_POST['business_category']) : '';
    $business_service_area = isset($_POST['business_service_area']) ? sanitize_text_field($_POST['business_service_area']) : '';
    $products_services = isset($_POST['products_services']) ? sanitize_textarea_field($_POST['products_services']) : '';

    // Clean the titles to remove escaped slashes before saving
    $cleaned_titles = clean_blog_titles($titles);
    $titles_string = implode("\n", $cleaned_titles);

    update_option('my_auto_blog_title', $titles_string);
    update_option('my_auto_blog_keywords', $keywords);
    update_option('my_auto_blog_business_category', $business_category);
    update_option('my_auto_blog_service_area', $business_service_area);
    update_option('my_auto_blog_products_services', $products_services);

    // Redirect back with success parameter
    wp_redirect(add_query_arg([
        'page' => 'ai-blogger',
        'tab' => 'content_planner',
        'settings-updated' => 'true'
    ], admin_url('admin.php')));
    exit;
}
add_action('admin_post_save_content_planner', 'save_content_planner_settings');

/**
 * Quick AJAX save for planner fields (business details + keywords + titles) before generation
 */
function save_content_planner_quick_ajax() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');

    $keywords = isset($_POST['my_auto_blog_keywords']) ? sanitize_textarea_field(wp_unslash($_POST['my_auto_blog_keywords'])) : '';
    $titles = isset($_POST['my_auto_blog_title']) ? sanitize_textarea_field(wp_unslash($_POST['my_auto_blog_title'])) : '';
    $business_category = isset($_POST['business_category']) ? sanitize_text_field(wp_unslash($_POST['business_category'])) : '';
    $business_service_area = isset($_POST['business_service_area']) ? sanitize_text_field(wp_unslash($_POST['business_service_area'])) : '';
    $products_services = isset($_POST['products_services']) ? sanitize_textarea_field(wp_unslash($_POST['products_services'])) : '';

    // Persist current edits so the generator has latest context
    update_option('my_auto_blog_keywords', $keywords);
    update_option('my_auto_blog_title', $titles);
    update_option('my_auto_blog_business_category', $business_category);
    update_option('my_auto_blog_service_area', $business_service_area);
    update_option('my_auto_blog_products_services', $products_services);

    wp_send_json_success(['saved' => true]);
}
add_action('wp_ajax_save_content_planner_quick', 'save_content_planner_quick_ajax');

/**
 * Get business details for use in AI generation
 */
function get_business_details() {
    return array(
        'business_name' => get_bloginfo('name'),
        'business_category' => get_option('my_auto_blog_business_category', ''),
        'business_service_area' => get_option('my_auto_blog_service_area', ''),
        'products_services' => get_option('my_auto_blog_products_services', '')
    );
}