<?php
/**
 * Debug Log Viewer
 *
 * Simple admin page to view error logs when file access isn't available
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Add debug viewer menu item
 */
function sica_add_debug_viewer_menu() {
    // Only show if user has manage_options capability
    if (!current_user_can('manage_options')) {
        return;
    }

    // Only show if WP_DEBUG is enabled
    if (!defined('WP_DEBUG') || !WP_DEBUG) {
        return;
    }

    add_submenu_page(
        'my_auto_blog_settings',
        'Debug Logs',
        'Debug Logs',
        'manage_options',
        'sica-debug-logs',
        'sica_render_debug_viewer_page'
    );
}
add_action('admin_menu', 'sica_add_debug_viewer_menu', 100);

/**
 * Render debug viewer page
 */
function sica_render_debug_viewer_page() {
    if (!current_user_can('manage_options')) {
        wp_die('Insufficient permissions');
    }

    // Get file logs if available
    $file_log_path = sica_get_log_file_path();
    $file_logs = array();
    if ($file_log_path && file_exists($file_log_path)) {
        $file_logs = sica_get_recent_logs(100);
    }

    ?>
    <div class="wrap">
        <h1>Sica Content AI - Debug Logs</h1>

        <div class="notice notice-info">
            <p><strong>File-Based Technical Debugging</strong></p>
            <p style="margin: 8px 0 0 0; font-size: 13px;">
                This page shows low-level technical logs from the file system.
                For post generation activity (successes/failures), see the
                <a href="<?php echo admin_url('admin.php?page=my_auto_blog_settings&tab=scheduler'); ?>">Activity Log on the Scheduler tab</a>.
            </p>
        </div>

        <div class="notice notice-info">
            <p><strong>Debugging Information:</strong></p>
            <ul>
                <li>WP_DEBUG: <?php echo defined('WP_DEBUG') && WP_DEBUG ? '✅ Enabled' : '❌ Disabled'; ?></li>
                <li>WP_DEBUG_LOG: <?php echo defined('WP_DEBUG_LOG') && WP_DEBUG_LOG ? '✅ Enabled' : '❌ Disabled'; ?></li>
                <li>Log File Path: <?php echo $file_log_path ? esc_html($file_log_path) : 'Not available'; ?></li>
                <li>File Writable: <?php echo $file_log_path && is_writable(dirname($file_log_path)) ? '✅ Yes' : '❌ No'; ?></li>
            </ul>
        </div>

        <!-- File Logs Section -->
        <?php if (!empty($file_logs)): ?>
            <h2 style="margin-top: 40px;">File Logs (Last 100 Entries)</h2>
            <p>These logs are stored at: <code><?php echo esc_html($file_log_path); ?></code></p>

            <div style="background: #000; color: #0f0; padding: 15px; font-family: monospace; font-size: 12px; overflow-x: auto; max-height: 500px; overflow-y: scroll;">
                <?php foreach ($file_logs as $log): ?>
                    <div><?php echo esc_html($log); ?></div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <h2 style="margin-top: 40px;">Log Locations</h2>
        <ul>
            <li><strong>WordPress Debug Log:</strong> <code>wp-content/debug.log</code>
                <?php
                $debug_log = WP_CONTENT_DIR . '/debug.log';
                echo file_exists($debug_log) ? '✅ Exists (' . size_format(filesize($debug_log)) . ')' : '❌ Not found';
                ?>
            </li>
            <li><strong>Sica Custom Logs:</strong> <code>wp-content/uploads/sica-logs/sica-<?php echo date('Y-m-d'); ?>.log</code>
                <?php
                echo $file_log_path && file_exists($file_log_path) ? '✅ Exists (' . size_format(filesize($file_log_path)) . ')' : '❌ Not found';
                ?>
            </li>
            <li><strong>Activity Log:</strong> See <a href="<?php echo admin_url('admin.php?page=my_auto_blog_settings&tab=scheduler'); ?>">Scheduler tab</a> for post generation activity (manual & scheduled posts)
            </li>
        </ul>

        <h2 style="margin-top: 40px;">Troubleshooting</h2>
        <div class="notice notice-info">
            <p><strong>If logs aren't appearing:</strong></p>
            <ol>
                <li>Enable WP_DEBUG in wp-config.php (see configuration above)</li>
                <li>Check that <code>wp-content/uploads/</code> directory is writable (permissions 755 or 775)</li>
                <li>Check your server's PHP error log for permission errors</li>
                <li>For post generation activity, use the <a href="<?php echo admin_url('admin.php?page=my_auto_blog_settings&tab=scheduler'); ?>">Activity Log</a> (always available, no file access needed)</li>
            </ol>
        </div>
    </div>
    <?php
}
