<?php
/**
 * Smart Logging Utility for Sica Content AI
 *
 * Provides structured, filterable logging with automatic context capture.
 * Respects WP_DEBUG settings while ensuring critical errors are always logged.
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Smart logging function with automatic context capture
 *
 * @param string $message The log message
 * @param string $level Log level: 'debug', 'info', 'warning', 'error', 'critical'
 * @param array $context Additional context data
 * @param bool $force Force logging even if debugging is disabled
 */
function sica_log($message, $level = 'info', $context = array(), $force = false) {
    // Always log errors and critical issues
    $always_log_levels = array('error', 'critical');
    $should_log = in_array($level, $always_log_levels) || $force;

    // Log warnings if WP_DEBUG is enabled
    if (!$should_log && $level === 'warning' && defined('WP_DEBUG') && WP_DEBUG) {
        $should_log = true;
    }

    // Log info and debug only if WP_DEBUG is enabled
    if (!$should_log && in_array($level, array('info', 'debug')) && defined('WP_DEBUG') && WP_DEBUG) {
        $should_log = true;
    }

    if (!$should_log) {
        return;
    }

    // Capture context automatically
    $trace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 2);
    $caller = isset($trace[1]) ? $trace[1] : $trace[0];

    $file = isset($caller['file']) ? basename($caller['file']) : 'unknown';
    $line = isset($caller['line']) ? $caller['line'] : '0';
    $function = isset($caller['function']) ? $caller['function'] : 'global';

    // Build structured log message
    $timestamp = current_time('Y-m-d H:i:s');
    $level_upper = strtoupper($level);

    // Format context for logging
    $context_str = '';
    if (!empty($context)) {
        if (is_array($context) || is_object($context)) {
            $context_str = ' | Context: ' . json_encode($context, JSON_UNESCAPED_SLASHES);
        } else {
            $context_str = ' | Context: ' . (string)$context;
        }
    }

    // Structured log format for easy grepping
    $log_message = sprintf(
        '[SICA %s] [%s] %s:%s %s() - %s%s',
        $level_upper,
        $timestamp,
        $file,
        $line,
        $function,
        $message,
        $context_str
    );

    // Write to error log
    error_log($log_message);

    // Also log to custom file if configured (for easier debugging)
    sica_log_to_file($log_message, $level);

    // Note: Activity logging (success/failure of posts) is handled by sica_add_scheduler_log_entry()
    // This file-based logging is for technical debugging only
}

/**
 * Write log to custom file for easier debugging
 *
 * @param string $message Formatted log message
 * @param string $level Log level
 */
function sica_log_to_file($message, $level) {
    // Only write to custom file if WP_DEBUG is enabled or level is error/critical
    if (!defined('WP_DEBUG') || !WP_DEBUG) {
        if (!in_array($level, array('error', 'critical'))) {
            return;
        }
    }

    // Get WordPress uploads directory for log file
    $upload_dir = wp_upload_dir();
    $log_dir = $upload_dir['basedir'] . '/sica-logs';

    // Create log directory if it doesn't exist
    if (!file_exists($log_dir)) {
        wp_mkdir_p($log_dir);

        // Add .htaccess to prevent direct access
        $htaccess_file = $log_dir . '/.htaccess';
        if (!file_exists($htaccess_file)) {
            file_put_contents($htaccess_file, 'Deny from all');
        }

        // Add index.php to prevent directory listing
        $index_file = $log_dir . '/index.php';
        if (!file_exists($index_file)) {
            file_put_contents($index_file, '<?php // Silence is golden');
        }
    }

    // Daily log file rotation
    $log_file = $log_dir . '/sica-' . date('Y-m-d') . '.log';

    // Write log with newline
    file_put_contents($log_file, $message . PHP_EOL, FILE_APPEND | LOCK_EX);

    // Clean up old log files (keep last 7 days)
    sica_cleanup_old_logs($log_dir);
}

/**
 * Clean up log files older than 7 days
 *
 * @param string $log_dir Log directory path
 */
function sica_cleanup_old_logs($log_dir) {
    // Only run cleanup occasionally (once per day)
    $last_cleanup = get_transient('sica_last_log_cleanup');
    if ($last_cleanup) {
        return;
    }

    // Set transient for 24 hours
    set_transient('sica_last_log_cleanup', time(), DAY_IN_SECONDS);

    // Find and delete old log files
    $files = glob($log_dir . '/sica-*.log');
    if ($files) {
        $cutoff_time = time() - (7 * DAY_IN_SECONDS);
        foreach ($files as $file) {
            if (filemtime($file) < $cutoff_time) {
                @unlink($file);
            }
        }
    }
}

/**
 * Generate a unique request ID for correlating plugin <-> server logs
 *
 * @return string Unique request ID
 */
function sica_get_request_id() {
    static $request_id = null;

    if ($request_id === null) {
        // Check if we already have a request ID in this request
        $stored_id = get_transient('sica_current_request_id_' . getmypid());
        if ($stored_id) {
            $request_id = $stored_id;
        } else {
            // Generate new request ID
            $request_id = uniqid('req_', true);
            // Store for 5 minutes (longer than any single request should take)
            set_transient('sica_current_request_id_' . getmypid(), $request_id, 300);
        }
    }

    return $request_id;
}

/**
 * Log API request/response for debugging
 *
 * @param string $endpoint API endpoint
 * @param array $request_data Request data
 * @param mixed $response Response data
 * @param float $duration Request duration in seconds
 */
function sica_log_api_call($endpoint, $request_data = array(), $response = null, $duration = 0) {
    $request_id = sica_get_request_id();

    // Sanitize request data (remove sensitive info)
    $safe_request = $request_data;
    if (isset($safe_request['api_token'])) {
        $safe_request['api_token'] = '***' . substr($safe_request['api_token'], -4);
    }
    if (isset($safe_request['license_key'])) {
        $safe_request['license_key'] = '***' . substr($safe_request['license_key'], -4);
    }

    // Prepare context
    $context = array(
        'request_id' => $request_id,
        'endpoint' => $endpoint,
        'duration' => round($duration, 2) . 's',
        'request_size' => strlen(json_encode($request_data)) . ' bytes'
    );

    // Log request
    sica_log(
        sprintf('API Call to %s', $endpoint),
        'info',
        $context
    );

    // Log response if it's an error
    if (is_wp_error($response)) {
        sica_log(
            sprintf('API Error: %s', $response->get_error_message()),
            'error',
            array_merge($context, array(
                'error_code' => $response->get_error_code(),
                'error_data' => $response->get_error_data()
            ))
        );
    } elseif ($response === false || $response === null) {
        sica_log(
            'API call returned null or false',
            'warning',
            $context
        );
    }

    // Log slow API calls
    if ($duration > 30) {
        sica_log(
            sprintf('Slow API call detected: %s took %.2fs', $endpoint, $duration),
            'warning',
            $context
        );
    }
}

/**
 * Log performance metrics
 *
 * @param string $operation Operation name
 * @param float $start_time Start microtime
 * @param array $context Additional context
 */
function sica_log_performance($operation, $start_time, $context = array()) {
    $duration = microtime(true) - $start_time;
    $memory = memory_get_peak_usage(true);

    $perf_context = array_merge($context, array(
        'duration' => round($duration, 2) . 's',
        'memory_peak' => round($memory / 1024 / 1024, 2) . 'MB'
    ));

    $level = 'info';
    if ($duration > 60) {
        $level = 'warning';
    } elseif ($duration > 120) {
        $level = 'error';
    }

    sica_log(
        sprintf('%s completed', $operation),
        $level,
        $perf_context
    );
}

/**
 * Log exception with full stack trace
 *
 * @param Exception|Throwable $exception The exception to log
 * @param string $context_message Additional context message
 */
function sica_log_exception($exception, $context_message = '') {
    $message = $context_message ? $context_message . ': ' : '';
    $message .= $exception->getMessage();

    $context = array(
        'exception_class' => get_class($exception),
        'file' => $exception->getFile(),
        'line' => $exception->getLine(),
        'trace' => $exception->getTraceAsString()
    );

    sica_log($message, 'error', $context, true);
}

/**
 * Helper function for debugging - pretty print variable
 * Only works when WP_DEBUG is enabled
 *
 * @param mixed $var Variable to debug
 * @param string $label Optional label
 */
function sica_debug($var, $label = '') {
    if (!defined('WP_DEBUG') || !WP_DEBUG) {
        return;
    }

    $label_str = $label ? "$label: " : '';

    if (is_array($var) || is_object($var)) {
        sica_log($label_str . print_r($var, true), 'debug');
    } else {
        sica_log($label_str . var_export($var, true), 'debug');
    }
}

/**
 * Get the log file path for today
 *
 * @return string|false Log file path or false if logging disabled
 */
function sica_get_log_file_path() {
    $upload_dir = wp_upload_dir();
    $log_dir = $upload_dir['basedir'] . '/sica-logs';

    if (!file_exists($log_dir)) {
        return false;
    }

    return $log_dir . '/sica-' . date('Y-m-d') . '.log';
}

/**
 * Get recent log entries (for admin display)
 *
 * @param int $limit Number of entries to retrieve
 * @param string $level_filter Filter by log level (optional)
 * @return array Array of log entries
 */
function sica_get_recent_logs($limit = 100, $level_filter = '') {
    $log_file = sica_get_log_file_path();

    if (!$log_file || !file_exists($log_file)) {
        return array();
    }

    // Read file in reverse (most recent first)
    $lines = file($log_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    if (!$lines) {
        return array();
    }

    $lines = array_reverse($lines);

    $entries = array();
    foreach ($lines as $line) {
        // Filter by level if specified
        if ($level_filter && strpos($line, '[' . strtoupper($level_filter) . ']') === false) {
            continue;
        }

        $entries[] = $line;

        if (count($entries) >= $limit) {
            break;
        }
    }

    return $entries;
}

// Database logging functions removed - unified activity log is now in scheduler.php
// Use sica_add_scheduler_log_entry() for activity logging
// Use sica_log() for technical debugging (writes to files)
