<?php
/**
 * Shared Article Generation Progress Component
 * 
 * Reusable component for showing real-time progress during AI article generation
 * Can be used in welcome screen, manual generation, or anywhere article creation happens
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Render the article generation progress component
 */
function render_article_progress_component($container_id = 'article-progress') {
    ?>
    <div id="<?php echo esc_attr($container_id); ?>" class="article-progress-component">
        <div class="progress-header">
            <h4 class="progress-title">Generating Your Article</h4>
            <span class="progress-percentage">0%</span>
        </div>
        
        <!-- Progress Bar -->
        <div class="progress-bar-container">
            <div class="progress-bar">
                <div class="progress-fill" style="width: 0%;"></div>
            </div>
        </div>
        
        <!-- Current Step Display -->
        <div class="progress-current-step">
            <span class="step-icon">⚡</span>
            <span class="step-text">Starting generation...</span>
        </div>
        
        <!-- Detailed Steps List -->
        <div class="progress-steps-list">
            <div class="progress-step" data-step="analyze">
                <div class="step-indicator">
                    <span class="step-number">1</span>
                    <span class="step-status">pending</span>
                </div>
                <div class="step-content">
                    <div class="step-title">Analyzing Business Context</div>
                    <div class="step-description">Understanding your industry and target keywords</div>
                </div>
            </div>
            
            <div class="progress-step" data-step="title">
                <div class="step-indicator">
                    <span class="step-number">2</span>
                    <span class="step-status">pending</span>
                </div>
                <div class="step-content">
                    <div class="step-title">Creating Engaging Title</div>
                    <div class="step-description">Crafting SEO-optimized, click-worthy headlines</div>
                </div>
            </div>
            
            <div class="progress-step" data-step="research">
                <div class="step-indicator">
                    <span class="step-number">3</span>
                    <span class="step-status">pending</span>
                </div>
                <div class="step-content">
                    <div class="step-title">Research & Fact-Checking</div>
                    <div class="step-description">Gathering current information and verifying facts</div>
                </div>
            </div>
            
            <div class="progress-step" data-step="content">
                <div class="step-indicator">
                    <span class="step-number">4</span>
                    <span class="step-status">pending</span>
                </div>
                <div class="step-content">
                    <div class="step-title">Writing Article Content</div>
                    <div class="step-description">Creating comprehensive, valuable content</div>
                </div>
            </div>
            
            <div class="progress-step" data-step="seo">
                <div class="step-indicator">
                    <span class="step-number">5</span>
                    <span class="step-status">pending</span>
                </div>
                <div class="step-content">
                    <div class="step-title">SEO Optimization</div>
                    <div class="step-description">Adding schema markup and meta descriptions</div>
                </div>
            </div>
            
            <div class="progress-step" data-step="image">
                <div class="step-indicator">
                    <span class="step-number">6</span>
                    <span class="step-status">pending</span>
                </div>
                <div class="step-content">
                    <div class="step-title">Generating Featured Image</div>
                    <div class="step-description">Creating custom AI-generated visuals</div>
                </div>
            </div>
            
            <div class="progress-step" data-step="finalize">
                <div class="step-indicator">
                    <span class="step-number">7</span>
                    <span class="step-status">pending</span>
                </div>
                <div class="step-content">
                    <div class="step-title">Final Review</div>
                    <div class="step-description">Polishing and preparing for publication</div>
                </div>
            </div>
        </div>
        
        <!-- Estimated Time -->
        <div class="progress-time-estimate">
            <span class="time-icon">⏱️</span>
            <span class="time-text">Estimated time: <span class="time-remaining">2-3 minutes</span></span>
        </div>
        
        <!-- Cancel Button -->
        <div class="progress-actions">
            <button class="progress-cancel-btn" onclick="cancelArticleGeneration()">
                Cancel Generation
            </button>
        </div>
    </div>
    
    <style>
    .article-progress-component {
        background: #f8f9fa;
        border: 1px solid #e9ecef;
        border-radius: 12px;
        padding: 24px;
        max-width: 600px;
        margin: 0 auto;
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    }
    
    .progress-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 20px;
    }
    
    .progress-title {
        margin: 0;
        color: #343a40;
        font-size: 18px;
        font-weight: 600;
    }
    
    .progress-percentage {
        font-size: 16px;
        font-weight: 600;
        color: #007cba;
        background: #e3f2fd;
        padding: 4px 12px;
        border-radius: 20px;
    }
    
    .progress-bar-container {
        margin-bottom: 20px;
    }
    
    .progress-bar {
        background: #e9ecef;
        height: 8px;
        border-radius: 4px;
        overflow: hidden;
        position: relative;
    }
    
    .progress-fill {
        background: linear-gradient(90deg, #007cba, #00a0d2);
        height: 100%;
        transition: width 0.5s ease;
        border-radius: 4px;
    }
    
    .progress-current-step {
        display: flex;
        align-items: center;
        gap: 8px;
        padding: 12px 16px;
        background: #fff;
        border: 1px solid #dee2e6;
        border-radius: 8px;
        margin-bottom: 20px;
        font-weight: 500;
        color: #495057;
    }
    
    .step-icon {
        font-size: 18px;
    }
    
    .progress-steps-list {
        display: none; /* Hidden by default, can be toggled */
        gap: 12px;
        flex-direction: column;
        margin-bottom: 20px;
    }
    
    .progress-step {
        display: flex;
        align-items: flex-start;
        gap: 12px;
        padding: 8px 0;
        opacity: 0.5;
        transition: opacity 0.3s ease;
    }
    
    .progress-step.active {
        opacity: 1;
    }
    
    .progress-step.completed {
        opacity: 0.7;
    }
    
    .step-indicator {
        display: flex;
        align-items: center;
        justify-content: center;
        position: relative;
        flex-shrink: 0;
    }
    
    .step-number {
        display: flex;
        align-items: center;
        justify-content: center;
        width: 24px;
        height: 24px;
        border-radius: 50%;
        background: #e9ecef;
        color: #6c757d;
        font-size: 12px;
        font-weight: 600;
        transition: all 0.3s ease;
    }
    
    .progress-step.active .step-number {
        background: #007cba;
        color: white;
    }
    
    .progress-step.completed .step-number {
        background: #28a745;
        color: white;
    }
    
    .step-content {
        flex: 1;
    }
    
    .step-title {
        font-weight: 500;
        color: #343a40;
        margin-bottom: 2px;
    }
    
    .step-description {
        font-size: 13px;
        color: #6c757d;
        line-height: 1.4;
    }
    
    .progress-time-estimate {
        display: flex;
        align-items: center;
        gap: 8px;
        padding: 12px 16px;
        background: #fff3cd;
        border: 1px solid #ffeaa7;
        border-radius: 6px;
        margin-bottom: 16px;
        font-size: 14px;
        color: #856404;
    }
    
    .progress-actions {
        text-align: center;
    }
    
    .progress-cancel-btn {
        background: none;
        border: 1px solid #dc3545;
        color: #dc3545;
        padding: 8px 16px;
        border-radius: 4px;
        cursor: pointer;
        font-size: 14px;
        transition: all 0.3s ease;
    }
    
    .progress-cancel-btn:hover {
        background: #dc3545;
        color: white;
    }
    
    /* Responsive */
    @media (max-width: 768px) {
        .article-progress-component {
            padding: 16px;
            margin: 0 16px;
        }
        
        .progress-header {
            flex-direction: column;
            align-items: flex-start;
            gap: 8px;
        }
    }
    </style>
    <?php
}

/**
 * JavaScript functions for progress component
 */
function progress_component_scripts() {
    ?>
    <script>
    class ArticleProgressTracker {
        constructor(containerId = 'article-progress') {
            this.container = document.getElementById(containerId);
            this.currentStep = 0;
            this.totalSteps = 7;
            this.steps = [
                'analyze', 'title', 'research', 'content', 'seo', 'image', 'finalize'
            ];
            this.stepMessages = {
                'analyze': 'Analyzing your business and industry...',
                'title': 'Creating engaging, SEO-optimized title...',
                'research': 'Researching current information and trends...',
                'content': 'Writing comprehensive article content...',
                'seo': 'Optimizing for search engines...',
                'image': 'Generating custom featured image...',
                'finalize': 'Final review and polishing...'
            };
        }
        
        updateProgress(stepName, percentage, customMessage = null) {
            // Update percentage
            const percentageEl = this.container?.querySelector('.progress-percentage');
            const progressFill = this.container?.querySelector('.progress-fill');
            
            if (percentageEl) percentageEl.textContent = percentage + '%';
            if (progressFill) progressFill.style.width = percentage + '%';
            
            // Update current step
            const currentStepEl = this.container?.querySelector('.step-text');
            if (currentStepEl) {
                currentStepEl.textContent = customMessage || this.stepMessages[stepName] || 'Processing...';
            }
            
            // Update step indicators
            this.updateStepIndicators(stepName);
            
            // Update time estimate
            this.updateTimeEstimate(percentage);
        }
        
        updateStepIndicators(activeStepName) {
            const stepIndex = this.steps.indexOf(activeStepName);
            const stepElements = this.container?.querySelectorAll('.progress-step');
            
            stepElements?.forEach((el, index) => {
                el.classList.remove('active', 'completed');
                
                if (index < stepIndex) {
                    el.classList.add('completed');
                } else if (index === stepIndex) {
                    el.classList.add('active');
                }
            });
        }
        
        updateTimeEstimate(percentage) {
            const timeEl = this.container?.querySelector('.time-remaining');
            if (!timeEl) return;
            
            const remainingPercentage = 100 - percentage;
            const estimatedMinutes = Math.ceil((remainingPercentage / 100) * 3); // 3 minutes total estimate
            
            if (estimatedMinutes <= 1) {
                timeEl.textContent = 'Almost done!';
            } else {
                timeEl.textContent = `${estimatedMinutes} minutes remaining`;
            }
        }
        
        complete(successMessage = 'Article generated successfully!') {
            this.updateProgress('finalize', 100, successMessage);
            
            // Hide cancel button
            const cancelBtn = this.container?.querySelector('.progress-cancel-btn');
            if (cancelBtn) cancelBtn.style.display = 'none';
            
            // Update time estimate
            const timeEl = this.container?.querySelector('.time-remaining');
            if (timeEl) timeEl.textContent = 'Complete!';
        }
        
        error(errorMessage = 'Generation failed. Please try again.') {
            const currentStepEl = this.container?.querySelector('.step-text');
            if (currentStepEl) {
                currentStepEl.innerHTML = '❌ ' + errorMessage;
                currentStepEl.style.color = '#dc3545';
            }
            
            // Show retry option
            const actionsEl = this.container?.querySelector('.progress-actions');
            if (actionsEl) {
                actionsEl.innerHTML = `
                    <button class="progress-cancel-btn" onclick="retryGeneration()" style="background: #007cba; color: white; border-color: #007cba;">
                        🔄 Try Again
                    </button>
                `;
            }
        }
        
        toggleDetailedSteps(show = true) {
            const stepsList = this.container?.querySelector('.progress-steps-list');
            if (stepsList) {
                stepsList.style.display = show ? 'flex' : 'none';
            }
        }
    }
    
    // Global function to cancel generation
    function cancelArticleGeneration() {
        if (confirm('Are you sure you want to cancel article generation?')) {
            // Trigger cancel via AJAX or other method
            // Article generation cancelled
            
            // Hide progress component or redirect
            const progressComponent = document.querySelector('.article-progress-component');
            if (progressComponent) {
                progressComponent.style.display = 'none';
            }
        }
    }
    
    // Global function for retry
    function retryGeneration() {
        location.reload(); // Simple retry - reload the page
    }
    
    // Make ArticleProgressTracker globally available
    window.ArticleProgressTracker = ArticleProgressTracker;
    </script>
    <?php
}

/**
 * Enqueue progress component scripts
 */
function enqueue_progress_component_scripts() {
    // Only load on pages that might use the progress component
    $screen = get_current_screen();
    if (!$screen || strpos($screen->id, 'sica') === false) {
        return;
    }
    
    add_action('admin_footer', 'progress_component_scripts');
}
add_action('admin_enqueue_scripts', 'enqueue_progress_component_scripts');

/**
 * AJAX handler for real-time progress updates
 */
function handle_progress_update() {
    check_ajax_referer('sica_progress_nonce', 'nonce');
    
    $step = sanitize_text_field($_POST['step']);
    $percentage = intval($_POST['percentage']);
    $message = sanitize_text_field($_POST['message']);
    
    // Store progress in transient for polling-based updates
    $progress_data = array(
        'step' => $step,
        'percentage' => $percentage,
        'message' => $message,
        'timestamp' => time()
    );
    
    set_transient('sica_progress_' . get_current_user_id(), $progress_data, 300); // 5 minutes
    
    wp_send_json_success($progress_data);
}
add_action('wp_ajax_sica_progress_update', 'handle_progress_update');

/**
 * AJAX handler to get current progress
 */
function get_current_progress() {
    check_ajax_referer('sica_progress_nonce', 'nonce');
    
    $progress_data = get_transient('sica_progress_' . get_current_user_id());
    
    if ($progress_data) {
        wp_send_json_success($progress_data);
    } else {
        wp_send_json_error('No progress data found');
    }
}
add_action('wp_ajax_sica_get_progress', 'get_current_progress');