<?php
/**
 * Prompts Management
 * 
 * Handles AI prompt templates, customization, and management
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Render the prompts tab interface
 */
function my_auto_blog_prompts_tab() {
    // Determine if free tier (lite) with robust fallback logic
    $is_free_tier = true;
    if (function_exists('sica_check_subscription_status')) {
        $status = sica_check_subscription_status();
        $customer_email = get_option('sica_customer_email', '');
        $is_email_verified = !empty($customer_email) && get_option('sica_email_verified', false);
        
        // Use robust tier detection with fallback logic
        if ($status && $status['ok'] && $is_email_verified) {
            $tier = $status['tier'] ?? 'lite';
            $is_free_tier = ($tier === 'lite');
        } else {
            // Fallback to local license data when server is unavailable
            $license_status = get_option('sica_license_status', 'inactive');
            $license_type = get_option('sica_license_type', 'free');
            $tier = $license_status === 'active' && $license_type === 'paid' ? 'starter' : 'lite';
            $is_free_tier = ($tier === 'lite');
        }
    }
    // Display success message if settings were updated
    if (get_transient('my_auto_blog_prompts_updated')) {
        ?>
        <div class="notice notice-success is-dismissible">
            <p>AI prompts updated successfully.</p>
        </div>
        <?php
        delete_transient('my_auto_blog_prompts_updated');
    }

    // Get current prompt settings
    $master_prompt = get_option('my_auto_blog_master_prompt');
    $default_prompt = get_default_blog_prompt();
    $is_using_custom = ($master_prompt !== false);
    $display_prompt = $is_using_custom ? $master_prompt : $default_prompt;

    // Get image prompt settings
    $image_style_prompt = get_option('my_auto_blog_image_style_prompt');
    $default_image_prompt = get_default_image_prompt();
    
    // Check if user's prompt is actually custom (different from all default templates)
    $is_using_custom_image = false;
    
    // If there's no saved prompt or it's empty, it's using default
    if ($image_style_prompt === false || empty(trim($image_style_prompt))) {
        $is_using_custom_image = false;
    } else {
        // Get all template styles to compare against (including the default)
        $template_styles = ['professional_photography', 'cinematic_dramatic', 'graphic_novel', 'minimalist_clean', 'macro_detailed', 'abstract_conceptual', 'cartoon_friendly'];
        $is_template_match = false;
        
        foreach ($template_styles as $style) {
            $template = get_style_template($style);
            if (trim($image_style_prompt) === trim($template)) {
                $is_template_match = true;
                break;
            }
        }
        
        // It's custom only if it doesn't match any template
        $is_using_custom_image = !$is_template_match;
    }
    
    $display_image_prompt = ($image_style_prompt !== false && !empty(trim($image_style_prompt))) ? $image_style_prompt : $default_image_prompt;

    // Get business details for variable preview
    $business_name = get_bloginfo('name') ?: 'Not set';
    $business_category = get_option('my_auto_blog_business_category', '') ?: 'Not set';
    $business_service_area = get_option('my_auto_blog_service_area', '') ?: 'Not set';
    $products_services = get_option('my_auto_blog_products_services', '') ?: 'Not set';
    $keywords = get_option('my_auto_blog_keywords', '') ?: 'Not set';
    $keywords_str = !empty($keywords) ? implode(", ", array_filter(array_map('trim', explode("\n", $keywords)))) : 'Not set';
    ?>

    <h3><?php echo esc_html__('AI Prompt Management', 'sica-content-ai'); ?></h3>
    <p><?php echo esc_html__('Customize how AI generates your blog content and images. Use variables to make prompts dynamic and personalized.', 'sica-content-ai'); ?></p>

    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
        <input type="hidden" name="action" value="save_prompts_settings">
        <?php wp_nonce_field('save_prompts_settings_nonce', 'prompts_nonce'); ?>
        
        <div class="sica-ai-settings-container">
            <!-- Card: Blog Gen Prompt -->
                    <div class="sica-card">
                        <div class="sica-card-header">
                            <span class="sica-card-title">Blog Gen Prompt</span>
                            <span class="sica-card-header-right">
                                <a href="#" id="reset_prompt_btn" class="sica-link" style="margin-right:8px;">Reset to Default</a>
                                <?php if ($is_using_custom): ?>
                                    <span class="sica-badge sica-badge-info">Custom</span>
                                <?php else: ?>
                                    <span class="sica-badge sica-badge-success">Default</span>
                                <?php endif; ?>
                                <span id="blog_prompt_unsaved" class="sica-badge sica-badge-info" style="display:none;">Unsaved</span>
                            </span>
                        </div>
                        <div class="sica-card-body">

                            <textarea id="master_blog_prompt" name="master_blog_prompt" class="sica-textarea" style="height: 300px;" <?php echo $is_free_tier ? 'readonly' : ''; ?>><?php echo esc_textarea($display_prompt); ?></textarea>
                            <p class="sica-help">This prompt controls how AI writes your blog content. Use variables below to personalize for your business.</p>

                            <!-- Available Variables Dropdown -->
                            <div class="sica-accordion">
                                <button type="button" id="toggle_variables" class="sica-accordion-toggle">
                                    <span id="toggle_text">▼ Available Variables</span>
                                </button>
                                <div id="variable_reference" class="sica-accordion-panel" style="display:none;">
                                    <h4 style="margin-top: 0; color: #495057;">📋 Supported Variables & Current Values</h4>
                                    <p class="sica-help">These variables will be automatically replaced when generating blog content:</p>
                                    
                                    <div style="display: grid; grid-template-columns: 1fr; gap: 8px; font-size: 12px;">
                                        <div style="border-bottom: 1px solid #e5e7eb; padding-bottom: 8px;">
                                            <code>{business_name}</code>
                                            <span style="margin-left: 10px; color: #0d6efd;">→ <?php echo esc_html($business_name); ?></span>
                                        </div>
                                        <div style="border-bottom: 1px solid #e5e7eb; padding-bottom: 8px;">
                                            <code>{business_category}</code>
                                            <span style="margin-left: 10px; color: #0d6efd;">→ <?php echo esc_html($business_category); ?></span>
                                        </div>
                                        <div style="border-bottom: 1px solid #e5e7eb; padding-bottom: 8px;">
                                            <code>{service_area}</code>
                                            <span style="margin-left: 10px; color: #0d6efd;">→ <?php echo esc_html($business_service_area); ?></span>
                                        </div>
                                        <div style="border-bottom: 1px solid #e5e7eb; padding-bottom: 8px;">
                                            <code>{products_services}</code>
                                            <span style="margin-left: 10px; color: #0d6efd;">→ <?php echo esc_html(strlen($products_services) > 60 ? substr($products_services, 0, 60) . '...' : $products_services); ?></span>
                                        </div>
                                        <div style="border-bottom: 1px solid #e5e7eb; padding-bottom: 8px;">
                                            <code>{title}</code>
                                            <span style="margin-left: 10px; color: #6c757d; font-style: italic;">→ Dynamic (blog post title)</span>
                                        </div>
                                        <div style="padding-bottom: 8px;">
                                            <code>{keywords_str}</code>
                                            <span style="margin-left: 10px; color: #0d6efd;">→ <?php echo esc_html(strlen($keywords_str) > 60 ? substr($keywords_str, 0, 60) . '...' : $keywords_str); ?></span>
                                        </div>
                                    </div>

                                    <div class="sica-alert sica-alert-warning" style="margin-top:12px;">
                                        ⚠️ Important: Use only the variables listed above. Unknown variables like <code>{custom_var}</code> will be left as‑is in your blog content.
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

            <!-- Card: Image Gen Model -->
                    <div class="sica-card">
                        <div class="sica-card-header">
                            <span class="sica-card-title">Image Gen Model</span>
                            <?php if ($is_free_tier): ?><span class="sica-badge sica-badge-warning">Lite</span><?php endif; ?>
                        </div>
                        <div class="sica-card-body">
                            <?php 
                            // Set tier-based defaults: DALL-E 3 for free tier (cheaper), GPT-Image-1 for paid
                            // Master and premium tiers get gpt-image-1, free tier gets dall-e-3
            $tier_default = $is_free_tier ? 'dall-e-3' : 'gpt-image-1';
                            $current_image_model = get_option('sica_image_model', $tier_default); 
                            ?>
                            <select name="image_model" id="image_model" class="sica-input" <?php echo $is_free_tier ? 'disabled' : ''; ?>>
                                <option value="gpt-image-1" <?php selected($current_image_model, 'gpt-image-1'); ?>>OpenAI GPT-Image-1 (Latest, higher quality)</option>
                                <option value="dall-e-3" <?php selected($current_image_model, 'dall-e-3'); ?>>DALL-E 3 (Faster, creative)</option>
                            </select>
                            <?php if ($is_free_tier): ?>
                            <div class="sica-help">You can choose from more image models when you upgrade.</div>
                            <?php else: ?>
                            <div class="sica-help">Choose from available image models.</div>
                            <?php endif; ?>
                        </div>
                    </div>

                    <!-- Card: Image Gen Prompt -->
                    <div class="sica-card" style="margin-top:16px;">
                        <div class="sica-card-header">
                            <span class="sica-card-title">Image Gen Prompt</span>
                            <span class="sica-card-header-right">
                                <a href="#" id="reset_image_prompt_btn" class="sica-link" style="margin-right:8px;">Reset to Default</a>
                                <?php if ($is_using_custom_image): ?>
                                    <span class="sica-badge sica-badge-info">Custom</span>
                                <?php else: ?>
                                    <span class="sica-badge sica-badge-success">Default</span>
                                <?php endif; ?>
                                <span id="image_prompt_unsaved" class="sica-badge sica-badge-info" style="display:none;"><?php echo esc_html__('Unsaved', 'sica-content-ai'); ?></span>
                            </span>
                        </div>
                        <div class="sica-card-body">
                            <textarea id="image_style_prompt" name="image_style_prompt" rows="6" class="sica-textarea" placeholder="Enter your custom prompt or use a template below..." <?php echo $is_free_tier ? 'readonly' : ''; ?>><?php echo esc_textarea($display_image_prompt); ?></textarea>
                            <p class="sica-help">This prompt controls how AI generates featured images for your blog posts.</p>

                            <!-- Template Browser Accordion -->
                            <div class="sica-accordion">
                                <button type="button" id="toggle_template_browser" class="sica-accordion-toggle">
                                    <span id="template_toggle_text">▼ Browse Prompt Templates</span>
                                </button>
                                <div id="template_browser_section" class="sica-accordion-panel" style="display:none;">
                                    <label for="template_browser" class="sica-label">Select Template Style</label>
                                    <?php $image_style = get_option('my_auto_blog_image_style', 'professional_photography'); ?>
                                    <select id="template_browser" class="sica-input" <?php echo $is_free_tier ? 'disabled' : ''; ?>>
                                        <option value="professional_photography" <?php selected($image_style, 'professional_photography'); ?>>📸 Professional Photography</option>
                                        <option value="cinematic_dramatic" <?php selected($image_style, 'cinematic_dramatic'); ?>>🎬 Cinematic/Dramatic</option>
                                        <option value="graphic_novel" <?php selected($image_style, 'graphic_novel'); ?>>🎨 Graphic Novel/Illustration</option>
                                        <option value="minimalist_clean" <?php selected($image_style, 'minimalist_clean'); ?>>✨ Minimalist/Clean</option>
                                        <option value="macro_detailed" <?php selected($image_style, 'macro_detailed'); ?>>🔍 Macro/Detail Focus</option>
                                        <option value="abstract_conceptual" <?php selected($image_style, 'abstract_conceptual'); ?>>🌀 Abstract/Conceptual</option>
                                        <option value="cartoon_friendly" <?php selected($image_style, 'cartoon_friendly'); ?>>😊 Cartoon/Friendly</option>
                                    </select>
                                    <div id="template_preview" class="sica-preview">
                                        <div class="sica-preview-header">
                                            <strong>📋 Template Preview</strong>
                                            <button type="button" id="apply_template_btn" class="button button-primary" <?php echo $is_free_tier ? 'disabled' : ''; ?>>📥 Apply This Template</button>
                                        </div>
                                        <div id="template_preview_text" class="sica-preview-text"></div>
                                    </div>
                                    <input type="hidden" id="image_style" name="image_style" value="<?php echo esc_attr($image_style); ?>">
                                </div>
                            </div>

                            <!-- Available Variables Dropdown -->
                            <div class="sica-accordion">
                                <button type="button" id="toggle_image_variables" class="sica-accordion-toggle">
                                    <span id="image_toggle_text">▼ Available Variables</span>
                                </button>
                                <div id="image_variable_reference" class="sica-accordion-panel" style="display:none;">
                                    <p class="sica-help">These variables will be automatically replaced when generating images:</p>
                                    <div style="display: grid; grid-template-columns: 1fr; gap: 8px; font-size: 12px;">
                                        <div style="border-bottom: 1px solid #e5e7eb; padding-bottom: 8px;">
                                            <code>{title}</code>
                                            <span style="margin-left: 10px; color: #6c757d; font-style: italic;">→ Dynamic (blog post title)</span>
                                        </div>
                                        <div style="border-bottom: 1px solid #e5e7eb; padding-bottom: 8px;">
                                            <code>{business_name}</code>
                                            <span style="margin-left: 10px; color: #0d6efd;">→ <?php echo esc_html($business_name); ?></span>
                                        </div>
                                        <div style="border-bottom: 1px solid #e5e7eb; padding-bottom: 8px;">
                                            <code>{business_category}</code>
                                            <span style="margin-left: 10px; color: #0d6efd;">→ <?php echo esc_html($business_category); ?></span>
                                        </div>
                                        <div style="border-bottom: 1px solid #e5e7eb; padding-bottom: 8px;">
                                            <code>{service_area}</code>
                                            <span style="margin-left: 10px; color: #0d6efd;">→ <?php echo esc_html($business_service_area); ?></span>
                                        </div>
                                        <div style="padding-bottom: 8px;">
                                            <code>{products_services}</code>
                                            <span style="margin-left: 10px; color: #0d6efd;">→ <?php echo esc_html(strlen($products_services) > 60 ? substr($products_services, 0, 60) . '...' : $products_services); ?></span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
        </div>

        <?php 
            if ($is_free_tier) {
                echo '<p class="description" style="margin-top:12px;">Editing prompts is a Starter feature. You can preview the default prompts here.</p>';
                submit_button('Upgrade to edit prompts', 'secondary', '', false, [
                    'onclick' => "window.open('https://sica.ai/pricing','_blank')",
                ]);
            } else {
                submit_button('Save AI Settings');
            }
        ?>
    </form>

    <script>
    jQuery(document).ready(function($) {
        // Toggle variable reference panel
        $('#toggle_variables').click(function() {
            var $panel = $('#variable_reference');
            var $text = $('#toggle_text');
            
            if ($panel.is(':visible')) {
                $panel.slideUp(200);
                $text.text('▼ Available Variables');
            } else {
                $panel.slideDown(200);
                $text.text('▲ Available Variables');
            }
        });

        // Toggle image variables reference panel
        $('#toggle_image_variables').click(function() {
            var $panel = $('#image_variable_reference');
            var $text = $('#image_toggle_text');
            
            if ($panel.is(':visible')) {
                $panel.slideUp(200);
                $text.text('▼ Available Variables');
            } else {
                $panel.slideDown(200);
                $text.text('▲ Available Variables');
            }
        });
        
        // Blog prompt unsaved changes tracking (still tracks but fields may be read-only on free tier)
        var originalPrompt = $('#master_blog_prompt').val();
        $('#master_blog_prompt').on('input', function() {
            var currentPrompt = $(this).val();
            var $unsavedIndicator = $('#blog_prompt_unsaved');
            
            if (currentPrompt !== originalPrompt) {
                $unsavedIndicator.show();
            } else {
                $unsavedIndicator.hide();
            }
        });
        
        // Reset prompt button
        $('#reset_prompt_btn').on('click', function(e) {
            e.preventDefault();
            if (confirm('Are you sure you want to reset the blog prompt to default? This will lose any custom changes.')) {
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'reset_prompt_and_reload',
                        _ajax_nonce: '<?php echo esc_attr(wp_create_nonce("my_auto_blog_nonce")); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            location.reload();
                        }
                    }
                });
            }
        });
        
        // Template browser accordion toggle
        $('#toggle_template_browser').click(function() {
            var $section = $('#template_browser_section');
            var $toggleText = $('#template_toggle_text');
            
            if ($section.is(':visible')) {
                $section.slideUp(300);
                $toggleText.text('▼ Browse Prompt Templates');
            } else {
                $section.slideDown(300);
                $toggleText.text('▲ Browse Prompt Templates');
                
                // Load template preview when accordion opens (if not already loaded)
                if ($('#template_preview_text').text() === '') {
                    var currentStyle = $('#template_browser').val();
                    loadTemplatePreview(currentStyle);
                }
            }
        });
        
        // Template browser functionality
        function loadTemplatePreview(style) {
            $.post(ajaxurl, {
                action: 'get_style_template',
                style: style,
                _ajax_nonce: '<?php echo esc_attr(wp_create_nonce("my_auto_blog_nonce")); ?>',
                cache_bust: Date.now()
            }, function(response) {
                if (response.success) {
                    $('#template_preview_text').text(response.data);
                } else {
                    $('#template_preview_text').text('Error loading template');
                }
            }).fail(function() {
                $('#template_preview_text').text('Network error loading template');
            });
        }
        
        // Template browser change handler
        $('#template_browser').change(function() {
            if (<?php echo $is_free_tier ? 'true' : 'false'; ?>) { return; }
            var selectedStyle = $(this).val();
            loadTemplatePreview(selectedStyle);
        });
        
        // Apply template button functionality
        $('#apply_template_btn').click(function() {
            if (<?php echo $is_free_tier ? 'true' : 'false'; ?>) { return; }
            var currentPrompt = $('#image_style_prompt').val().trim();
            var selectedStyle = $('#template_browser').val();
            var styleName = $('#template_browser option:selected').text();
            
            if (currentPrompt !== '' && currentPrompt !== $('#template_preview_text').text()) {
                var confirmMessage = '⚠️ Apply ' + styleName + ' template?\n\n' +
                                    'This will replace your current prompt with the template.\n' +
                                    'Your custom work will be lost.\n\n' +
                                    'Continue?';
                
                if (!confirm(confirmMessage)) {
                    return;
                }
            }
            
            // Apply the template
            var templateText = $('#template_preview_text').text();
            $('#image_style_prompt').val(templateText);
            $('#image_style').val(selectedStyle);
        });
        
        // Reset image prompt functionality
        $('#reset_image_prompt_btn').click(function(e) {
            e.preventDefault();
            if (confirm('⚠️ Reset to default?\n\nThis will clear your custom prompt and use the default intelligent image generation.')) {
                // Clear the field to use default behavior
                $('#image_style_prompt').val('');
                // Update the template browser selection
                $('#template_browser').val('professional_photography');
                
                // Save the change immediately to update the status
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'reset_image_prompt_and_reload',
                        _ajax_nonce: '<?php echo esc_attr(wp_create_nonce("my_auto_blog_nonce")); ?>'
                    },
                    success: function(response) {
                        if (response.success) {
                            location.reload();
                        }
                    }
                });
            }
        });
        
        // Unsaved changes detection
        var originalBlogPrompt = $('#master_blog_prompt').val();
        var originalImagePrompt = $('#image_style_prompt').val();
        var hasUnsavedChanges = false;
        
        function updateUnsavedState() {
            var currentBlogPrompt = $('#master_blog_prompt').val();
            var currentImagePrompt = $('#image_style_prompt').val();
            
            hasUnsavedChanges = (currentBlogPrompt !== originalBlogPrompt) || 
                               (currentImagePrompt !== originalImagePrompt);
            
            if (hasUnsavedChanges) {
                // Show unsaved indicator
                if (!$('#unsaved-indicator').length) {
                    $('input[type="submit"]').after('<span id="unsaved-indicator" style="color: #d63638; margin-left: 10px; font-weight: bold;">⚠️ Unsaved changes</span>');
                }
                $('input[type="submit"]').text('💾 Save AI Settings (Changes Detected)').addClass('button-primary').removeClass('button-secondary');
            } else {
                // Hide unsaved indicator
                $('#unsaved-indicator').remove();
                $('input[type="submit"]').text('Save AI Settings').removeClass('button-primary').addClass('button-secondary');
            }
        }
        
        // Monitor changes to prompts and image model
        $('#master_blog_prompt, #image_style_prompt, #image_model').on('input keyup paste change', function() {
            updateUnsavedState();
        });
        
        // Prevent navigation with unsaved changes
        $(window).on('beforeunload', function(e) {
            if (hasUnsavedChanges) {
                var message = 'You have unsaved changes to your prompts. Are you sure you want to leave?';
                e.originalEvent.returnValue = message;
                return message;
            }
        });
        
        // Clear unsaved state when form is being submitted
        $('form[action*="save_prompts_settings"]').on('submit', function() {
            hasUnsavedChanges = false;
        });
        
        // Also clear warning when save button is clicked
        $('input[type="submit"]').on('click', function() {
            hasUnsavedChanges = false;
        });
        
        // Update original values after successful save or template apply
        function updateOriginalValues() {
            originalBlogPrompt = $('#master_blog_prompt').val();
            originalImagePrompt = $('#image_style_prompt').val();
            hasUnsavedChanges = false;
            updateUnsavedState();
        }
        
        // Call updateUnsavedState on page load to set initial state
        updateUnsavedState();
    });
    </script>
    <?php
}

/**
 * Save prompts settings
 */
function save_prompts_settings() {
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized user');
    }

    check_admin_referer('save_prompts_settings_nonce', 'prompts_nonce');

    // Save blog prompt
    if (isset($_POST['master_blog_prompt'])) {
        $blog_prompt = wp_unslash(sanitize_textarea_field($_POST['master_blog_prompt']));
        if (!empty($blog_prompt)) {
            update_option('my_auto_blog_master_prompt', $blog_prompt);
        } else {
            delete_option('my_auto_blog_master_prompt');
        }
    }

    // Save image style and prompt
    if (isset($_POST['image_style'])) {
        update_option('my_auto_blog_image_style', sanitize_text_field(wp_unslash($_POST['image_style'])));
    }
    
    if (isset($_POST['image_style_prompt'])) {
        $image_prompt = wp_unslash(sanitize_textarea_field($_POST['image_style_prompt']));
        if (!empty($image_prompt)) {
            update_option('my_auto_blog_image_style_prompt', $image_prompt);
        } else {
            delete_option('my_auto_blog_image_style_prompt');
        }
    }

    // Save image model selection
    if (isset($_POST['image_model'])) {
        $image_model = sanitize_text_field(wp_unslash($_POST['image_model']));
        // Validate the model is one of the allowed options
        if (in_array($image_model, ['gpt-image-1', 'dall-e-3'])) {
            update_option('sica_image_model', $image_model);
        }
    }

    set_transient('my_auto_blog_prompts_updated', true, 30);

    $redirect_url = add_query_arg([
        'page' => 'ai-blogger',
        'tab' => 'prompts',
        'settings-updated' => 'true',
        'cache_bust' => time()
    ], admin_url('admin.php'));
    
    wp_redirect($redirect_url);
    exit;
}
add_action('admin_post_save_prompts_settings', 'save_prompts_settings');

/**
 * Get the default blog prompt
 */
function get_default_blog_prompt() {
    return 'You are an experienced {business_category} expert writing a comprehensive, authoritative blog post for {business_name} in {service_area}.' . "\n" .
'The topic is: "{title}"' . "\n\n" .
'RESEARCH REQUIREMENTS - SEARCH FOR CURRENT, LOCAL INFORMATION' . "\n" .
'Before writing, conduct targeted web searches to gather:' . "\n" .
'- Recent trends, statistics, and developments related to "{title}" in {service_area}' . "\n" .
'- Current pricing, regulations, or industry standards specific to {service_area}' . "\n" .
'- Local case studies, news, or recent examples from {service_area} area' . "\n" .
'- Up-to-date expert opinions, research studies, or industry reports on "{title}"' . "\n" .
'- Recent government announcements, bylaw changes, or policy updates affecting {business_category} in {service_area}' . "\n" .
'- Current market conditions, seasonal factors, or recent events impacting "{title}"' . "\n\n" .
'Cite these sources naturally throughout the content and include specific dates/timeframes when mentioning recent information.' . "\n\n" .
'CORE OBJECTIVES' . "\n" .
'- Deliver 1,800 – 2,800 words of high-quality content that demonstrates deep expertise (E-E-A-T)' . "\n" .
'- Help readers make informed decisions through actionable, locally relevant guidance using CURRENT information' . "\n" .
'- Earn backlinks, answer high-intent questions, and nudge readers toward using {business_name}' . "\n" .
'- Provide genuine value with up-to-date, locally-specific insights, not generic filler' . "\n\n" .
'WRITING APPROACH - AUTHORITATIVE WITHOUT PERSONAL CLAIMS' . "\n" .
'- Write with the authority of industry expertise WITHOUT claiming personal experience' . "\n" .
'- Lead with strong professional insights and "here\'s what industry experts know" perspectives' . "\n" .
'- Open with a compelling hook: surprising industry data, common costly mistakes, or contrarian insights' . "\n" .
'- Choose ONE engaging format that demonstrates knowledge:' . "\n" .
'  • "The Real Truth About..." (myth-busting with industry insights)' . "\n" .
'  • "What Industry Professionals Know About..." (knowledge-driven)' . "\n" .
'  • "Why Most {service_area} Clients Get This Wrong" (problem-solving focus)' . "\n" .
'  • "The Professional\'s Guide to..." (expert knowledge sharing)' . "\n" .
'  • "What Recent Industry Data Reveals About..." (data-driven insights)' . "\n\n" .
'VOICE GUIDELINES (Critical - NO Personal Claims):' . "\n" .
'❌ NEVER WRITE: "As a seasoned professional with 15+ years..."' . "\n" .
'❌ NEVER WRITE: "In my experience..." or "I\'ve seen..."' . "\n" .
'❌ NEVER WRITE: "After years in the industry..."' . "\n" .
'❌ NEVER WRITE: "I recommend..." or "I always tell clients..."' . "\n\n" .
'✅ DO WRITE: "Industry professionals recommend..."' . "\n" .
'✅ DO WRITE: "Experts in {business_category} consistently find..."' . "\n" .
'✅ DO WRITE: "Recent industry data shows..."' . "\n" .
'✅ DO WRITE: "Professional standards in {business_category} suggest..."' . "\n" .
'✅ DO WRITE: "Leading {business_category} professionals in {service_area} report..."' . "\n\n" .
'- Write with confident authority but NEVER claim personal experience or credentials' . "\n" .
'- Maintain authoritative tone through industry knowledge, not personal claims' . "\n\n" .
'CONTENT BALANCE:' . "\n" .
'- 60% professional insights and industry knowledge' . "\n" .
'- 25% supporting facts and local data (2-3 strategic citations max)' . "\n" .
'- 15% actionable advice and next steps' . "\n" .
'- Conversational yet professional tone - authoritative neighbor giving advice' . "\n\n" .
'STORYTELLING ELEMENTS (No Personal Claims):' . "\n" .
'- Start sections with industry scenarios: "Recent work in {service_area} demonstrates..." or "Industry case studies show..."' . "\n" .
'- Share professional insights: "Industry experts consistently find that..."' . "\n" .
'- Include professional recommendations: "Professional standards in {business_category} recommend..."' . "\n" .
'- Add authority: "Here\'s what most people don\'t realize..." or "Industry data consistently shows..."' . "\n" .
'- Use strategic citations to reinforce professional knowledge, not personal experience' . "\n\n" .
'SECTION STRUCTURE:' . "\n" .
'- Lead with your professional take, then support with data (not the reverse)' . "\n" .
'- Include a "Local Expert FAQ" section with answers that showcase insider knowledge' . "\n" .
'- End with confident recommendations and clear next steps' . "\n\n" .
'CONTENT ELEMENTS TO INCLUDE WHEN RELEVANT' . "\n" .
'- Real-world examples and case studies from {service_area}' . "\n" .
'- Expert tips, insider knowledge, best practices, common pitfalls' . "\n" .
'- Cost or ROI considerations, compliance guidelines, industry standards, future trends' . "\n\n" .
'HTML FORMATTING REQUIREMENTS' . "\n" .
'- `<h1>` for the main title (use {title})' . "\n" .
'- `<h2>` for major sections (4–6 recommended)' . "\n" .
'- `<h3>` for subsections as needed' . "\n" .
'- Wrap all paragraphs in `<p>` tags' . "\n" .
'- Use `<ul><li>` for bullet lists, `<ol><li>` for numbered steps' . "\n" .
'- Employ `<strong>` or `<em>` for emphasis sparingly' . "\n" .
'- Keep HTML clean and readable; break content into clear sections' . "\n\n" .
'STRUCTURE GUIDELINES' . "\n" .
'1. Engaging intro (hook)' . "\n" .
'2. Core sections under descriptive `<h2>` headings following chosen format' . "\n" .
'3. Inline FAQs where contextually appropriate' . "\n" .
'4. Key Takeaways list' . "\n" .
'5. Actionable closing paragraph with next-step recommendation' . "\n" .
'6. MANDATORY DISCLAIMER (see below)' . "\n\n" .
'MANDATORY DISCLAIMER - MUST INCLUDE AT END:' . "\n" .
'Add this disclaimer as the final paragraph before the call-to-action:' . "\n\n" .
'<p><em>While we aim for accuracy, please verify details for your specific situation. For personalized advice and to discuss how these insights apply to your specific needs, we\'d love to chat with you directly.</em></p>' . "\n\n" .
'REMEMBER' . "\n" .
'- Write for humans first; incorporate keywords naturally: ( {keywords_str} )' . "\n" .
'- Use vivid nouns, active verbs; purge filler phrases' . "\n" .
'- Support claims with current, web-researched specifics: recent numbers, local bylaws, expert observations with dates' . "\n" .
'- Prioritize fresh, locally-relevant information from your web searches over general knowledge' . "\n" .
'- Include source URLs and publication dates where appropriate for credibility' . "\n" .
'- Maintain professional local relevance for the target market using current data' . "\n" .
'- Ensure logical flow from section to section and clear HTML hierarchy' . "\n" .
'- ALWAYS include the mandatory disclaimer before the final call-to-action';
}

/**
 * Get the default image prompt
 */
function get_default_image_prompt() {
    return 'Create a blog featured image that VISUALLY ILLUSTRATES the subject of this article: "{title}".

CRITICAL: Analyze the article title and create an image showing the ACTUAL TOPIC being discussed.
- If about home repairs, construction, or damage: show the materials, damage patterns, or repair work
- If about services: show the service being performed or its results
- If about products: show the products in use
- If about a specific location or weather: show relevant environmental context

Professional photograph with high-quality composition and modern lighting. Sharp focus, clean aesthetic suitable for blog header. No text, no logos, no generic business imagery.';
}

/**
 * Get style template for image generation
 */
function get_style_template($style) {
    $style_templates = [
        'professional_photography' => 'Create a blog featured image that VISUALLY ILLUSTRATES the subject of this article: "{title}".

CRITICAL: Analyze the article title and create an image showing the ACTUAL TOPIC being discussed - not generic business imagery.
- If about home repairs/damage: show the actual materials, damage, or repair work
- If about services: show the service being performed or its results
- If about products: show the products in use or their benefits
- If about a location: show relevant local scenery or context

Professional editorial photograph with 85mm lens at f/2.8, studio-quality lighting with shallow depth of field. High-end commercial aesthetic with rich color grading. Clean composition following rule of thirds. Sharp focus, no text, no logos, no people unless essential to the topic.',

        'cinematic_dramatic' => 'Create a blog featured image that VISUALLY ILLUSTRATES the subject of this article: "{title}".

CRITICAL: The image must depict what the article is actually about - derive the visual subject from the title.
- Extract the main topic/subject from the title
- Show that subject in a dramatic, cinematic way
- Do NOT default to generic business or office imagery

Cinematic dramatic style shot with wide-angle lens (24mm), theatrical three-point lighting with deep shadows. Hollywood aesthetic with moody atmosphere and film grain. Dynamic composition with epic scale and dramatic tension. Rich contrast, movie poster quality. No text, no logos.',

        'graphic_novel' => 'Create a blog featured image that VISUALLY ILLUSTRATES the subject of this article: "{title}".

CRITICAL: Analyze the title and illustrate the ACTUAL TOPIC in graphic novel style.
- What is the article about? Show THAT subject.
- Home improvement? Show the materials/work in comic style.
- Technical topic? Illustrate the concept visually.
- Do NOT create generic business illustrations.

Stylized graphic novel illustration with bold graphic design and vibrant saturated colors. Comic book art style with clean vector-style lines, dynamic composition. Modern illustration aesthetic with visual metaphors making the concept engaging. No text, no logos.',

        'minimalist_clean' => 'Create a blog featured image that VISUALLY ILLUSTRATES the subject of this article: "{title}".

CRITICAL: Show the article\'s actual subject in minimalist style - not generic business imagery.
- Parse the title to identify the main visual subject
- Represent that subject with clean, minimal design
- Use relevant objects, materials, or concepts from the topic

Clean minimalist image with 50mm lens, soft even lighting, abundant negative space. Scandinavian aesthetic with geometric composition. Limited color palette with high contrast against clean background. Ultra-clean composition, sharp focus, no people, no text, no logos.',

        'macro_detailed' => 'Create a blog featured image that VISUALLY ILLUSTRATES the subject of this article: "{title}".

CRITICAL: Show extreme detail of whatever the article is actually about.
- If about materials (stucco, wood, etc.): macro shot of that material
- If about damage/repairs: detailed view of the issue or solution
- If about products: extreme close-up of product details

Extreme macro photograph with 100mm macro lens at f/8, focus stacking. Professional lighting reveals intricate textures and patterns relevant to the topic. National Geographic quality with rich detail. Sharp focus highlighting the subject matter. Ultra-detailed, texture-focused, no people, no text, no logos.',

        'abstract_conceptual' => 'Create a blog featured image that VISUALLY ILLUSTRATES the subject of this article: "{title}".

CRITICAL: Create an abstract representation of the article\'s actual topic.
- What concept or subject does the title convey?
- Create an artistic interpretation of THAT specific subject
- Do NOT default to generic abstract business art

Abstract conceptual artwork with sophisticated color palette and dynamic movement. Creative visual metaphor capturing the essence of the topic. Professional fine art photography with dramatic lighting. Thought-provoking composition. No people, no text, no logos.',

        'cartoon_friendly' => 'Create a blog featured image that VISUALLY ILLUSTRATES the subject of this article: "{title}".

CRITICAL: Illustrate what the article is actually about in friendly cartoon style.
- Analyze the title to identify the main subject
- Show that subject (materials, situations, concepts) in cartoon form
- Do NOT create generic business cartoons with laptops/offices

Friendly cartoon illustration in Pixar-quality digital artwork style. Warm inviting aesthetic with bright cheerful colors, soft rounded forms. Make the actual topic approachable and engaging. Clean illustration style with warm lighting and positive energy. No text, no logos.'
    ];
    
    $tpl = isset($style_templates[$style]) ? $style_templates[$style] : $style_templates['professional_photography'];
    // Force fresh string (avoid any opcache/shared reference oddities)
    return ''. $tpl;
}

/**
 * AJAX handler to get style template
 */
function ajax_get_style_template() {
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    
    $style = isset($_POST['style']) ? sanitize_text_field(wp_unslash($_POST['style'])) : '';
    $template = get_style_template($style);
    
    wp_send_json_success($template);
}
add_action('wp_ajax_get_style_template', 'ajax_get_style_template');

/**
 * AJAX handler to reset blog prompt
 */
function ajax_reset_prompt_and_reload() {
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    delete_option('my_auto_blog_master_prompt');
    wp_send_json_success('Blog prompt reset successfully');
}
add_action('wp_ajax_reset_prompt_and_reload', 'ajax_reset_prompt_and_reload');





/**
 * AJAX handler for saving prompts (auto-save functionality)
 */
function save_prompts_ajax() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized user');
        return;
    }

    check_admin_referer('save_prompts_settings_nonce', 'prompts_nonce');

    // Save blog prompt
    if (isset($_POST['master_blog_prompt'])) {
        $blog_prompt = wp_unslash(sanitize_textarea_field($_POST['master_blog_prompt']));
        if (!empty($blog_prompt)) {
            update_option('my_auto_blog_master_prompt', $blog_prompt);
        } else {
            delete_option('my_auto_blog_master_prompt');
        }
    }

    // Save image style and prompt
    if (isset($_POST['image_style'])) {
        update_option('my_auto_blog_image_style', sanitize_text_field(wp_unslash($_POST['image_style'])));
    }
    
    if (isset($_POST['image_style_prompt'])) {
        $image_prompt = wp_unslash(sanitize_textarea_field($_POST['image_style_prompt']));
        if (!empty($image_prompt)) {
            update_option('my_auto_blog_image_style_prompt', $image_prompt);
        } else {
            delete_option('my_auto_blog_image_style_prompt');
        }
    }

    // Save image model selection
    if (isset($_POST['image_model'])) {
        $image_model = sanitize_text_field(wp_unslash($_POST['image_model']));
        // Validate the model is one of the allowed options
        if (in_array($image_model, ['gpt-image-1', 'dall-e-3'])) {
            update_option('sica_image_model', $image_model);
        }
    }

    wp_send_json_success('AI settings saved successfully');
}
add_action('wp_ajax_save_prompts_ajax', 'save_prompts_ajax');