<?php
/**
 * Scheduler and Cron Management
 * 
 * Handles automated blog post scheduling and WordPress cron functionality
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Render the scheduler tab interface
 */
function my_auto_blog_scheduler_tab() {
    // Check if user has access to scheduler functionality using new subscription system
    $has_scheduling_access = function_exists('sica_tier_supports_feature') ? sica_tier_supports_feature('scheduling') : false;
    $is_free_tier = !$has_scheduling_access;
    
    // Get current scheduler status
    $cron_active = get_option('my_auto_blog_cron_active', 0);
    $current_schedule = get_option('my_auto_blog_schedule', 'weekly');
    $next_timestamp = wp_next_scheduled('my_auto_blog_cron_event');
    $upcoming_titles = array_filter(explode("\n", get_option('my_auto_blog_title', '')));
    
    // Calculate next run display
    if ($cron_active && $next_timestamp) {
        $next_run_text = wp_date('M j, Y g:i A', $next_timestamp);
        $next_run_relative = human_time_diff($next_timestamp, current_time('timestamp'));
        if ($next_timestamp > current_time('timestamp')) {
            $next_run_status = "in {$next_run_relative}";
        } else {
            $next_run_status = "{$next_run_relative} ago (may be pending)";
        }
    } elseif ($cron_active) {
        if (empty($upcoming_titles)) {
            $next_run_text = 'No titles in queue';
            $next_run_status = 'Add titles to enable scheduling';
        } else {
            $next_run_text = 'Scheduling in progress...';
            $next_run_status = 'Setting up schedule';
        }
    } else {
        $next_run_text = 'Scheduler paused';
        $next_run_status = 'Enable to start scheduling';
    }
    
    $start_date = get_option('my_auto_blog_start_date', date('Y-m-d'));
    $start_time = get_option('my_auto_blog_start_time', '09:00');
    ?>
    
    <div class="sica-ai-settings-container" style="margin-top: 20px;">
        
        <!-- Upgrade Banner as Card -->
        <?php if (isset($_GET['message']) && $_GET['message'] === 'upgrade_required'): ?>
        <div class="sica-card" style="margin-bottom: 20px; border-left: 4px solid #dc3545; background: #fff5f5;">
            <p style="margin: 0; color: #dc3545;"><strong>Paid Plan Required:</strong> Automatic scheduling is only available with a paid subscription. <a href="https://sica.ai/pricing" target="_blank">Upgrade now</a> to enable this feature.</p>
        </div>
        <?php endif; ?>
        
        <?php if ($is_free_tier): ?>
        <!-- Upgrade Notice Card -->
        <div class="sica-card" style="margin-bottom: 20px; border-left: 4px solid #2196f3; background: #f8faff;">
            <div style="display: flex; align-items: center; gap: 15px;">
                <div style="font-size: 24px;">🚀</div>
                <div style="flex: 1;">
                    <h3 style="margin: 0 0 8px 0; color: #1976d2; font-size: 16px;">Automated Scheduling Available with Paid Plans</h3>
                    <p style="margin: 0; color: #666; font-size: 14px;">
                        The scheduler interface is available to explore, but automatic post scheduling requires a paid subscription. 
                        You can still <strong>manually create posts</strong> anytime using the Content Planner.
                    </p>
                </div>
                <div>
                    <a href="https://sica.ai/pricing" target="_blank" class="button button-primary" style="background: #2196f3;">
                        View Pricing
                    </a>
                </div>
            </div>
        </div>
        <?php endif; ?>
        <!-- Configuration Form (First Card) -->
        <div class="sica-card">
            <div class="sica-card-header">
                <h3 class="sica-card-title"><?php echo esc_html__('Scheduler Configuration', 'sica-content-ai'); ?></h3>
            </div>
            <div class="sica-card-body">
                <p style="margin: 0 0 20px 0; color: #6b7280;"><?php echo esc_html__('Configure when and how often new posts are automatically created from your title queue.', 'sica-content-ai'); ?></p>
                
                <form id="automated-scheduler-form" method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                    <input type="hidden" name="action" value="save_scheduler_settings">
                    <?php wp_nonce_field('save_scheduler_settings_action', 'save_scheduler_settings_nonce'); ?>
                    
                    <!-- Active/Inactive Toggle -->
                    <div style="margin-bottom: 24px;">
                        <label for="my_auto_blog_cron_active" style="display: flex; align-items: center; gap: 10px; cursor: <?php echo $is_free_tier ? 'not-allowed' : 'pointer'; ?>; font-weight: 600; opacity: <?php echo $is_free_tier ? '0.6' : '1'; ?>;">
                            <input type="checkbox" id="my_auto_blog_cron_active" name="my_auto_blog_cron_active" value="1" <?php checked(1, $is_free_tier ? 0 : get_option('my_auto_blog_cron_active', 0)); ?> <?php disabled($is_free_tier); ?> style="transform: scale(1.2);" />
                            <?php echo esc_html__('Enable automatic post generation', 'sica-content-ai'); ?>
                            <?php if ($is_free_tier): ?>
                                <span style="color: #f59e0b; font-size: 12px; font-weight: normal;">(Paid Plan)</span>
                            <?php endif; ?>
                        </label>
                        <div style="margin-top: 6px; font-size: 13px; color: #6b7280;">
                            When enabled, posts will be automatically created from your title queue based on the schedule below.
                        </div>
                    </div>
                    
                    <!-- Schedule Configuration -->
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(180px, 1fr)); gap: 16px; margin-bottom: 24px;">
                        <div>
                            <label class="sica-label" for="my_auto_blog_start_date"><?php echo esc_html__('Next post date', 'sica-content-ai'); ?></label>
                            <input type="date" id="my_auto_blog_start_date" name="my_auto_blog_start_date" value="<?php echo esc_attr(get_option('my_auto_blog_start_date')); ?>" class="sica-input" <?php disabled($is_free_tier); ?> />
                            <div style="margin-top:6px;color:#6b7280;font-size:12px;">This is the date of your next scheduled post.</div>
                        </div>
                        <div>
                            <label class="sica-label" for="my_auto_blog_start_time"><?php echo esc_html__('Next post time', 'sica-content-ai'); ?></label>
                            <input type="time" id="my_auto_blog_start_time" name="my_auto_blog_start_time" value="<?php echo esc_attr(get_option('my_auto_blog_start_time')); ?>" class="sica-input" <?php disabled($is_free_tier); ?> />
                            <div style="margin-top:6px;color:#6b7280;font-size:12px;">Posts will publish at this time on the selected schedule.</div>
                        </div>
                        <div>
                            <label class="sica-label" for="my_auto_blog_schedule"><?php echo esc_html__('Frequency', 'sica-content-ai'); ?></label>
                            <select id="my_auto_blog_schedule" name="my_auto_blog_schedule" class="sica-input" <?php disabled($is_free_tier); ?>>
                                <option value="daily" <?php selected(get_option('my_auto_blog_schedule'), 'daily'); ?>><?php echo esc_html__('Daily', 'sica-content-ai'); ?></option>
                                <option value="every_three_days" <?php selected(get_option('my_auto_blog_schedule'), 'every_three_days'); ?>><?php echo esc_html__('Every 3 Days', 'sica-content-ai'); ?></option>
                                <option value="weekly" <?php selected(get_option('my_auto_blog_schedule'), 'weekly'); ?>><?php echo esc_html__('Weekly', 'sica-content-ai'); ?></option>
                                <option value="monthly" <?php selected(get_option('my_auto_blog_schedule'), 'monthly'); ?>><?php echo esc_html__('Monthly', 'sica-content-ai'); ?></option>
                            </select>
                            <div style="margin-top:6px;color:#6b7280;font-size:12px;">After the next post, subsequent posts will repeat on this schedule.</div>
                        </div>
                    </div>

                    <?php /* Removed inline preview to avoid duplication; Status card is the single source of truth */ ?>
                    
                    <?php if ($is_free_tier): ?>
                        <button type="button" disabled class="button button-primary" style="margin: 0; opacity: 0.6;">
                            Save Scheduler Settings (Paid Plan Required)
                        </button>
                    <?php else: ?>
                        <?php submit_button('Save Scheduler Settings', 'primary', 'submit', false, array('style' => 'margin: 0;')); ?>
                    <?php endif; ?>
                </form>
            </div>
        </div>

        <!-- Scheduler Status (Second Card) -->
        <div class="sica-card">
            <div class="sica-card-header">
                <h3 class="sica-card-title">Scheduler Status</h3>
                <div class="sica-card-header-right">
                    <span class="sica-badge <?php echo $cron_active ? 'sica-badge-success' : 'sica-badge-warning'; ?>">
                        <?php echo $cron_active ? 'Active' : 'Paused'; ?>
                    </span>
                </div>
            </div>
            <div class="sica-card-body">
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin-bottom: 16px;">
                    <div>
                        <div style="font-weight: 600; margin-bottom: 4px;">Next scheduled post</div>
                        <div style="color: #374151; margin-bottom: 2px;">
                            <?php echo esc_html($next_run_text); ?> (<?php echo esc_html(wp_date('T', time())); ?>)
                        </div>
                        <?php if ($cron_active && $next_timestamp): ?>
                            <div style="font-size: 12px; color: #6b7280;"><?php echo esc_html($next_run_status); ?></div>
                        <?php endif; ?>
                    </div>
                    <div>
                        <div style="font-weight: 600; margin-bottom: 4px;">Frequency</div>
                        <div style="color: #374151;">
                            <?php 
                            $freq_map = array(
                                'daily' => 'Daily',
                                'every_three_days' => 'Every 3 days',
                                'weekly' => 'Weekly',
                                'monthly' => 'Monthly'
                            );
                            echo esc_html($freq_map[$current_schedule] ?? 'Weekly');
                            ?>
                        </div>
                    </div>
                    <div>
                        <div style="font-weight: 600; margin-bottom: 4px;">Queue Status</div>
                        <div style="color: #374151;">
                            <?php echo count($upcoming_titles); ?> titles ready
                        </div>
                    </div>
                </div>
                
                <?php if (!$cron_active): ?>
                <div class="sica-alert sica-alert-warning">
                    <strong>Scheduler is paused.</strong> Enable automatic post generation above to start the schedule.
                </div>
                <?php elseif (empty($upcoming_titles)): ?>
                <div class="sica-alert sica-alert-warning">
                    <strong>No titles in queue.</strong> Add titles in the Content Planner to enable automatic posting.
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <?php
    // Display success message if log was cleared
    if (isset($_GET['log_cleared'])) {
        echo '<div class="notice notice-success is-dismissible"><p>Scheduler log cleared successfully.</p></div>';
    }
    
    // Display the scheduler activity log within the container
    echo '<div class="sica-ai-settings-container" style="margin-top: 0;">';
    sica_display_scheduler_log();
    echo '</div>';
    ?>
    <?php
}

/**
 * Save scheduler settings
 */
function save_scheduler_settings() {
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized user');
    }

    // Verify nonce
    check_admin_referer('save_scheduler_settings_action', 'save_scheduler_settings_nonce');
    
    // Check if user has access to scheduler functionality using new subscription system
    $has_scheduling_access = function_exists('sica_tier_supports_feature') ? sica_tier_supports_feature('scheduling') : false;
    if (!$has_scheduling_access) {
        // Redirect back with error message for free users
        $redirect_url = add_query_arg([
            'page' => 'ai-blogger',
            'tab' => 'scheduler',
            'settings-updated' => 'false',
            'message' => 'upgrade_required'
        ], admin_url('admin.php'));
        wp_redirect($redirect_url);
        exit;
    }

    // Sanitize and save the settings
    $cron_active = isset($_POST['my_auto_blog_cron_active']) ? 1 : 0;
    $start_date_raw = isset($_POST['my_auto_blog_start_date']) ? $_POST['my_auto_blog_start_date'] : '';
    $start_time_raw = isset($_POST['my_auto_blog_start_time']) ? $_POST['my_auto_blog_start_time'] : '';
    $schedule = sanitize_text_field($_POST['my_auto_blog_schedule']);

    // Normalize date to Y-m-d
    $start_date = sanitize_text_field($start_date_raw);
    if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', (string)$start_date)) {
        $ts = strtotime($start_date_raw);
        if ($ts) {
            $start_date = date('Y-m-d', $ts);
        } else {
            $start_date = date('Y-m-d');
        }
    }

    // Normalize time to 24h H:i (accepts inputs like 07:55 AM)
    $start_time_sanitized = sanitize_text_field($start_time_raw);
    if (preg_match('/^\d{2}:\d{2}$/', (string)$start_time_sanitized)) {
        $start_time = $start_time_sanitized;
    } else {
        $tts = strtotime($start_time_sanitized);
        $start_time = $tts ? date('H:i', $tts) : '09:00';
    }

    // Debugging: Log the sanitized values
    // error_log("Cron Active: {$cron_active}");
    // error_log("Start Date: {$start_date}");
    // error_log("Start Time: {$start_time}");
    // error_log("Schedule: {$schedule}");

    update_option('my_auto_blog_cron_active', $cron_active);
    update_option('my_auto_blog_start_date', $start_date);
    update_option('my_auto_blog_start_time', $start_time);
    update_option('my_auto_blog_schedule', $schedule);

    // Debugging: Log the updated options
    // error_log('Updated options in the database.');

    // Activate or deactivate the cron job based on the 'cron_active' setting
    if ($cron_active) {
        my_auto_blog_activate_cron();
        // Debugging: Log cron activation
        // error_log('Cron job is active.');
    } else {
        my_auto_blog_deactivate_cron();
        // Debugging: Log cron deactivation
        // error_log('Cron job deactivated.');
    }

    // Redirect with a success message for the scheduler section
    $redirect_url = add_query_arg([
        'page' => 'ai-blogger',
        'tab' => 'scheduler',
        'settings-updated' => 'true',
        'message' => 'scheduler'
    ], admin_url('admin.php'));
    wp_redirect($redirect_url);
    exit;
}
add_action('admin_post_save_scheduler_settings', 'save_scheduler_settings');

/**
 * Activate cron job for automatic blog generation
 */
function my_auto_blog_activate_cron() {
    // error_log('[SICA Scheduler] Activating cron job');
    
    // Clear any existing cron event
    my_auto_blog_deactivate_cron();

    // Retrieve the scheduled date and time from the options
    $scheduled_date = get_option('my_auto_blog_start_date');
    $scheduled_time = get_option('my_auto_blog_start_time');
    $schedule_option = get_option('my_auto_blog_schedule');

    // error_log('[SICA Scheduler] Settings - Date: ' . $scheduled_date . ', Time: ' . $scheduled_time . ', Schedule: ' . $schedule_option);

    // Get the timezone set in WordPress settings
    $wp_timezone = get_option('timezone_string');
    if (!$wp_timezone) {
        $wp_timezone = 'UTC';
        // error_log('[SICA Scheduler] No timezone set, using UTC');
    }

    // Combine the date and time to create the timestamp
    $timezone = new DateTimeZone($wp_timezone);
    $datetime = DateTime::createFromFormat('Y-m-d H:i', $scheduled_date . ' ' . $scheduled_time, $timezone);
    $timestamp = false;
    
    if ($datetime) {
        $timestamp = $datetime->getTimestamp();
        // error_log('[SICA Scheduler] Calculated timestamp: ' . date('Y-m-d H:i:s', $timestamp));
    } else {
        // error_log('[SICA Scheduler] Failed to create DateTime object from: ' . $scheduled_date . ' ' . $scheduled_time);
        // If datetime creation fails, set timestamp to next occurrence from now
        $timestamp = strtotime('tomorrow 9:00:00'); // Default to tomorrow at 9 AM
        // error_log('[SICA Scheduler] Using fallback timestamp: ' . date('Y-m-d H:i:s', $timestamp));
    }

    // Determine the correct interval
    $cron_schedule = 'weekly'; // Default
    switch ($schedule_option) {
        case 'daily':
            $cron_schedule = 'daily';
            break;
        case 'every_three_days':
            $cron_schedule = 'every_three_days';
            break;
        case 'weekly':
            $cron_schedule = 'weekly';
            break;
        case 'monthly':
            $cron_schedule = 'monthly';
            break;
    }
    
    // error_log('[SICA Scheduler] Using schedule: ' . $cron_schedule);

    // If timestamp is in the past, set it to the next occurrence
    if ($timestamp <= time()) {
        // error_log('[SICA Scheduler] Timestamp is in the past, calculating next occurrence');
        // Calculate next occurrence based on schedule
        switch ($cron_schedule) {
            case 'daily':
                $timestamp = strtotime('tomorrow ' . $scheduled_time);
                break;
            case 'every_three_days':
                $timestamp = strtotime('+3 days ' . $scheduled_time);
                break;
            case 'weekly':
                $timestamp = strtotime('next week ' . $scheduled_time);
                break;
            case 'monthly':
                $timestamp = strtotime('first day of next month ' . $scheduled_time);
                break;
        }
        // error_log('[SICA Scheduler] New timestamp: ' . date('Y-m-d H:i:s', $timestamp));
    }

    // Schedule the event
    $scheduled = wp_schedule_event($timestamp, $cron_schedule, 'my_auto_blog_cron_event');
    
    if ($scheduled === false) {
        // error_log('[SICA Scheduler] Failed to schedule cron event');
    } else {
        // error_log('[SICA Scheduler] Successfully scheduled cron event for ' . date('Y-m-d H:i:s', $timestamp) . ' with schedule ' . $cron_schedule);
        
        // Verify the schedule was created
        $next_scheduled = wp_next_scheduled('my_auto_blog_cron_event');
        if ($next_scheduled) {
            // error_log('[SICA Scheduler] Verified next run: ' . date('Y-m-d H:i:s', $next_scheduled));
        } else {
            // error_log('[SICA Scheduler] Warning: Could not verify next scheduled run');
        }
    }
    
    // Log all scheduled cron jobs for debugging
    $cron_array = _get_cron_array();
    // error_log('[SICA Scheduler] Current cron jobs: ' . print_r($cron_array, true));
}

/**
 * Deactivate cron job
 */
function my_auto_blog_deactivate_cron() {
    // Find out when the next scheduled event is
    $timestamp = wp_next_scheduled('my_auto_blog_cron_event');

    // Unschedules the event
    if ($timestamp) {
        wp_unschedule_event($timestamp, 'my_auto_blog_cron_event');
    }
}

/**
 * Add custom cron intervals
 */
function my_auto_blog_add_cron_interval($schedules) {
    // Add daily schedule if not already set
    if (!isset($schedules['daily'])) {
        $schedules['daily'] = array(
            'interval' => DAY_IN_SECONDS,
            'display'  => esc_html__('Once Daily', 'sica-content-ai'),
        );
    }
    // Add every 3 days schedule if not already set
    if (!isset($schedules['every_three_days'])) {
        $schedules['every_three_days'] = array(
            'interval' => 3 * DAY_IN_SECONDS,
            'display'  => esc_html__('Every 3 Days', 'sica-content-ai'),
        );
    }
    // Add weekly schedule if not already set
    if (!isset($schedules['weekly'])) {
        $schedules['weekly'] = array(
            'interval' => WEEK_IN_SECONDS,
            'display'  => esc_html__('Once Weekly', 'sica-content-ai'),
        );
    }
    // Add monthly schedule if not already set
    if (!isset($schedules['monthly'])) {
        $schedules['monthly'] = array(
            'interval' => 30 * DAY_IN_SECONDS,
            'display'  => esc_html__('Once Monthly', 'sica-content-ai'),
        );
    }
    return $schedules;
}
add_filter('cron_schedules', 'my_auto_blog_add_cron_interval');

/**
 * The main cron job function that generates blog posts
 */
function my_auto_blog_cron_job() {
    $start_time = current_time('Y-m-d H:i:s');
    // error_log('[SICA Scheduler] ===== CRON JOB TRIGGERED ===== ' . $start_time);
    
    // Initialize log entry with proper timestamp format
    $log_entry = array(
        'timestamp' => $start_time,
        'status' => 'started',
        'message' => 'Scheduler cron job triggered automatically',
        'post_title' => '',
        'post_url' => '',
        'error' => ''
    );
    
    // IMMEDIATELY save a log entry to test if logging works
    // error_log('[SICA Scheduler] About to save initial log entry...');
    sica_add_scheduler_log_entry($log_entry);
    // error_log('[SICA Scheduler] Initial log entry saved.');
    
    // Verify scheduler is active
    if (!get_option('my_auto_blog_cron_active')) {
        // error_log('[SICA Scheduler] Scheduler is not active. Exiting.');
        $log_entry['status'] = 'skipped';
        $log_entry['message'] = 'Scheduler is not active';
        sica_add_scheduler_log_entry($log_entry);
        return;
    }

    // Set extended timeouts for cron job execution - 15 minutes for slow hosts
    @set_time_limit(900);
    @ini_set('max_execution_time', 900); // 15 minutes for cron jobs
    @ini_set('memory_limit', '1024M'); // More memory for cron processing

    // Set longer HTTP request timeouts
    add_filter('http_request_timeout', function($timeout) {
        return 600; // 10 minutes for HTTP requests in cron
    });
    
    // Ensure media functions are available for cron context
    if (!function_exists('media_sideload_image')) {
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');
    }
    
    // Log next scheduled run
    $next_scheduled = wp_next_scheduled('my_auto_blog_cron_event');
    // error_log('[SICA Scheduler] Next scheduled run: ' . ($next_scheduled ? date('Y-m-d H:i:s', $next_scheduled) : 'Not scheduled'));
    
    try {
        $titles = get_option('my_auto_blog_title');
        if (!empty($titles)) {
            $titles_array = explode("\n", $titles);
            $titles_array = array_filter(array_map('trim', $titles_array)); // Clean up empty lines
            
            if (empty($titles_array)) {
                // error_log('[SICA Scheduler] No valid titles in queue after filtering');
                $log_entry['status'] = 'failed';
                $log_entry['message'] = 'No valid titles in queue';
                $log_entry['error'] = 'Title queue is empty or contains only blank lines';
                sica_add_scheduler_log_entry($log_entry);
                return;
            }
            
            $title = $titles_array[0]; // Get first title but don't remove yet
            $log_entry['post_title'] = $title;
            // error_log('[SICA Scheduler] Attempting to create post with title: ' . $title);
            
            // Check for duplicate posts with same title
            $existing_post = get_page_by_title($title, OBJECT, 'post');
            if ($existing_post) {
                // error_log('[SICA Scheduler] Duplicate post detected with title: ' . $title);
                // Remove this title and try the next one
                array_shift($titles_array);
                update_option('my_auto_blog_title', implode("\n", $titles_array));
                
                $log_entry['status'] = 'skipped';
                $log_entry['message'] = 'Skipped duplicate title: ' . $title;
                $log_entry['error'] = 'Post with this title already exists';
                sica_add_scheduler_log_entry($log_entry);
                
                // Try next title if available
                if (!empty($titles_array)) {
                    sica_log('[SICA Scheduler] Attempting next title after duplicate skip', 'info');
                    // Recursively call the function to try the next title
                    my_auto_blog_cron_job();
                }
                return;
            }
            
            $generation_result = generate_and_create_blog_post($title, true);

            // Check if result is an array (new format with details) or boolean (legacy format)
            $success = false;
            $post_id = 0;
            $post_url = '';
            $error_details = '';

            if (is_array($generation_result)) {
                $success = !empty($generation_result['success']);
                if ($success) {
                    $post_id = isset($generation_result['post_id']) ? $generation_result['post_id'] : 0;
                    $post_url = isset($generation_result['post_url']) ? $generation_result['post_url'] : '';
                } else {
                    $error_details = isset($generation_result['error']) ? $generation_result['error'] : 'Unknown error';
                    if (isset($generation_result['exception'])) {
                        $error_details .= ' | Exception: ' . $generation_result['exception'];
                    }
                }
            } else {
                // Legacy boolean format
                $success = (bool)$generation_result;
            }

            if ($success) {
                sica_log('[SICA Scheduler] Successfully created blog post', 'info', array(
                    'title' => $title,
                    'post_id' => $post_id
                ));

                // Only remove title from queue after successful generation
                array_shift($titles_array);
                update_option('my_auto_blog_title', implode("\n", $titles_array));
                sica_log('[SICA Scheduler] Removed title from queue after successful generation', 'info');

                // Get post details if not already in result
                if (empty($post_url) && !empty($post_id)) {
                    $post_url = get_permalink($post_id);
                }

                // If we still don't have post details, get the most recent post
                if (empty($post_id)) {
                    $recent_posts = get_posts(array('post_type' => 'post', 'posts_per_page' => 1));
                    if (!empty($recent_posts)) {
                        $post_id = $recent_posts[0]->ID;
                        $post_url = get_permalink($post_id);
                    }
                }

                $log_entry['post_url'] = $post_url;

                // Auto-generate more titles if queue is running low
                auto_generate_titles_if_needed();

                // Send notification email with complete details
                sica_log('[SICA Scheduler] Sending post creation notification via scheduled automation', 'info');
                send_notification_email('post_created', array(
                    'post_title' => $title,
                    'post_url' => $post_url,
                    'post_id' => $post_id,
                    'mode' => 'auto' // Indicates this is from scheduled automation
                ));

                $log_entry['status'] = 'success';
                $log_entry['message'] = 'Post created successfully: ' . $title;
                sica_add_scheduler_log_entry($log_entry);
            } else {
                // Log detailed error information
                sica_log('[SICA Scheduler] Failed to create blog post', 'error', array(
                    'title' => $title,
                    'error_details' => $error_details,
                    'result_type' => is_array($generation_result) ? 'array' : gettype($generation_result)
                ));

                // Don't remove title from queue on failure - allow retry next time
                $log_entry['status'] = 'failed';
                $log_entry['message'] = 'Failed to create blog post';
                // Use detailed error if available, otherwise use generic message
                $log_entry['error'] = !empty($error_details) ? $error_details : 'Blog post generation function returned false or null';
                $log_entry['error_details'] = json_encode($generation_result);
                sica_add_scheduler_log_entry($log_entry);
            }
        } else {
            // Try to generate initial titles if none exist
            sica_log('[SICA Scheduler] No titles available. Attempting to generate new titles.', 'warning');
            $new_titles = generate_initial_title_set();
            if (!empty($new_titles)) {
                update_option('my_auto_blog_title', implode("\n", $new_titles));
                sica_log('[SICA Scheduler] Generated initial title set with ' . count($new_titles) . ' titles', 'info');
                $log_entry['status'] = 'titles_generated';
                $log_entry['message'] = 'Generated ' . count($new_titles) . ' new titles';
                sica_add_scheduler_log_entry($log_entry);
            } else {
                sica_log('[SICA Scheduler] Failed to generate new titles. Sending notification.', 'error');
                send_notification_email('titles_empty');
                $log_entry['status'] = 'failed';
                $log_entry['message'] = 'No titles available and failed to generate new ones';
                $log_entry['error'] = 'Title generation failed';
                sica_add_scheduler_log_entry($log_entry);
            }
        }
    } catch (Exception $e) {
        sica_log_exception($e, '[SICA Scheduler] Error in cron job');
        $log_entry['status'] = 'error';
        $log_entry['message'] = 'Exception occurred during execution';
        $log_entry['error'] = $e->getMessage();
        $log_entry['trace'] = substr($e->getTraceAsString(), 0, 500);
        sica_add_scheduler_log_entry($log_entry);
    }


    sica_log('[SICA Scheduler] Completed cron job execution', 'info', array(
        'timestamp' => date('Y-m-d H:i:s')
    ));
}
add_action('my_auto_blog_cron_event', 'my_auto_blog_cron_job');

/**
 * Initialize cron settings
 */
function my_auto_blog_cron_settings_init() {
    register_setting('my_auto_blog_cron_settings', 'my_auto_blog_cron_active');
    register_setting('my_auto_blog_cron_settings', 'my_auto_blog_start_date');
    register_setting('my_auto_blog_cron_settings', 'my_auto_blog_start_time');
    register_setting('my_auto_blog_cron_settings', 'my_auto_blog_schedule');
}
add_action('admin_init', 'my_auto_blog_cron_settings_init');

/**
 * Minimal self-heal and migration for legacy schedule values.
 * - Normalizes 12-hour times (e.g., 07:55 AM -> 07:55)
 * - Ensures stored date/time are valid and advances to next occurrence if past
 * - Re-schedules if scheduler is active but no event is queued
 */
function sica_maybe_migrate_and_selfheal_cron() {
    $cron_active = (int) get_option('my_auto_blog_cron_active', 0);
    if (!$cron_active) {
        return;
    }

    $start_date = get_option('my_auto_blog_start_date');
    $start_time = get_option('my_auto_blog_start_time');
    $schedule_option = get_option('my_auto_blog_schedule', 'weekly');

    // Normalize time format to 24h H:i if needed (handles inputs like 07:55 AM)
    if (!preg_match('/^\d{2}:\d{2}$/', (string) $start_time)) {
        $ts = $start_time ? strtotime($start_time) : false;
        if ($ts) {
            $start_time = date('H:i', $ts);
            update_option('my_auto_blog_start_time', $start_time);
        }
    }

    // Resolve timezone
    $wp_tz = get_option('timezone_string');
    if (!$wp_tz) {
        $offset = get_option('gmt_offset');
        if ($offset !== false && $offset !== '' && $offset !== null) {
            $seconds = (float) $offset * 3600;
            $name = @timezone_name_from_abbr('', (int) $seconds, 0);
            if ($name) { $wp_tz = $name; }
        }
        if (!$wp_tz) { $wp_tz = 'UTC'; }
    }
    $tz = new DateTimeZone($wp_tz);

    // Default/validate date/time
    $now = new DateTime('now', $tz);
    if (empty($start_date) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', (string) $start_date)) {
        $start_date = $now->format('Y-m-d');
        update_option('my_auto_blog_start_date', $start_date);
    }
    if (empty($start_time) || !preg_match('/^\d{2}:\d{2}$/', (string) $start_time)) {
        $start_time = $now->format('H:i');
        update_option('my_auto_blog_start_time', $start_time);
    }

    $scheduled_datetime = DateTime::createFromFormat('Y-m-d H:i', $start_date . ' ' . $start_time, $tz);
    if ($scheduled_datetime && $scheduled_datetime <= $now) {
        $interval_map = [
            'daily' => '+1 day',
            'every_three_days' => '+3 days',
            'weekly' => '+1 week',
            'monthly' => '+1 month'
        ];
        $interval = $interval_map[$schedule_option] ?? '+1 week';
        $scheduled_datetime->modify($interval);
        update_option('my_auto_blog_start_date', $scheduled_datetime->format('Y-m-d'));
        update_option('my_auto_blog_start_time', $scheduled_datetime->format('H:i'));
    }

    // If active but not scheduled, try to schedule now that values are valid
    if (!wp_next_scheduled('my_auto_blog_cron_event')) {
        my_auto_blog_activate_cron();
    }
}
add_action('admin_init', 'sica_maybe_migrate_and_selfheal_cron');

/**
 * Add entry to scheduler activity log
 */
function sica_add_scheduler_log_entry($entry) {
    $log = get_option('sica_scheduler_log', array());

    // Ensure entry has required fields
    $entry['timestamp'] = isset($entry['timestamp']) ? $entry['timestamp'] : current_time('Y-m-d H:i:s');
    $entry['status'] = isset($entry['status']) ? $entry['status'] : 'unknown';
    $entry['message'] = isset($entry['message']) ? $entry['message'] : 'No message';
    $entry['source'] = isset($entry['source']) ? $entry['source'] : 'scheduled'; // Track if manual or scheduled

    // Add new entry to the beginning
    array_unshift($log, $entry);

    // Keep only the last 100 entries (increased for manual + scheduled)
    $log = array_slice($log, 0, 100);

    $saved = update_option('sica_scheduler_log', $log);

    // Also log to file for debugging
    if (function_exists('sica_log')) {
        $log_level = ($entry['status'] === 'success') ? 'info' : 'error';
        sica_log(
            sprintf('[Activity Log] %s - %s', $entry['status'], $entry['message']),
            $log_level,
            $entry
        );
    }
}

/**
 * Get scheduler activity log
 */
function sica_get_scheduler_log() {
    $log = get_option('sica_scheduler_log', array());
    // error_log('[SICA Scheduler Log] Retrieved from database: ' . count($log) . ' entries');
    
    // Also check if the option exists at all
    $option_exists = get_option('sica_scheduler_log', 'NOT_FOUND');
    if ($option_exists === 'NOT_FOUND') {
        // error_log('[SICA Scheduler Log] Option does not exist in database!');
    }
    
    return $log;
}

/**
 * Clear scheduler activity log
 */
function sica_clear_scheduler_log() {
    delete_option('sica_scheduler_log');
}

/**
 * Display scheduler activity log
 */
function sica_display_scheduler_log() {
    $log = sica_get_scheduler_log();
    
    // Debug logging
    // error_log('[SICA Scheduler Log Display] Retrieved ' . count($log) . ' log entries');
    if (!empty($log)) {
        // error_log('[SICA Scheduler Log Display] Latest entry: ' . json_encode($log[0]));
    }
    
    echo '<div class="sica-card">';
    echo '<div class="sica-card-header">';
    echo '<h3 class="sica-card-title">Activity Log <span style="font-weight: normal; font-size: 13px; color: #6b7280;">(Manual & Scheduled Posts)</span></h3>';
    
    // Only show debug buttons in debug mode
    if (defined('WP_DEBUG') && WP_DEBUG): 
        echo '<div class="sica-card-header-right">';
        echo '<button type="button" class="button button-small" onclick="location.reload()" style="margin-right: 8px;">Refresh</button>';
        echo '<button type="button" class="button button-small" onclick="sicaTestLogEntry()" style="margin-right: 8px;">Test Log</button>';
        echo '<button type="button" class="button button-small" onclick="sicaTestCron()" style="margin-right: 8px;">Test Cron</button>';
        echo '<button type="button" class="button button-small" onclick="sicaCleanupDuplicates()" style="margin-right: 8px;">Cleanup Duplicates</button>';
        echo '<button type="button" class="button button-small" onclick="if(confirm(\'Clear all log entries?\')) { window.location.href=\'' . esc_url(wp_nonce_url(admin_url('admin-post.php?action=sica_clear_scheduler_log'), 'sica_clear_log')) . '\'; }">Clear Log</button>';
        echo '</div>';
    endif;
    
    echo '</div>';
    
    echo '<div class="sica-card-body">';
    
    if (empty($log)) {
        echo '<div style="text-align: center; padding: 40px 20px; color: #6b7280;">';
        echo '<div style="font-size: 48px; margin-bottom: 16px;">📅</div>';
        echo '<div style="font-weight: 600; margin-bottom: 8px;">No activity yet</div>';
        echo '<div style="font-size: 13px;">Scheduler activity will appear here once the automated posting starts.</div>';
        
        // Add debugging information in development mode
        if (defined('WP_DEBUG') && WP_DEBUG) {
            echo '<div style="margin-top: 20px; padding: 12px; background: #fff3cd; border-radius: 4px; font-size: 11px; text-align: left;">';
            echo '<strong>Debug Info:</strong><br>';
            echo 'Log entries retrieved: ' . count($log) . '<br>';
            
            // Check if option exists in database
            global $wpdb;
            $option_value = $wpdb->get_var($wpdb->prepare("SELECT option_value FROM {$wpdb->options} WHERE option_name = %s", 'sica_scheduler_log'));
            
            if ($option_value) {
                $decoded_log = maybe_unserialize($option_value);
                echo 'Raw database entries: ' . (is_array($decoded_log) ? count($decoded_log) : 'Invalid data') . '<br>';
                echo 'Raw data (first 200 chars): ' . esc_html(substr($option_value, 0, 200)) . '<br>';
            } else {
                echo 'Database option "sica_scheduler_log" does not exist<br>';
            }
            
            echo 'Current time: ' . esc_html(current_time('Y-m-d H:i:s')) . '<br>';
            echo 'WordPress timezone: ' . esc_html(wp_timezone_string()) . '<br>';
            echo '</div>';
        }
        
        echo '</div>';
        echo '</div>';
        echo '</div>';
        return;
    }
    
    echo '<div style="overflow-x: auto;">';
    echo '<table style="width: 100%; border-collapse: collapse; font-size: 13px;">';
    echo '<thead>';
    echo '<tr style="border-bottom: 2px solid #e5e7eb;">';
    echo '<th style="padding: 12px 8px; text-align: left; font-weight: 600; color: #374151; border-right: 1px solid #f3f4f6;">Date/Time</th>';
    echo '<th style="padding: 12px 8px; text-align: left; font-weight: 600; color: #374151; border-right: 1px solid #f3f4f6; width: 80px;">Source</th>';
    echo '<th style="padding: 12px 8px; text-align: left; font-weight: 600; color: #374151; border-right: 1px solid #f3f4f6;">Status</th>';
    echo '<th style="padding: 12px 8px; text-align: left; font-weight: 600; color: #374151; border-right: 1px solid #f3f4f6;">Message</th>';
    echo '<th style="padding: 12px 8px; text-align: left; font-weight: 600; color: #374151;">Details</th>';
    echo '</tr>';
    echo '</thead>';
    echo '<tbody>';
    
    $display_count = 0;
    $default_limit = 25;
    foreach ($log as $entry) {
        if ($display_count >= $default_limit) { break; }
        $status_color = '#666';
        switch ($entry['status']) {
            case 'success': $status_color = '#0a7c0a'; break;
            case 'failed': 
            case 'error': $status_color = '#d63638'; break;
            case 'skipped': $status_color = '#996600'; break;
            case 'titles_generated': $status_color = '#135e96'; break;
        }
        
        // Get source for display
        $source = isset($entry['source']) ? $entry['source'] : 'scheduled';
        $source_icon = $source === 'manual' ? '👤' : '🤖';
        $source_label = ucfirst($source);

        echo '<tr style="border-bottom: 1px solid #f3f4f6;">';
        echo '<td style="padding: 12px 8px; border-right: 1px solid #f3f4f6; font-size: 12px; color: #6b7280;">' . esc_html($entry['timestamp']) . '</td>';
        echo '<td style="padding: 12px 8px; border-right: 1px solid #f3f4f6; font-size: 12px; color: #6b7280;"><span title="' . esc_attr($source_label) . '">' . $source_icon . ' ' . esc_html($source_label) . '</span></td>';
        echo '<td style="padding: 12px 8px; border-right: 1px solid #f3f4f6;"><span style="color: ' . esc_attr($status_color) . '; font-weight: 600; font-size: 12px;">' . esc_html(ucfirst($entry['status'])) . '</span></td>';
        echo '<td style="padding: 12px 8px; border-right: 1px solid #f3f4f6; color: #374151;">' . esc_html($entry['message']) . '</td>';
        echo '<td style="padding: 12px 8px; font-size: 12px;">';
        
        if (!empty($entry['post_title'])) {
            echo '<strong>Title:</strong> ' . esc_html($entry['post_title']) . '<br>';
        }
        if (!empty($entry['post_url'])) {
            echo '<strong>URL:</strong> <a href="' . esc_url($entry['post_url']) . '" target="_blank">View Post</a><br>';
        }
        if (!empty($entry['error'])) {
            echo '<strong>Error:</strong> <span style="color: #d63638;">' . esc_html($entry['error']) . '</span><br>';
        }
        
        // Add Generate Now button for failed entries with titles
        if (in_array($entry['status'], array('failed', 'error')) && !empty($entry['post_title'])) {
            $nonce = wp_create_nonce('sica_generate_now_nonce');
            echo '<button type="button" class="button button-small" onclick="generateNowFromScheduler(\'' . esc_js($entry['post_title']) . '\', \'' . esc_js($nonce) . '\')" style="margin-top: 4px; font-size: 11px;">Generate Now</button>';
        }
        
        echo '</td>';
        echo '</tr>';
        $display_count++;
    }
    
    echo '</tbody>';
    echo '</table>';
    echo '</div>'; // Close overflow-x container
    echo '</div>'; // Close sica-card-body
    echo '</div>'; // Close sica-card
    
    // Hide developer buttons unless WP_DEBUG is true
    ?>
    <script>
    (function(){
      var wpDebug = <?php echo (defined('WP_DEBUG') && WP_DEBUG) ? 'true' : 'false'; ?>;
      if (!wpDebug){
        // Hide all buttons in header-right
        const headerRight = document.querySelector('.sica-card-header-right');
        if (headerRight){
          headerRight.style.display = 'none';
        }
      }
    })();
    function generateNowFromScheduler(title, nonce) {
        if (!title) return;
        
        const button = event.target;
        const originalText = button.textContent;
        button.disabled = true;
        button.textContent = 'Generating...';
        
        const formData = new FormData();
        formData.append('action', 'sica_generate_now_from_scheduler');
        formData.append('title', title);
        formData.append('_ajax_nonce', nonce);
        
        fetch('<?php echo esc_url(admin_url('admin-ajax.php')); ?>', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                button.textContent = 'Success!';
                button.style.background = '#00a32a';
                button.style.color = '#fff';
                
                // Reload page after 2 seconds to show updated log
                setTimeout(() => {
                    location.reload();
                }, 2000);
            } else {
                button.textContent = 'Failed';
                button.style.background = '#d63638';
                button.style.color = '#fff';
                alert('Error: ' + (data.data || 'Failed to generate post'));
                
                // Reset button after 3 seconds
                setTimeout(() => {
                    button.disabled = false;
                    button.textContent = originalText;
                    button.style.background = '';
                    button.style.color = '';
                }, 3000);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            button.disabled = false;
            button.textContent = originalText;
            alert('Network error occurred');
        });
    }
    
    function sicaCleanupDuplicates() {
        if (!confirm('This will remove duplicate posts with the same title, keeping only the most recent. Continue?')) {
            return;
        }
        
        const button = event.target;
        const originalText = button.textContent;
        button.disabled = true;
        button.textContent = 'Cleaning...';
        
        const formData = new FormData();
        formData.append('action', 'sica_cleanup_duplicate_posts');
        formData.append('_ajax_nonce', '<?php echo esc_attr(wp_create_nonce('sica_cleanup_nonce')); ?>');
        
        fetch('<?php echo esc_url(admin_url('admin-ajax.php')); ?>', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                button.textContent = 'Cleaned!';
                button.style.background = '#00a32a';
                button.style.color = '#fff';
                alert('Cleanup completed: ' + data.data);
                
                // Reload page after 2 seconds
                setTimeout(() => {
                    location.reload();
                }, 2000);
            } else {
                button.textContent = 'Failed';
                button.style.background = '#d63638';
                button.style.color = '#fff';
                alert('Error: ' + (data.data || 'Cleanup failed'));
                
                // Reset button after 3 seconds
                setTimeout(() => {
                    button.disabled = false;
                    button.textContent = originalText;
                    button.style.background = '';
                    button.style.color = '';
                }, 3000);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            button.disabled = false;
            button.textContent = originalText;
            alert('Network error occurred');
        });
    }
    
    function sicaTestLogEntry() {
        const formData = new FormData();
        formData.append("action", "sica_test_log_entry");
        formData.append("_ajax_nonce", "<?php echo esc_attr(wp_create_nonce('sica_test_log_nonce')); ?>");
        
        fetch("<?php echo esc_url(admin_url('admin-ajax.php')); ?>", {
            method: "POST",
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert("Test log entry added successfully!");
                location.reload();
            } else {
                alert("Failed to add test log entry: " + (data.data || "Unknown error"));
            }
        })
        .catch(error => {
            console.error("Error:", error);
            alert("Network error occurred");
        });
    }
    
    function sicaTestCron() {
        if (!confirm("This will manually trigger the scheduler cron job. Continue?")) return;
        
        const formData = new FormData();
        formData.append("action", "sica_test_cron");
        formData.append("_ajax_nonce", "<?php echo esc_attr(wp_create_nonce('sica_test_cron_nonce')); ?>");
        
        fetch("<?php echo esc_url(admin_url('admin-ajax.php')); ?>", {
            method: "POST",
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert("Cron job triggered successfully! Check the activity log and error logs.");
                location.reload();
            } else {
                alert("Failed to trigger cron job: " + (data.data || "Unknown error"));
            }
        })
        .catch(error => {
            console.error("Error:", error);
            alert("Network error occurred");
        });
    }
    </script>
    <?php
}

/**
 * Handle clearing scheduler log
 */
function sica_handle_clear_scheduler_log() {
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized');
    }
    
    check_admin_referer('sica_clear_log');
    sica_clear_scheduler_log();
    
    wp_redirect(add_query_arg(array(
        'page' => 'ai-blogger',
        'tab' => 'scheduler',
        'log_cleared' => '1'
    ), admin_url('admin.php')));
    exit;
}
add_action('admin_post_sica_clear_scheduler_log', 'sica_handle_clear_scheduler_log');

/**
 * AJAX handler for Generate Now from scheduler
 */
function sica_ajax_generate_now_from_scheduler() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('sica_generate_now_nonce', '_ajax_nonce');
    
    $title = isset($_POST['title']) ? trim(sanitize_text_field($_POST['title'])) : '';
    if (empty($title)) {
        wp_send_json_error('No title provided');
        return;
    }
    
    // Check subscription status
    if (function_exists('sica_check_subscription_status')) {
        $subscription_data = sica_check_subscription_status();
        if (!$subscription_data || !$subscription_data['ok']) {
            wp_send_json_error('Subscription verification failed');
            return;
        }
        
        // Check if posts remaining (skip for Agency tier only)
        $posts_used = $subscription_data['posts_used'] ?? 0;
        $posts_limit = $subscription_data['posts_limit'] ?? 2;
        $tier = $subscription_data['tier'] ?? 'lite';
        $is_unlimited = ($posts_limit === 'unlimited' || $tier === 'agency');

        if (!$is_unlimited && $posts_used >= $posts_limit) {
            wp_send_json_error('Monthly post limit reached. Upgrade to continue publishing.');
            return;
        }
    }
    
    try {
        // error_log('[SICA Scheduler] Manual generation requested for title: ' . $title);
        
        // Generate the blog post using the existing function
        if (function_exists('generate_and_create_blog_post')) {
            $result = generate_and_create_blog_post($title, true);
            
            if ($result) {
                // error_log('[SICA Scheduler] Manual generation successful for: ' . $title);
                
                // Log the manual generation
                $log_entry = array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'status' => 'success',
                    'message' => 'Manual generation successful: ' . $title,
                    'post_title' => $title,
                    'post_url' => '',
                    'error' => ''
                );
                
                // Get the URL of the created post
                $recent_posts = get_posts(array('post_type' => 'post', 'posts_per_page' => 1));
                if (!empty($recent_posts)) {
                    $post_url = get_permalink($recent_posts[0]->ID);
                    $log_entry['post_url'] = $post_url;
                }
                
                sica_add_scheduler_log_entry($log_entry);
                
                wp_send_json_success('Post generated successfully');
            } else {
                // error_log('[SICA Scheduler] Manual generation failed for: ' . $title);
                
                // Log the failed manual generation
                $log_entry = array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'status' => 'failed',
                    'message' => 'Manual generation failed: ' . $title,
                    'post_title' => $title,
                    'post_url' => '',
                    'error' => 'Post generation function returned false'
                );
                
                sica_add_scheduler_log_entry($log_entry);
                
                wp_send_json_error('Failed to generate post');
            }
        } else {
            wp_send_json_error('Post generation function not available');
        }
    } catch (Exception $e) {
        // error_log('[SICA Scheduler] Exception during manual generation: ' . $e->getMessage());
        
        // Log the exception
        $log_entry = array(
            'timestamp' => time(),
            'date' => date('Y-m-d H:i:s'),
            'status' => 'error',
            'message' => 'Manual generation error: ' . $title,
            'post_title' => $title,
            'post_url' => '',
            'error' => $e->getMessage()
        );
        
        sica_add_scheduler_log_entry($log_entry);
        
        wp_send_json_error('Error: ' . $e->getMessage());
    }
}
add_action('wp_ajax_sica_generate_now_from_scheduler', 'sica_ajax_generate_now_from_scheduler');

/**
 * AJAX handler for testing log entry functionality
 */
function sica_ajax_test_log_entry() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('sica_test_log_nonce', '_ajax_nonce');
    
    // Add a test log entry
    $test_entry = array(
        'timestamp' => current_time('Y-m-d H:i:s'),
        'status' => 'success',
        'message' => 'Test log entry created manually via Test Log button',
        'post_title' => 'Manual Test Entry',
        'post_url' => '#test',
        'details' => 'This is a test to verify the logging system works'
    );
    
    sica_add_scheduler_log_entry($test_entry);
    
    wp_send_json_success('Test log entry added successfully');
}
add_action('wp_ajax_sica_test_log_entry', 'sica_ajax_test_log_entry');

/**
 * AJAX handler for testing cron functionality
 */
function sica_ajax_test_cron() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('sica_test_cron_nonce', '_ajax_nonce');
    
    // error_log('[SICA Scheduler] Manual cron test triggered by user');
    
    // Manually trigger the cron job
    my_auto_blog_cron_job();
    
    wp_send_json_success('Cron job triggered manually - check logs and activity log');
}
add_action('wp_ajax_sica_test_cron', 'sica_ajax_test_cron');

/**
 * Clean up duplicate posts (utility function)
 */
function sica_cleanup_duplicate_posts($title) {
    $posts = get_posts(array(
        'post_type' => 'post',
        'title' => $title,
        'post_status' => array('publish', 'draft'),
        'numberposts' => -1
    ));
    
    if (count($posts) > 1) {
        // error_log('[SICA Cleanup] Found ' . count($posts) . ' duplicate posts with title: ' . $title);
        
        // Keep the most recent post, delete the others
        usort($posts, function($a, $b) {
            return strtotime($b->post_date) - strtotime($a->post_date);
        });
        
        $keep_post = array_shift($posts); // Keep the first (most recent)
        
        foreach ($posts as $duplicate_post) {
            wp_delete_post($duplicate_post->ID, true); // Force delete
            // error_log('[SICA Cleanup] Deleted duplicate post ID: ' . $duplicate_post->ID);
        }
        
        // error_log('[SICA Cleanup] Kept most recent post ID: ' . $keep_post->ID);
        return $keep_post->ID;
    }
    
    return false;
}

/**
 * AJAX handler for cleanup duplicate posts
 */
function sica_ajax_cleanup_duplicate_posts() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('sica_cleanup_nonce', '_ajax_nonce');
    
    // Get all posts and find duplicates by title
    $all_posts = get_posts(array(
        'post_type' => 'post',
        'post_status' => array('publish', 'draft'),
        'numberposts' => -1
    ));
    
    $title_groups = array();
    foreach ($all_posts as $post) {
        $title = trim($post->post_title);
        if (!isset($title_groups[$title])) {
            $title_groups[$title] = array();
        }
        $title_groups[$title][] = $post;
    }
    
    $duplicates_found = 0;
    $posts_deleted = 0;
    
    foreach ($title_groups as $title => $posts) {
        if (count($posts) > 1) {
            $duplicates_found++;
            
            // Sort by date descending (newest first)
            usort($posts, function($a, $b) {
                return strtotime($b->post_date) - strtotime($a->post_date);
            });
            
            // Keep the first (newest), delete the rest
            $keep_post = array_shift($posts);
            
            foreach ($posts as $duplicate_post) {
                wp_delete_post($duplicate_post->ID, true);
                $posts_deleted++;
                // error_log('[SICA Cleanup] Deleted duplicate post ID: ' . $duplicate_post->ID . ' with title: ' . $title);
            }
            
            // error_log('[SICA Cleanup] Kept newest post ID: ' . $keep_post->ID . ' with title: ' . $title);
        }
    }
    
    if ($duplicates_found > 0) {
        wp_send_json_success("Found {$duplicates_found} duplicate titles, deleted {$posts_deleted} posts");
    } else {
        wp_send_json_success("No duplicate posts found");
    }
}
add_action('wp_ajax_sica_cleanup_duplicate_posts', 'sica_ajax_cleanup_duplicate_posts');

/**
 * AJAX handler for testing email notifications
 */
function sica_ajax_test_email_notifications() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('sica_test_email_nonce', '_ajax_nonce');
    
    // Test email notification with sample data using improved system
    $test_details = array(
        'post_title' => 'Test Blog Post - Email System Check',
        'post_url' => home_url('/test-post-url'),
        'post_id' => 0,
        'mode' => 'test' // This will trigger the proper test email subject and content
    );
    
    // error_log('[SICA Email Test] Starting email notification test');
    
    $result = send_notification_email('post_created', $test_details);
    
    // Always run diagnosis to help with troubleshooting
    // error_log('[SICA Email Test] Running comprehensive email diagnosis...');
    sica_diagnose_email_issues();
    
    if ($result) {
        wp_send_json_success('WordPress reports email sent successfully, but this does not guarantee delivery. Check your inbox and error logs for detailed diagnosis and recommendations.');
    } else {
        wp_send_json_error('Test email failed to send. A comprehensive diagnosis has been logged - check your error logs for detailed troubleshooting information.');
    }
}
add_action('wp_ajax_sica_test_email_notifications', 'sica_ajax_test_email_notifications');