<?php
/**
 * SEO and Schema Management
 * 
 * Handles SEO meta tags, schema markup, and search optimization features
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Output AI Blogger schema markup in head
 */
function output_ai_blogger_schema() {
    if (!is_singular('post')) {
        return;
    }
    
    global $post;
    $schema_data = get_post_meta($post->ID, '_ai_blogger_schema', true);
    
    if (!empty($schema_data)) {
        echo '<script type="application/ld+json">' . "\n";
        echo json_encode($schema_data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . "\n";
        echo '</script>' . "\n";
    }
}
add_action('wp_head', 'output_ai_blogger_schema');
/**
 * Output AI Blogger meta description in head
 */
function output_ai_blogger_meta_description() {
    if (!is_singular('post')) {
        return;
    }
    
    global $post;
    $meta_description = get_post_meta($post->ID, '_ai_blogger_meta_description', true);
    
    if (!empty($meta_description)) {
        echo '<meta name="description" content="' . esc_attr($meta_description) . '">' . "\n";
    }
}
add_action('wp_head', 'output_ai_blogger_meta_description');
/**
 * Generate schema markup for articles
 */
function generate_article_schema($title, $content, $business_name, $post_id) {
    // Get business details
    $business_category = get_option('my_auto_blog_business_category', '');
    $service_area = get_option('my_auto_blog_service_area', '');
    $products_services = get_option('my_auto_blog_products_services', '');
    
    // Extract FAQs from content for FAQ schema
    $faqs = extract_faqs_from_content($content);
    
    // Extract citations from content (injected as data-citations attr on references section)
    $citations = array();
    if (preg_match('/data-citations=\"([^\"]+)\"/', $content, $m) || preg_match('/class=\"ai-citations\"[^>]*data-citations=\"([^\"]+)\"/', $content, $m)) {
        $raw = html_entity_decode($m[1]);
        $parts = array_filter(array_map('trim', explode('|', $raw)));
        // Basic URL validate and dedupe
        $seen = array();
        foreach ($parts as $u) {
            if (filter_var($u, FILTER_VALIDATE_URL) && !isset($seen[$u])) {
                $citations[] = $u;
                $seen[$u] = true;
            }
        }
    }

    // Base schema
    $schema = array(
        '@context' => 'https://schema.org',
        '@type' => 'Article',
        'headline' => $title,
        'description' => generate_meta_description($content),
        'author' => array(
            '@type' => 'Organization',
            'name' => $business_name,
            '@id' => home_url() . '#organization'
        ),
        'datePublished' => gmdate('c'),
        'dateModified' => gmdate('c'),
        'publisher' => array(
            '@type' => 'Organization',
            '@id' => home_url() . '#organization',
            'name' => $business_name,
            'address' => array(
                '@type' => 'PostalAddress',
                'addressLocality' => $service_area,
                'addressRegion' => 'US'
            )
        ),
        'isPartOf' => array(
            '@type' => 'WebSite',
            '@id' => home_url() . '#website',
            'url' => home_url(),
            'name' => $business_name,
            'description' => get_bloginfo('description')
        ),
        'mainEntity' => array(
            '@type' => 'WebPage',
            '@id' => get_permalink($post_id) . '#webpage',
            'url' => get_permalink($post_id),
            'name' => $title,
            'isPartOf' => array(
                '@id' => home_url() . '#website'
            ),
            'about' => array(
                '@type' => 'Thing',
                'name' => $business_category
            ),
            'mentions' => array(
                '@type' => 'Organization',
                'name' => $business_name,
                'areaServed' => $service_area
            )
        )
    );

    // Add citations to schema if present
    if (!empty($citations)) {
        $schema['citation'] = array_values($citations);
    }
    
    // Add FAQ schema if FAQs were found
    if (!empty($faqs)) {
        $faq_entities = array();
        foreach ($faqs as $faq) {
            $faq_entities[] = array(
                '@type' => 'Question',
                'name' => $faq['question'],
                'acceptedAnswer' => array(
                    '@type' => 'Answer',
                    'text' => $faq['answer']
                )
            );
        }
        
        $schema['mainEntity']['mainEntity'] = array(
            '@type' => 'FAQPage',
            'mainEntity' => $faq_entities
        );
    }
    
    return $schema;
}

/**
 * Extract FAQs from content for schema markup
 */
function extract_faqs_from_content($content) {
    $faqs = array();
    
    // Remove HTML tags for analysis
    $text_content = wp_strip_all_tags($content);
    
    // Pattern 1: Q: Question A: Answer
    if (preg_match_all('/Q:\s*([^\n]+)\s*A:\s*([^\n]+)/i', $text_content, $matches, PREG_SET_ORDER)) {
        foreach ($matches as $match) {
            if (count($match) >= 3) {
                $faqs[] = array(
                    'question' => trim($match[1]),
                    'answer' => trim($match[2])
                );
            }
        }
    }
    
    // Pattern 2: Question? followed by paragraph
    if (preg_match_all('/([^.!?]+\?)\s*([^.!?]+[.!?])/s', $text_content, $matches, PREG_SET_ORDER)) {
        foreach ($matches as $match) {
            if (count($match) >= 3 && strlen($match[1]) > 10 && strlen($match[2]) > 20) {
                $question = trim($match[1]);
                $answer = trim($match[2]);
                
                // Basic quality checks
                if (str_word_count($question) >= 3 && str_word_count($answer) >= 5) {
                    $faqs[] = array(
                        'question' => $question,
                        'answer' => $answer
                    );
                }
            }
        }
    }
    
    // Limit to 10 FAQs maximum for schema
    return array_slice($faqs, 0, 10);
}

/**
 * Generate meta description from content
 */
function generate_meta_description($content) {
    // Strip HTML and get plain text
    $text = wp_strip_all_tags($content);
    
    // Remove extra whitespace
    $text = preg_replace('/\s+/', ' ', $text);
    $text = trim($text);
    
    // Get first sentence or 155 characters
    $sentences = preg_split('/[.!?]+/', $text);
    $description = '';
    
    foreach ($sentences as $sentence) {
        $sentence = trim($sentence);
        if (empty($sentence)) continue;
        
        if (strlen($description . $sentence) <= 155) {
            $description .= $sentence . '. ';
        } else {
            break;
        }
    }
    
    // If description is too short, use character limit
    if (strlen($description) < 120) {
        $description = substr($text, 0, 155);
        // Try to end at a word boundary
        $last_space = strrpos($description, ' ');
        if ($last_space !== false && $last_space > 100) {
            $description = substr($description, 0, $last_space);
        }
        $description .= '...';
    }
    
    return trim($description);
}

/**
 * Get business-adaptive category keywords based on business type
 */
function get_business_adaptive_categories($business_category) {
    $business_lower = strtolower($business_category);
    
    // Universal categories that work for all businesses
    $universal_categories = array(
        'Tips & Guides' => ['tips', 'guide', 'how to', 'step by step', 'tutorial', 'advice', 'help', 'considerations', 'essential', 'important', 'key', 'must know', 'best practices'],
        'Industry News' => ['news', 'update', 'announcement', 'latest', 'trending', 'industry', 'new', 'recent', 'current', 'development'],
        'Cost & Pricing' => ['cost', 'price', 'pricing', 'budget', 'affordable', 'expensive', 'estimate', 'value', 'investment'],
        'Business Insights' => ['business', 'company', 'professional', 'commercial', 'enterprise', 'strategy', 'growth', 'success'],
        'Regulatory & Compliance' => ['bylaw', 'regulation', 'code', 'permit', 'legal', 'compliance', 'rules', 'standards', 'requirements']
    );
    
    // Business-specific categories based on detected business type
    $specific_categories = array();
    
    // Construction, Contracting, Building
    if (preg_match('/\b(construction|contractor|builder|deck|roofing|plumbing|electrical|hvac|renovation)\b/i', $business_lower)) {
        $specific_categories = array(
            'Installation & Setup' => ['install', 'installation', 'setup', 'assembly', 'mounting', 'construction', 'building', 'project'],
            'Maintenance & Repair' => ['maintenance', 'repair', 'fix', 'service', 'upkeep', 'care', 'replacement', 'restore', 'inspection'],
            'Safety & Standards' => ['safety', 'secure', 'protection', 'safe', 'risk', 'danger', 'hazard', 'standards', 'quality'],
            'Seasonal Services' => ['winter', 'summer', 'spring', 'fall', 'weather', 'season', 'climate', 'temperature', 'storm', 'preparation']
        );
    }
    
    // Healthcare, Medical, Dental
    elseif (preg_match('/\b(medical|health|dental|doctor|clinic|therapy|wellness|fitness)\b/i', $business_lower)) {
        $specific_categories = array(
            'Health & Wellness' => ['health', 'wellness', 'fitness', 'nutrition', 'exercise', 'lifestyle', 'prevention'],
            'Patient Care' => ['patient', 'care', 'treatment', 'therapy', 'consultation', 'diagnosis', 'recovery'],
            'Medical Information' => ['medical', 'symptoms', 'condition', 'disease', 'procedure', 'medication', 'specialist'],
            'Preventive Care' => ['prevention', 'screening', 'checkup', 'early', 'detection', 'vaccine', 'immunization']
        );
    }
    
    // Legal, Law Firms
    elseif (preg_match('/\b(legal|law|lawyer|attorney|court|litigation|divorce|estate)\b/i', $business_lower)) {
        $specific_categories = array(
            'Legal Advice' => ['legal', 'law', 'rights', 'advice', 'counsel', 'representation', 'advocate'],
            'Case Studies' => ['case', 'court', 'verdict', 'settlement', 'litigation', 'trial', 'judgment'],
            'Legal Process' => ['process', 'procedure', 'filing', 'documentation', 'paperwork', 'application'],
            'Law Changes' => ['amendment', 'change', 'new law', 'legislation', 'statute', 'regulation']
        );
    }
    
    // Real Estate, Property
    elseif (preg_match('/\b(real estate|property|realtor|mortgage|home|house|apartment)\b/i', $business_lower)) {
        $specific_categories = array(
            'Market Trends' => ['market', 'trends', 'prices', 'value', 'appreciation', 'forecast', 'analysis'],
            'Buying & Selling' => ['buying', 'selling', 'purchase', 'sale', 'offer', 'negotiation', 'closing'],
            'Property Management' => ['property', 'management', 'rental', 'tenant', 'landlord', 'maintenance'],
            'Investment Advice' => ['investment', 'roi', 'return', 'profit', 'portfolio', 'equity', 'appreciation']
        );
    }
    
    // Technology, IT, Software
    elseif (preg_match('/\b(technology|software|app|digital|web|it|computer|tech)\b/i', $business_lower)) {
        $specific_categories = array(
            'Technology Solutions' => ['technology', 'tech', 'digital', 'software', 'app', 'online', 'platform', 'system'],
            'Implementation' => ['implementation', 'setup', 'deployment', 'integration', 'installation', 'configuration'],
            'Security & Support' => ['security', 'backup', 'support', 'maintenance', 'troubleshooting', 'protection'],
            'Innovation' => ['innovation', 'new features', 'updates', 'advancement', 'breakthrough', 'cutting edge']
        );
    }
    
    // Food, Restaurant, Catering
    elseif (preg_match('/\b(restaurant|food|catering|chef|cooking|cuisine|dining)\b/i', $business_lower)) {
        $specific_categories = array(
            'Menu & Recipes' => ['menu', 'recipe', 'dish', 'ingredient', 'cooking', 'preparation', 'cuisine'],
            'Nutrition & Health' => ['nutrition', 'healthy', 'diet', 'calories', 'organic', 'fresh', 'wholesome'],
            'Food Safety' => ['safety', 'hygiene', 'sanitation', 'storage', 'handling', 'temperature', 'quality'],
            'Dining Experience' => ['dining', 'experience', 'service', 'atmosphere', 'event', 'special', 'celebration']
        );
    }
    
    // Retail, E-commerce, Shopping
    elseif (preg_match('/\b(retail|shop|store|ecommerce|sales|product|merchandise)\b/i', $business_lower)) {
        $specific_categories = array(
            'Product Information' => ['product', 'item', 'merchandise', 'features', 'specifications', 'quality', 'brand'],
            'Shopping Guides' => ['shopping', 'guide', 'comparison', 'review', 'recommendation', 'selection'],
            'Customer Service' => ['customer', 'service', 'support', 'satisfaction', 'experience', 'return', 'warranty'],
            'Trends & Fashion' => ['trends', 'fashion', 'style', 'collection', 'seasonal', 'popular', 'trending']
        );
    }
    
    // Financial, Accounting, Insurance
    elseif (preg_match('/\b(financial|accounting|insurance|tax|investment|banking)\b/i', $business_lower)) {
        $specific_categories = array(
            'Financial Planning' => ['financial', 'planning', 'investment', 'retirement', 'savings', 'portfolio', 'wealth'],
            'Tax & Accounting' => ['tax', 'accounting', 'bookkeeping', 'deduction', 'filing', 'records', 'audit'],
            'Insurance Coverage' => ['insurance', 'coverage', 'policy', 'claim', 'protection', 'premium', 'deductible'],
            'Market Analysis' => ['market', 'analysis', 'trends', 'forecast', 'performance', 'economy', 'rates']
        );
    }
    
    // Default/Generic business categories for unmatched types
    else {
        $specific_categories = array(
            'Services' => ['service', 'offering', 'solution', 'consultation', 'delivery', 'support'],
            'Process & Procedures' => ['process', 'procedure', 'method', 'approach', 'system', 'workflow'],
            'Quality & Standards' => ['quality', 'standard', 'excellence', 'professional', 'certified', 'expertise'],
            'Customer Focus' => ['customer', 'client', 'satisfaction', 'experience', 'relationship', 'trust']
        );
    }
    
    // Merge universal and specific categories
    return array_merge($universal_categories, $specific_categories);
}

/**
 * Ensure a category exists and return its term ID without relying on wp_create_category
 * which may not be loaded in cron contexts. Uses core taxonomy APIs.
 */
function sica_safe_ensure_category($name) {
    $name = trim($name);
    if ($name === '') {
        return (int) get_option('default_category');
    }
    // Check by slug first
    $slug = sanitize_title($name);
    $by_slug = get_category_by_slug($slug);
    if ($by_slug && isset($by_slug->term_id)) {
        return (int) $by_slug->term_id;
    }
    // Check generic term existence
    $exists = term_exists($name, 'category');
    if ($exists) {
        return is_array($exists) ? (int) $exists['term_id'] : (int) $exists;
    }
    // Insert term
    $inserted = wp_insert_term($name, 'category', array('slug' => $slug));
    if (!is_wp_error($inserted) && isset($inserted['term_id'])) {
        return (int) $inserted['term_id'];
    }
    // Fallback to default category
    return (int) get_option('default_category');
}

/**
 * Assign intelligent category based on content
 */
function assign_intelligent_category($title, $content, $business_category) {
    // Convert content to lowercase for analysis
    $content_lower = strtolower($content . ' ' . $title);
    
    // Get business-adaptive categories
    $category_keywords = get_business_adaptive_categories($business_category);
    
    // Score each category
    $category_scores = array();
    foreach ($category_keywords as $category => $keywords) {
        $score = 0;
        foreach ($keywords as $keyword) {
            $score += substr_count($content_lower, $keyword);
        }
        $category_scores[$category] = $score;
    }
    
    // Find the category with highest score
    $max_score = max($category_scores);
    $best_category = array_search($max_score, $category_scores);
    
    
    // Create category if it doesn't exist and has a reasonable score
    if ($best_category && $max_score > 0) {
        $category = get_category_by_slug(sanitize_title($best_category));
        if (!$category) {
            $category_id = sica_safe_ensure_category($best_category);
            return $category_id;
        } else {
            return $category->term_id;
        }
    }
    
    // Fallback 1: Use business category as category name
    if (!empty($business_category)) {
        $fallback_category = ucfirst(trim($business_category));
        $category = get_category_by_slug(sanitize_title($fallback_category));
        if (!$category) {
            $category_id = sica_safe_ensure_category($fallback_category);
            return $category_id;
        } else {
            return $category->term_id;
        }
    }
    
    // Fallback 2: Default to 'Tips & Guides' if no other category works
    $default_category = 'Tips & Guides';
    $category = get_category_by_slug(sanitize_title($default_category));
    if (!$category) {
        $category_id = sica_safe_ensure_category($default_category);
        if (defined('WP_DEBUG') && WP_DEBUG) { 
            error_log('Created default category: ' . $default_category);
        }
        return $category_id;
    } else {
        if (defined('WP_DEBUG') && WP_DEBUG) { 
            error_log('Using default category: ' . $default_category);
        }
        return $category->term_id;
    }
    
    // This should never happen now
    return false;
}

/**
 * Add internal links to content
 */
function add_internal_links($content, $current_post_id) {
    if (empty($content) || !is_string($content)) {
        if (defined('WP_DEBUG') && WP_DEBUG) { 
            error_log('Invalid content passed to add_internal_links');
        }
        return $content;
    }

    // Get potential target posts (published only, skip self)
    $target_posts = get_potential_target_posts($current_post_id);
    
    if (empty($target_posts)) {
        if (defined('WP_DEBUG') && WP_DEBUG) { 
            error_log('No target posts found for internal linking');
        }
        return $content;
    }
    
    $linked_content = $content;
    $links_added = 0;
    $max_links = 3; // Limit internal links
    
    foreach ($target_posts as $target_post) {
        if ($links_added >= $max_links) {
            break;
        }
        
        $post_title = get_the_title($target_post->ID);
        $post_url = get_permalink($target_post->ID);
        
        // Try to find relevant text to link
        $keywords = explode(' ', strtolower($post_title));
        $keywords = array_filter($keywords, function($word) {
            return strlen($word) > 3; // Only use meaningful words
        });
        
        foreach ($keywords as $keyword) {
            if ($links_added >= $max_links) {
                break;
            }
            
            // Find the keyword in content (case insensitive)
            $pattern = '/\b(' . preg_quote($keyword, '/') . ')\b/i';
            if (preg_match($pattern, $linked_content)) {
                $replacement = '<a href="' . esc_url($post_url) . '">$1</a>';
                $linked_content = preg_replace($pattern, $replacement, $linked_content, 1);
                $links_added++;
                break; // Only one link per post
            }
        }
    }
    
    if (defined('WP_DEBUG') && WP_DEBUG) { 
    }
    return $linked_content;
}

/**
 * Get potential target posts for internal linking
 */
function get_potential_target_posts($current_post_id) {
    // Get posts in the same categories first
    $post_categories = array();
    if (is_numeric($current_post_id)) {
        $post_categories = wp_get_post_categories($current_post_id);
    }

    $related_posts = array();
    if (!empty($post_categories)) {
        $related_posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'posts_per_page' => 15,
            'post__not_in' => array($current_post_id),
            'category__in' => $post_categories,
            'orderby' => 'date',
            'order' => 'DESC'
        ));
    }

    // If we don't have enough related posts, get recent ones
    if (count($related_posts) < 5) {
        $recent_posts = get_posts(array(
            'post_type' => array('post', 'page'),
            'posts_per_page' => (15 - count($related_posts)),
            'post__not_in' => array_merge(array($current_post_id), wp_list_pluck($related_posts, 'ID')),
            'orderby' => 'date',
            'order' => 'DESC'
        ));
        $related_posts = array_merge($related_posts, $recent_posts);
    }

    return array_slice($related_posts, 0, 10);
}