<?php
/**
 * Sica Content AI - Subscription Functions
 * Replaces old license-based functions with new SaaS subscription system
 * 
 * These functions should replace the existing license functions in sica-content-ai.php
 */

// =============================================================================
// SUBSCRIPTION STATUS CHECKING (replaces sica_check_license_status)
// =============================================================================

function sica_check_subscription_status() {
    // Request-level memory cache to avoid multiple calls per page load
    static $memory_cache = null;
    if ($memory_cache !== null) {
        return $memory_cache;
    }
    
    // Check if email is verified first
    $customer_email = get_option('sica_customer_email', '');
    $email_verified = get_option('sica_email_verified', false);
    
    // If no email or not verified, return unverified status
    if (!$customer_email || !$email_verified) {
        $memory_cache = array(
            'ok' => false,
            'error' => 'Email not verified',
            'email_verified' => false,
            'requires_verification' => true
        );
        return $memory_cache;
    }
    
    // Check for month rollover and clear cache if needed
    sica_check_month_rollover();
    
    // CRITICAL FIX: Check transient cache before server call
    $cached_status = get_transient('sica_subscription_status');
    if ($cached_status !== false) {
        if (defined('WP_DEBUG') && WP_DEBUG) { 
            // error_log('Using cached subscription status: ' . json_encode($cached_status)); 
        }
        $memory_cache = $cached_status;
        return $cached_status;
    }
    
    // Cache miss - call server (only when needed)
    $server_status = sica_check_subscription_status_server();
    if ($server_status && $server_status['ok']) {
        if (defined('WP_DEBUG') && WP_DEBUG) { 
            // error_log('Using fresh server subscription status: ' . json_encode($server_status)); 
        }
        $memory_cache = $server_status;
        return $server_status;
    }
    
    // Server unavailable - deny access for security (both license and email-only users)
    if (defined('WP_DEBUG') && WP_DEBUG) { 
        // error_log('Server unavailable - denying access for security'); 
    }
    $fallback_response = array(
        'ok' => false,
        'error' => 'Server validation required',
        'tier' => 'lite',
        'posts_limit' => 0,
        'posts_used' => 0,
        'posts_remaining' => 0,
        'email_verified' => true,
        'requires_verification' => false
    );
    $memory_cache = $fallback_response;
    return $fallback_response;
}

// Check subscription status from server (used by both regular and license users)
function sica_check_subscription_status_server() {
    $customer_email = get_option('sica_customer_email', '');
    $site_url = get_site_url();

    // CRITICAL FIX: License users need license tokens, not email tokens
    $api_token = get_option('sica_api_token', '');
    if (empty($api_token) && !empty($customer_email)) {
        $license_key = get_option('sica_license_key', '');
        
        if (!empty($license_key)) {
            // License user - get license token from activate.php
            $reissue = wp_remote_post('https://api.sica.ai/activate.php', array(
                'headers' => array('Content-Type' => 'application/json'),
                'body' => json_encode(array(
                    'license_key' => $license_key,
                    'site_url' => $site_url
                )),
                'timeout' => 15
            ));
        } else {
            // Email-only user - get email token from verification.php
            $reissue = wp_remote_post('https://api.sica.ai/verification.php', array(
                'headers' => array('Content-Type' => 'application/json'),
                'body' => json_encode(array(
                    'action' => 'reissue_token',
                    'email' => $customer_email,
                    'site_url' => $site_url
                )),
                'timeout' => 15
            ));
        }
        
        if (!is_wp_error($reissue)) {
            $d = json_decode(wp_remote_retrieve_body($reissue), true);
            if ($d && !empty($d['ok']) && !empty($d['api_token'])) {
                update_option('sica_api_token', sanitize_text_field($d['api_token']), false); // autoload=false for security
                $api_token = $d['api_token'];
                // error_log('SUBSCRIPTION: ' . (!empty($license_key) ? 'License' : 'Email') . ' token reissued successfully');
            }
        }
    }

    $response = wp_remote_post('https://api.sica.ai/subscription.php', array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'X-Sica-Token' => get_option('sica_api_token', '')
        ),
        'body' => json_encode(array(
            'action' => 'check_usage',
            'site_url' => $site_url,
            'customer_email' => $customer_email
        )),
        'timeout' => 15
    ));
    
    if (is_wp_error($response)) {
        // error_log('Sica Content AI: Failed to get subscription status from server - ' . $response->get_error_message());
        return null;
    }
    
    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    $data = json_decode($response_body, true);
    
    // Enhanced debugging for development
    if (defined('WP_DEBUG') && WP_DEBUG) {
        // error_log('SUBSCRIPTION_STATUS_DEBUG: Response code=' . $response_code . ', Body=' . substr($response_body, 0, 500) . (strlen($response_body) > 500 ? '...' : ''));
        // error_log('SUBSCRIPTION_STATUS_DEBUG: Parsed data=' . json_encode($data));
        // error_log('SUBSCRIPTION_STATUS_DEBUG: API token present=' . (!empty(get_option('sica_api_token', '')) ? 'yes' : 'no'));
    }
    
    if (!$data || !$data['ok']) {
        // error_log('Sica Content AI: Invalid server response for subscription status - Code:' . $response_code . ' Data:' . json_encode($data));
        return null;
    }
    
    // Add email verification status to response
    $data['email_verified'] = true;
    $data['requires_verification'] = false;
    
    // Cache subscription status locally for performance (2 hours for UI decisions)
    set_transient('sica_subscription_status', $data, 2 * HOUR_IN_SECONDS); // 2 hours cache
        
    // error_log('Server status check successful: ' . json_encode($data));
    return $data;
}

// =============================================================================
// CACHE MANAGEMENT HELPERS
// =============================================================================

/**
 * Check for month rollover and clear cache if needed
 */
function sica_check_month_rollover() {
    $current_month = date('Y-m');
    $cached_month = get_option('sica_cached_month', '');
    
    if ($cached_month !== $current_month) {
        // Month changed - clear all caches
        delete_transient('sica_subscription_status');
        update_option('sica_cached_month', $current_month);
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // error_log("CACHE: Month rollover detected ($cached_month -> $current_month), cleared subscription cache");
        }
    }
}

/**
 * Force refresh subscription cache (for manual refresh buttons)
 */
function sica_force_refresh_subscription_cache() {
    delete_transient('sica_subscription_status');
    if (defined('WP_DEBUG') && WP_DEBUG) {
        // error_log('CACHE: Manually cleared subscription cache');
    }
}

/**
 * Clear token cache and force re-authentication
 */
function sica_force_token_refresh() {
    delete_option('sica_api_token');
    delete_transient('sica_subscription_status');
    if (defined('WP_DEBUG') && WP_DEBUG) {
        // error_log('CACHE: Cleared API token and subscription cache - will reissue on next request');
    }
}

// =============================================================================
// TOKEN MANAGEMENT HELPERS
// =============================================================================

/**
 * Ensure API token is present locally; attempt reissue if missing
 */
function sica_ensure_api_token($force_refresh = false) {
    $token = get_option('sica_api_token', '');
    $email = get_option('sica_customer_email', '');

    sica_log('sica_ensure_api_token called', 'info', array(
        'token_exists' => !empty($token) ? 'yes (length=' . strlen($token) . ')' : 'NO',
        'email_exists' => !empty($email) ? 'yes' : 'NO',
        'force_refresh' => $force_refresh ? 'yes' : 'no'
    ));

    // Return existing token unless force refresh is requested
    if (!empty($token) && !$force_refresh) {
        return $token;
    }

    if (empty($email)) {
        sica_log('Cannot reissue token - no customer email found', 'error');
        return '';
    }

    $site_url = get_site_url();
    $license_key = get_option('sica_license_key', '');

    sica_log('Attempting token reissue', 'info', array(
        'email' => substr($email, 0, 5) . '***',
        'has_license' => !empty($license_key) ? 'yes' : 'no',
        'site_url' => $site_url
    ));

    if (!empty($license_key)) {
        // License user - get license token
        $response = wp_remote_post('https://api.sica.ai/activate.php', array(
            'headers' => array('Content-Type' => 'application/json'),
            'body' => json_encode(array(
                'license_key' => $license_key,
                'site_url' => $site_url
            )),
            'timeout' => 15
        ));
    } else {
        // Email-only user - get email token
        $response = wp_remote_post('https://api.sica.ai/verification.php', array(
            'headers' => array('Content-Type' => 'application/json'),
            'body' => json_encode(array(
                'action' => 'reissue_token',
                'email' => $email,
                'site_url' => $site_url
            )),
            'timeout' => 15
        ));
    }

    if (is_wp_error($response)) {
        sica_log('Token reissue request FAILED (wp_error)', 'error', array(
            'error' => $response->get_error_message()
        ));
        return '';
    }

    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    $data = json_decode($response_body, true);

    if ($response_code !== 200) {
        sica_log('Token reissue returned non-200', 'error', array(
            'http_code' => $response_code,
            'body' => substr($response_body, 0, 200)
        ));
        return '';
    }

    if (!$data || !$data['ok']) {
        sica_log('Token reissue server returned error', 'error', array(
            'error' => $data['error'] ?? 'unknown',
            'body' => substr($response_body, 0, 200)
        ));
        return '';
    }

    if (empty($data['api_token'])) {
        sica_log('Token reissue response missing api_token field', 'error', array(
            'data_keys' => is_array($data) ? implode(',', array_keys($data)) : 'not_array'
        ));
        return '';
    }

    $new_token = sanitize_text_field($data['api_token']);
    update_option('sica_api_token', $new_token, false); // autoload=false for security
    sica_log('Token successfully reissued and saved', 'info', array(
        'token_length' => strlen($new_token)
    ));
    return $new_token;
}

// DEPRECATED: Local fallback removed for security
function sica_check_subscription_status_local() {
    // SECURITY: All subscription validation now server-side only
    // error_log('SECURITY WARNING: Local subscription check attempted - this is deprecated for security');
    
    return array(
        'ok' => false,
        'error' => 'Local validation disabled for security - server validation required',
        'tier' => 'lite',
        'posts_limit' => 0,
        'posts_used' => 0,
        'posts_remaining' => 0,
        'try_again_count' => 0,
        'can_generate' => false,
        'overage_available' => false,
        'overage_price' => null,
        'billing_period_end' => date('Y-m-t'),
        'upgrade_required' => true,
        'email_verified' => false,
        'requires_verification' => true
    );
}

// =============================================================================
// GENERATION PERMISSION CHECK (replaces sica_can_generate_post)
// =============================================================================

function sica_can_generate_post($is_try_again = false) {
    
    // Get current subscription status
    $status = sica_check_subscription_status();
    
    if (!$status['ok']) {
        return false;
    }
    
    // Try again is always allowed
    if ($is_try_again) {
        return true;
    }
    
    // Check if generation is allowed
    return $status['can_generate'];
}

// =============================================================================
// POST GENERATION TRACKING (replaces sica_track_post_generation)
// =============================================================================

function sica_track_post_generation($post_title = '', $is_try_again = false, $try_again_parent_id = null) {
    $customer_email = get_option('sica_customer_email', '');
    $site_url = get_site_url();
    $api_token = get_option('sica_api_token', '');

    sica_log('sica_track_post_generation: Starting', 'info', array(
        'post_title' => $post_title,
        'has_token' => !empty($api_token) ? 'yes (length=' . strlen($api_token) . ')' : 'NO TOKEN',
        'has_email' => !empty($customer_email) ? 'yes' : 'no',
        'site_url' => $site_url
    ));

    $start_time = microtime(true);
    $response = wp_remote_post('https://api.sica.ai/subscription.php', array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'X-Sica-Token' => $api_token
        ),
        'body' => json_encode(array(
            'action' => 'generate_post',
            'site_url' => $site_url,
            'customer_email' => $customer_email,
            'post_title' => $post_title,
            'is_try_again' => $is_try_again,
            'try_again_parent_id' => $try_again_parent_id
        )),
        'timeout' => 15
    ));
    $duration = round(microtime(true) - $start_time, 2);
    
    // Clear subscription cache after usage tracking (usage changed)
    delete_transient('sica_subscription_status');

    if (is_wp_error($response)) {
        sica_log('sica_track_post_generation: wp_remote_post FAILED', 'error', array(
            'error' => $response->get_error_message(),
            'duration' => $duration . 's'
        ));
        // Fallback to local tracking for local license or free tier
        $email = get_option('sica_customer_email', '');
        if (empty($email)) {
            sica_track_post_generation_local($is_try_again);
        }
        return null;
    }

    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    $data = json_decode($response_body, true);

    sica_log('sica_track_post_generation: Server responded', 'info', array(
        'http_code' => $response_code,
        'duration' => $duration . 's',
        'ok' => isset($data['ok']) ? ($data['ok'] ? 'true' : 'false') : 'not_set',
        'generation_id' => $data['generation_id'] ?? 'none'
    ));

    if (!$data || !$data['ok']) {
        $error_details = $data['error'] ?? 'unknown error';
        sica_log('sica_track_post_generation: Server returned error', 'error', array(
            'error' => $error_details,
            'body' => substr($response_body, 0, 300)
        ));
        
        // Industry best practice: Always sync local usage with server truth
        if (isset($data['current_usage']) && isset($data['limit'])) {
            // Server provided actual usage - sync immediately
            $current_month = date('Y-m');
            $old_usage = get_option("sica_monthly_usage_$current_month", 0);
            $new_usage = intval($data['current_usage']);
            
            // Safety check: Don't allow usage to go backwards unless it's a new month or explicit reset
            if ($new_usage < $old_usage && $old_usage > 0) {
                // error_log("WARNING: Server wants to reduce usage from {$old_usage} to {$new_usage}. This might be a bug. Keeping higher value.");
                $new_usage = $old_usage; // Prevent usage from going backwards
            }
            
            update_option("sica_monthly_usage_$current_month", $new_usage);
            // error_log("Usage sync: Server says {$data['current_usage']}/{$data['limit']}, updated local tracking from {$old_usage} to {$new_usage}");
            delete_transient('sica_subscription_status');
        } elseif (isset($data['posts_used']) && isset($data['posts_limit'])) {
            // Alternative server response format - sync with posts_used
            $current_month = date('Y-m');
            $old_usage = get_option("sica_monthly_usage_$current_month", 0);
            $new_usage = intval($data['posts_used']);
            
            // Safety check: Don't allow usage to go backwards
            if ($new_usage < $old_usage && $old_usage > 0) {
                // error_log("WARNING: Server wants to reduce usage from {$old_usage} to {$new_usage}. Keeping higher value to prevent reset bug.");
                $new_usage = $old_usage;
            }
            
            update_option("sica_monthly_usage_$current_month", $new_usage);
            // error_log("Usage sync (posts_used): Server says {$data['posts_used']}/{$data['posts_limit']}, updated local tracking from {$old_usage} to {$new_usage}");
            delete_transient('sica_subscription_status');
        } elseif (isset($data['upgrade_required']) && $data['upgrade_required']) {
            // Fallback: Infer usage from upgrade_required (less ideal)
            // error_log('Server indicates usage limit reached, inferring local usage sync...');
            $subscription_data = sica_check_subscription_status();
            if ($subscription_data && isset($subscription_data['posts_limit'])) {
                $current_month = date('Y-m');
                $old_usage = get_option("sica_monthly_usage_$current_month", 0);
                $inferred_usage = $subscription_data['posts_limit'];
                
                // Only update if the inferred usage is higher than current (never go backwards)
                if ($inferred_usage > $old_usage) {
                    update_option("sica_monthly_usage_$current_month", $inferred_usage);
                    // error_log("Inferred local usage sync to match server limit: {$old_usage} -> {$inferred_usage}");
                } else {
                    // error_log("Inferred usage {$inferred_usage} not higher than current {$old_usage}, keeping current value");
                }
                delete_transient('sica_subscription_status');
            }
        }
        
        // Conservative approach: Only block if server explicitly says so
        // Never block on network errors or unclear responses
        if ($error_details === 'upgrade_required' || $error_details === 'usage_limit_exceeded') {
            sica_log('sica_track_post_generation: Blocked by server - ' . $error_details, 'warning');
            return null;
        }

        // For network errors, timeouts, unclear responses:
        // Allow fallback with warning (customer benefit)
        sica_log('sica_track_post_generation: Using fallback_local', 'warning', array(
            'reason' => 'Server unclear/unavailable'
        ));
        $email = get_option('sica_customer_email', '');
        if (empty($email)) {
            sica_track_post_generation_local($is_try_again);
        }
        return 'fallback_local'; // Indicates local fallback was used
    }

    sica_log('sica_track_post_generation: SUCCESS', 'info', array(
        'generation_id' => $data['generation_id']
    ));

    // Clear subscription status cache to force refresh
    delete_transient('sica_subscription_status');

    return $data['generation_id'];
}

// Local fallback tracking (free tier only)
function sica_track_post_generation_local($is_try_again = false) {
    if (!$is_try_again) {
        // Only track regular posts against local limit
        $current_month = date('Y-m');
        $monthly_usage = get_option("sica_monthly_usage_$current_month", 0);
        update_option("sica_monthly_usage_$current_month", $monthly_usage + 1);
        // error_log('Sica Content AI: Local tracking - Posts used this month: ' . ($monthly_usage + 1));
        // Invalidate cached subscription status so UI reflects new usage
        delete_transient('sica_subscription_status');
    } else {
        // Track try again count separately (doesn't count against limit)
        $current_month = date('Y-m');
        $try_again_count = get_option("sica_try_again_count_$current_month", 0);
        update_option("sica_try_again_count_$current_month", $try_again_count + 1);
        // error_log('Sica Content AI: Local tracking - Try again count: ' . ($try_again_count + 1));
    }
}

// =============================================================================
// SUBSCRIPTION SETUP & MANAGEMENT
// =============================================================================

function sica_get_customer_email() {
    $email = get_option('sica_customer_email', '');
    
    if (empty($email)) {
        // Try to get from current user
        $current_user = wp_get_current_user();
        if ($current_user && $current_user->user_email) {
            $email = $current_user->user_email;
            update_option('sica_customer_email', $email);
        }
    }
    
    return $email;
}

function sica_set_customer_email($email) {
    // error_log('SICA: sica_set_customer_email called with: ' . $email);
    $email = sanitize_email($email);
    // error_log('SICA: After sanitization: ' . $email);
    
    if (is_email($email)) {
        // error_log('SICA: Email is valid, updating option');
        $result = update_option('sica_customer_email', $email);
        // error_log('SICA: update_option result: ' . ($result ? 'true' : 'false'));
        
        // Clear cached subscription status
        delete_transient('sica_subscription_status');
        // error_log('SICA: Cleared subscription status cache');
        return true;
    }
    // error_log('SICA: Email is invalid');
    return false;
}

/**
 * AJAX: Downgrade to free - remove license key but preserve email subscription
 */
function sica_ajax_downgrade_to_free() {
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
        return;
    }

    // Clear ONLY license-related data (preserve email registration)
    delete_option('sica_license_status');
    delete_option('sica_license_key');
    delete_option('sica_license_type');
    delete_option('sica_posts_remaining');
    delete_option('sica_monthly_limit');
    delete_option('sica_api_token');
    
    // PRESERVE customer email and verification status
    // User keeps their email-based subscription access (2 posts/month)
    // They don't need to re-verify their email
    
    // Clear local monthly usage (will refresh from server for email-based subscription)
    $current_month = date('Y-m');
    delete_option("sica_monthly_usage_$current_month");

    // Clear cached subscription status to force fresh server check
    delete_transient('sica_subscription_status');
    
    // Disable scheduler when license is removed
    update_option('my_auto_blog_cron_active', 0);
    if (function_exists('my_auto_blog_deactivate_cron')) {
        my_auto_blog_deactivate_cron();
    }

    // Check what subscription level they'll have after downgrade
    $customer_email = get_option('sica_customer_email', '');
    $email_verified = get_option('sica_email_verified', false);
    
    if ($customer_email && $email_verified) {
        wp_send_json_success('License key removed. You now have access to your email-based subscription benefits.');
    } else {
        wp_send_json_success('License key removed. Please verify your email to continue posting.');
    }
}
add_action('wp_ajax_sica_downgrade_to_free', 'sica_ajax_downgrade_to_free');

/**
 * AJAX: Change customer email address
 */
function sica_ajax_change_customer_email() {
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
        return;
    }

    $new_email = isset($_POST['new_email']) ? sanitize_email($_POST['new_email']) : '';
    if (empty($new_email) || !is_email($new_email)) {
        wp_send_json_error('Please enter a valid email address');
        return;
    }

    $current_email = get_option('sica_customer_email', '');
    if ($new_email === $current_email) {
        wp_send_json_error('New email is the same as current email');
        return;
    }

    // Update email and reset verification status
    update_option('sica_customer_email', $new_email);
    update_option('sica_email_verified', false);
    delete_option('sica_verification_code');
    
    // Clear cached subscription status
    delete_transient('sica_subscription_status');

    wp_send_json_success('Email address updated. Please check your new email for verification.');
}
add_action('wp_ajax_change_customer_email', 'sica_ajax_change_customer_email');

/**
 * AJAX: Remove email registration completely
 */
function sica_ajax_remove_email_registration() {
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Unauthorized');
        return;
    }

    // Clear all email-related data
    delete_option('sica_customer_email');
    delete_option('sica_email_verified');
    delete_option('sica_verification_code');
    
    // Clear cached subscription status
    delete_transient('sica_subscription_status');

    wp_send_json_success('Email registration removed successfully');
}
add_action('wp_ajax_remove_email_registration', 'sica_ajax_remove_email_registration');

/**
 * AJAX handler to reactivate license using stored license key (secure - no key exposure to client)
 */
function sica_ajax_reactivate_stored_license() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    
    // Get the stored license key
    $license_key = get_option('sica_license_key', '');
    
    if (empty($license_key)) {
        wp_send_json_error('No license key found. Please enter a license key first.');
        return;
    }
    
    // Call the server to reactivate the license
    $site_url = get_site_url();
    
    $response = wp_remote_post('https://api.sica.ai/activate.php', array(
        'headers' => array(
            'Content-Type' => 'application/json'
        ),
        'body' => json_encode(array(
            'license_key' => $license_key,
            'site_url' => $site_url
        )),
        'timeout' => 30
    ));
    
    if (is_wp_error($response)) {
        wp_send_json_error('Network error: ' . $response->get_error_message());
        return;
    }
    
    $data = json_decode(wp_remote_retrieve_body($response), true);
    
    if (defined('WP_DEBUG') && WP_DEBUG) { 
        // error_log('License reactivation response received for stored key'); 
    }
    
    if (!$data || !$data['ok']) {
        $error_msg = $data['error'] ?? 'Unable to refresh license';
        wp_send_json_error($error_msg);
        return;
    }
    
    // License reactivated successfully - update license data locally
    update_option('sica_license_status', 'active');
    update_option('sica_license_type', 'paid');
    update_option('sica_posts_remaining', $data['credits_limit']);
    update_option('sica_monthly_limit', $data['credits_limit']);

    // Save tier from server (used for feature checks and fallback)
    if (isset($data['tier'])) {
        update_option('sica_subscription_tier', sanitize_text_field($data['tier']));
    }

    // Update subscription system data
    if (isset($data['customer_email'])) {
        update_option('sica_customer_email', $data['customer_email']);
    }
    update_option('sica_email_verified', true);
    
    // Store/update API token if provided
    if (!empty($data['api_token'])) {
        update_option('sica_api_token', sanitize_text_field($data['api_token']), false); // autoload=false for security
        if (defined('WP_DEBUG') && WP_DEBUG) { 
            // error_log('API token updated from license reactivation'); 
        }
    }
    
    // Clear cached subscription status to force fresh server check
    delete_transient('sica_subscription_status');
    $current_month = date('Y-m');
    delete_option("sica_monthly_usage_$current_month");
    
    wp_send_json_success('License refreshed successfully! Your subscription data has been updated.');
}
add_action('wp_ajax_reactivate_stored_license', 'sica_ajax_reactivate_stored_license');

/**
 * AJAX handler to deactivate license for transfer to another site
 */
function sica_ajax_deactivate_license() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    
    // Get the stored license key
    $license_key = get_option('sica_license_key', '');
    
    if (empty($license_key)) {
        wp_send_json_error('No license key found to deactivate');
        return;
    }
    
    // Get confirmation from request
    $confirm = isset($_POST['confirm']) && $_POST['confirm'] === 'true';
    
    // Call the server to deactivate the license
    $site_url = get_site_url();
    
    $response = wp_remote_post('https://api.sica.ai/licence_deactivate.php', array(
        'headers' => array(
            'Content-Type' => 'application/json'
        ),
        'body' => json_encode(array(
            'license_key' => $license_key,
            'site_url' => $site_url,
            'confirm' => $confirm
        )),
        'timeout' => 30
    ));
    
    if (is_wp_error($response)) {
        wp_send_json_error('Network error: ' . $response->get_error_message());
        return;
    }
    
    $data = json_decode(wp_remote_retrieve_body($response), true);
    
    if (defined('WP_DEBUG') && WP_DEBUG) { 
        // error_log('License deactivation response: ' . json_encode($data)); 
    }
    
    if (!$data || !$data['ok']) {
        $error_msg = $data['error'] ?? 'Unable to deactivate license';
        
        // Special handling for confirmation requirement
        if ($error_msg === 'confirmation_required') {
            wp_send_json_error([
                'needs_confirmation' => true,
                'message' => $data['message'],
                'warning' => $data['warning'],
                'current_site' => $data['current_site']
            ]);
            return;
        }
        
        wp_send_json_error($error_msg);
        return;
    }
    
    // License deactivated successfully - clear local data
    delete_option('sica_license_status');
    delete_option('sica_license_key');
    delete_option('sica_license_type');
    delete_option('sica_posts_remaining');
    delete_option('sica_monthly_limit');
    delete_option('sica_api_token');
    
    // Clear cached subscription status
    delete_transient('sica_subscription_status');
    $current_month = date('Y-m');
    delete_option("sica_monthly_usage_$current_month");
    
    // Disable scheduler when license is deactivated
    update_option('my_auto_blog_cron_active', 0);
    if (function_exists('my_auto_blog_deactivate_cron')) {
        my_auto_blog_deactivate_cron();
    }
    
    wp_send_json_success([
        'message' => $data['message'],
        'next_steps' => $data['next_steps'],
        'deactivated_at' => $data['deactivated_at']
    ]);
}
add_action('wp_ajax_deactivate_license', 'sica_ajax_deactivate_license');

// =============================================================================
// USAGE DASHBOARD HELPERS
// =============================================================================

function sica_get_usage_dashboard_data() {
    $status = sica_check_subscription_status();
    
    // Return null if email verification is required (will show verification UI)
    if (!$status['ok'] || !empty($status['requires_verification'])) {
        return null;
    }
    
    $tier_names = [
        'lite' => 'Content AI Lite',
        'starter' => 'Content AI Starter',
        'standard' => 'Content AI Standard',
        'growth' => 'Content AI Growth',
        'pro' => 'Content AI Pro',
        'agency' => 'Content AI Agency'
    ];

    $tier_colors = [
        'lite' => '#6c757d',     // Gray
        'starter' => '#28a745',  // Green
        'standard' => '#28a745', // Green
        'growth' => '#ffc107',   // Yellow
        'pro' => '#dc3545',      // Red (premium tier)
        'agency' => '#7c3aed'    // Purple (agency tier)
    ];
    
    $usage_percentage = 0;
    if ($status['posts_limit'] > 0) {
        $usage_percentage = min(100, ($status['posts_used'] / $status['posts_limit']) * 100);
    }
    
    return [
        'tier' => $status['tier'],
        'tier_name' => $tier_names[$status['tier']] ?? $status['tier'],
        'tier_color' => $tier_colors[$status['tier']] ?? '#6c757d',
        'posts_used' => $status['posts_used'],
        'posts_limit' => $status['posts_limit'],
        'posts_remaining' => $status['posts_remaining'],
        'try_again_count' => $status['try_again_count'] ?? 0,
        'usage_percentage' => $usage_percentage,
        'can_generate' => $status['can_generate'],
        'overage_available' => $status['overage_available'] ?? false,
        'overage_price' => $status['overage_price'] ?? null,
        'billing_period_end' => $status['billing_period_end'] ?? date('Y-m-t'),
        'upgrade_required' => $status['upgrade_required'] ?? false,
        'is_unlimited' => $status['posts_limit'] === -1
    ];
}

// =============================================================================
// TIER-SPECIFIC FEATURE CHECKS
// =============================================================================

function sica_tier_supports_feature($feature) {
    $status = sica_check_subscription_status();
    $customer_email = get_option('sica_customer_email', '');
    $is_email_verified = !empty($customer_email) && get_option('sica_email_verified', false);
    
    // Use robust tier detection with fallback logic
    if ($status && $status['ok'] && $is_email_verified) {
        $tier = $status['tier'] ?? 'lite';
    } else {
        // Fallback to local license data when server is unavailable
        $license_status = get_option('sica_license_status', 'inactive');
        $license_type = get_option('sica_license_type', 'free');
        // Use stored tier from activation, or default to starter for paid licenses
        $stored_tier = get_option('sica_subscription_tier', '');
        if (!empty($stored_tier)) {
            $tier = $stored_tier;
        } else {
            $tier = $license_status === 'active' && $license_type === 'paid' ? 'starter' : 'lite';
        }
    }
    
    $tier_features = [
        'lite' => [
            'watermarked_images' => true,
            'industry_prompts' => false,
            'scheduling' => false,
            'bulk_tools' => false,
            'white_label' => false,
            'api_access' => false
        ],
        'starter' => [  // Alias for standard
            'watermarked_images' => false,
            'industry_prompts' => true,
            'scheduling' => true,
            'bulk_tools' => false,
            'white_label' => false,
            'api_access' => false
        ],
        'standard' => [
            'watermarked_images' => false,
            'industry_prompts' => true,
            'scheduling' => true,
            'bulk_tools' => false,
            'white_label' => false,
            'api_access' => false
        ],
        'growth' => [
            'watermarked_images' => false,
            'industry_prompts' => true,
            'scheduling' => true,
            'bulk_tools' => true,
            'white_label' => false,
            'api_access' => false
        ],
        'pro' => [
            'watermarked_images' => false,
            'industry_prompts' => true,
            'scheduling' => true,
            'bulk_tools' => true,
            'white_label' => true,
            'api_access' => true
        ],
        'agency' => [  // Same as pro
            'watermarked_images' => false,
            'industry_prompts' => true,
            'scheduling' => true,
            'bulk_tools' => true,
            'white_label' => true,
            'api_access' => true
        ]
    ];

    return $tier_features[$tier][$feature] ?? false;
}

// =============================================================================
// CONTENT GENERATION API CALL (replaces proxy_openai.php call)
// =============================================================================

function sica_generate_content_with_subscription($prompt, $title = '', $is_try_again = false, $try_again_parent_id = null, $generation_id = null) {
    // Get subscription info for SaaS proxy
    $customer_email = get_option('sica_customer_email', '');
    $site_url = get_site_url();
    
    // Use SaaS proxy for content generation
    $request_data = array(
        'site_url' => $site_url,
        'customer_email' => $customer_email,
        'content_type' => 'text',
        'model' => 'gpt-4o-search-preview',
        'messages' => array(
            array(
                'role' => 'system',
                'content' => 'You are a professional content writer creating SEO-optimized blog posts with access to current web information.'
            ),
            array(
                'role' => 'user',
                'content' => $prompt
            )
        ),
        'max_tokens' => 4000,
        'title' => $title,
        'is_try_again' => $is_try_again,
        'try_again_parent_id' => $try_again_parent_id,
        'generation_id' => $generation_id
    );
    
    // Ensure we have a valid API token before making the request
    $api_token = get_option('sica_api_token', '');
    if (empty($api_token)) {
        sica_log('No API token found, attempting reissue...', 'warning');
        $api_token = sica_ensure_api_token();
        if (empty($api_token)) {
            sica_log('Token reissue FAILED - no valid token', 'error');
            throw new Exception('No valid API token available. Please check your license key in Account Settings.');
        } else {
            sica_log('Token successfully reissued', 'info');
        }
    }
    
    sica_log('Making API request to generate.php', 'info', array(
        'model' => $request_data['model'] ?? 'unknown',
        'title' => $title,
        'site_url' => $site_url
    ));

    $start_time = microtime(true);
    $response = wp_remote_post('https://api.sica.ai/generate.php', array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'X-Sica-Token' => $api_token
        ),
        'body' => json_encode($request_data),
        'timeout' => 300,
    ));
    $request_duration = round(microtime(true) - $start_time, 2);

    if (is_wp_error($response)) {
        $error_msg = $response->get_error_message();
        sica_log('API request FAILED (wp_error)', 'error', array(
            'error' => $error_msg,
            'duration' => $request_duration . 's'
        ));

        // Log to Activity Log for visibility on live sites
        if (function_exists('sica_add_scheduler_log_entry')) {
            sica_add_scheduler_log_entry(array(
                'status' => 'failed',
                'message' => 'HTTP request failed: ' . substr($error_msg, 0, 100),
                'source' => 'debug',
                'error' => $error_msg
            ));
        }

        throw new Exception('SaaS API request failed: ' . sanitize_text_field($error_msg));
    }

    $response_code = wp_remote_retrieve_response_code($response);
    sica_log('API response received', 'info', array(
        'response_code' => $response_code,
        'duration' => $request_duration . 's'
    ));
    if ($response_code !== 200) {
        $error_body = wp_remote_retrieve_body($response);
        
        // If we get a 401, force reissue the token and retry once
        if ($response_code === 401) {
            sica_log('Got 401, forcing token refresh', 'warning');
            $new_token = sica_ensure_api_token(true);  // Force refresh
            if (!empty($new_token)) {
                // Retry with the new token
                $retry_response = wp_remote_post('https://api.sica.ai/generate.php', array(
                    'headers' => array(
                        'Content-Type' => 'application/json',
                        'X-Sica-Token' => $new_token
                    ),
                    'body' => json_encode($request_data),
                    'timeout' => 300,
                ));
                
                if (!is_wp_error($retry_response) && wp_remote_retrieve_response_code($retry_response) === 200) {
                    $response = $retry_response;
                    $response_code = 200;
                } else {
                    throw new Exception("Authentication failed even after token refresh. Please check your license key in Account Settings.");
                }
            } else {
                throw new Exception("Authentication failed. Please check your license key in Account Settings.");
            }
        } else {
            throw new Exception("SaaS API returned error code: " . sanitize_text_field($response_code) . " - " . sanitize_text_field($error_body));
        }
    }
    
    $response_body = wp_remote_retrieve_body($response);
    $data = json_decode($response_body, true);

    sica_log('API response parsed', 'info', array(
        'ok' => $data['ok'] ?? 'not_set',
        'has_data' => isset($data['data']) ? 'yes' : 'no',
        'body_length' => strlen($response_body)
    ));

    if (!$data || !$data['ok']) {
        $error_msg = $data['error'] ?? 'Unknown SaaS API error';
        sica_log('API returned error (ok=false)', 'error', array(
            'error' => $error_msg,
            'full_response' => substr($response_body, 0, 500),
            'data_keys' => is_array($data) ? implode(',', array_keys($data)) : 'not_array'
        ));
        throw new Exception('SaaS API error: ' . sanitize_text_field($error_msg));
    }

    $openai_data = $data['data'];
    if (!isset($openai_data['choices'][0]['message']['content'])) {
        sica_log('Invalid response structure', 'error', array(
            'data_keys' => array_keys($openai_data ?? [])
        ));
        throw new Exception('Invalid response structure from SaaS API');
    }

    $content = $openai_data['choices'][0]['message']['content'];
    $content_length = strlen($content);

    sica_log('Content generation successful', 'info', array(
        'content_length' => $content_length,
        'tokens_used' => $openai_data['usage']['total_tokens'] ?? 0
    ));

    return [
        'content' => $content,
        'tokens_used' => $openai_data['usage']['total_tokens'] ?? 0,
        'usage_info' => $data['usage'] ?? []
    ];
}

// =============================================================================
// EMAIL VERIFICATION AJAX HANDLERS (Setup Tab)
// =============================================================================

// WordPress AJAX handler for sending verification code
function sica_ajax_send_verification() {
    // Security check
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    if (!check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce', false)) {
        wp_send_json_error('Security check failed');
        return;
    }
    
    $email = sanitize_email($_POST['email']);
    $marketing_consent = isset($_POST['marketing_consent']) ? (bool)$_POST['marketing_consent'] : false;
    
    if (!is_email($email)) {
        wp_send_json_error('Invalid email address');
        return;
    }
    
    // Save email and marketing consent locally
    update_option('sica_customer_email', $email);
    update_option('sica_marketing_consent', $marketing_consent);
    update_option('sica_marketing_consent_date', current_time('mysql'));
    
    // Call real email verification API
    $site_url = get_site_url();
    $api_data = array(
        'action' => 'send_verification',
        'email' => $email,
        'site_url' => $site_url,
        'marketing_consent' => $marketing_consent,
        'consent_source' => 'welcome_signup',
        'client_ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
        'consent_timestamp' => current_time('c')
    );
    
    // error_log('[SICA_VERIFICATION] Sending verification request:');
    // error_log('[SICA_VERIFICATION] Email: ' . $email);
    // error_log('[SICA_VERIFICATION] Site URL: ' . $site_url);
    // error_log('[SICA_VERIFICATION] API Data: ' . json_encode($api_data));
    
    $response = wp_remote_post('https://api.sica.ai/verification.php', array(
        'headers' => array(
            'Content-Type' => 'application/json'
        ),
        'body' => json_encode($api_data),
        'timeout' => 15
    ));
    
    // error_log('[SICA_VERIFICATION] Response received:');
    if (is_wp_error($response)) {
        // error_log('[SICA_VERIFICATION] WP Error: ' . $response->get_error_message());
    } else {
        // error_log('[SICA_VERIFICATION] Response code: ' . wp_remote_retrieve_response_code($response));
        // error_log('[SICA_VERIFICATION] Response body: ' . wp_remote_retrieve_body($response));
    }
    
    if (is_wp_error($response)) {
        wp_send_json_error('Failed to send verification email. Please try again.');
        return;
    }
    
    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    $data = json_decode($response_body, true);
    
    if ($response_code === 200 && $data && $data['ok']) {
        wp_send_json_success(array(
            'message' => 'Verification code sent to your email!',
            'email' => $email
        ));
    } else {
        $error_message = $data['error'] ?? 'Failed to send verification email';
        wp_send_json_error($error_message);
    }
}
add_action('wp_ajax_send_verification', 'sica_ajax_send_verification');

// WordPress AJAX handler for verifying email code
function sica_ajax_verify_email() {
    // Security check
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    if (!check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce', false)) {
        wp_send_json_error('Security check failed');
        return;
    }
    
    $code = sanitize_text_field($_POST['code']);
    $customer_email = get_option('sica_customer_email', '');
    
    if (!$customer_email) {
        wp_send_json_error('Email not found. Please send verification code first.');
        return;
    }
    
    if (!$code || strlen($code) !== 6) {
        wp_send_json_error('Please enter a valid 6-digit code');
        return;
    }
    
    // Call real email verification API
    $site_url = get_site_url();
    $response = wp_remote_post('https://api.sica.ai/verification.php', array(
        'headers' => array(
            'Content-Type' => 'application/json'
        ),
        'body' => json_encode(array(
            'action' => 'verify_email',
            'email' => $customer_email,
            'code' => $code,
            'site_url' => $site_url
        )),
        'timeout' => 15
    ));
    
    if (is_wp_error($response)) {
        wp_send_json_error('Failed to verify email. Please try again.');
        return;
    }
    
    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    $data = json_decode($response_body, true);
    
    if ($response_code === 200 && $data && $data['ok']) {
        // Mark email as verified locally
        update_option('sica_email_verified', true);
        update_option('sica_license_status', 'active');
        // Store API token from server if provided
        if (!empty($data['api_token'])) {
            update_option('sica_api_token', sanitize_text_field($data['api_token']), false); // autoload=false for security
        }
        
        // Clear any cached subscription status
        delete_transient('sica_subscription_status');
        
        // Auto-add verified email to blog notifications
        $current_notifications = get_option('my_auto_blog_notification_emails', get_option('admin_email'));
        $notification_emails = array_filter(array_map('trim', explode(',', $current_notifications)));
        
        // Add the verified email if not already in the list
        if (!in_array($customer_email, $notification_emails)) {
            $notification_emails[] = $customer_email;
            update_option('my_auto_blog_notification_emails', implode(', ', $notification_emails));
        }
        
        wp_send_json_success(array(
            'message' => 'Email verified successfully! You can now generate blog posts.',
            'verified' => true
        ));
    } else {
        $error_message = $data['error'] ?? 'Invalid verification code';
        wp_send_json_error($error_message);
    }
}
add_action('wp_ajax_verify_email', 'sica_ajax_verify_email');

// =============================================================================
// ASYNC GENERATION FUNCTIONS (Server-side processing to bypass WP hosting limits)
// =============================================================================

/**
 * Start an async generation job on the server
 * Returns job_id immediately, server processes in background
 *
 * @param string $content_prompt The prompt for content generation
 * @param string $title Post title
 * @param string $image_prompt The prompt for image generation (optional)
 * @param string $image_model Image model to use (gpt-image-1 or dall-e-3)
 * @param int|null $generation_id Pre-validated generation ID from subscription.php
 * @return array|WP_Error Response with job_id or error
 */
function sica_start_async_generation($content_prompt, $title, $image_prompt = null, $image_model = 'gpt-image-1', $generation_id = null) {
    $customer_email = get_option('sica_customer_email', '');
    $site_url = get_site_url();
    $api_token = get_option('sica_api_token', '');

    // Ensure we have a valid API token
    if (empty($api_token)) {
        sica_log('No API token for async, attempting reissue...', 'warning');
        $api_token = sica_ensure_api_token();
        if (empty($api_token)) {
            return new WP_Error('no_token', 'No valid API token available');
        }
    }

    // Build the async request
    $request_data = array(
        'site_url' => $site_url,
        'customer_email' => $customer_email,
        'content_type' => 'text',
        'model' => 'gpt-4o-search-preview',
        'messages' => array(
            array(
                'role' => 'system',
                'content' => 'You are a professional content writer creating SEO-optimized blog posts with access to current web information.'
            ),
            array(
                'role' => 'user',
                'content' => $content_prompt
            )
        ),
        'max_tokens' => 4000,
        'title' => $title,
        'generation_id' => $generation_id,
        // ASYNC MODE FLAGS
        'async' => true,
        'include_image' => !empty($image_prompt),
        'image_prompt' => $image_prompt,
        'image_model' => $image_model
    );

    sica_log('Starting async generation job', 'info', array(
        'title' => $title,
        'include_image' => !empty($image_prompt) ? 'yes' : 'no',
        'image_model' => $image_model
    ));

    $start_time = microtime(true);
    $response = wp_remote_post('https://api.sica.ai/generate.php', array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'X-Sica-Token' => $api_token
        ),
        'body' => json_encode($request_data),
        'timeout' => 30 // Short timeout - server returns immediately for async
    ));
    $duration = round(microtime(true) - $start_time, 2);

    if (is_wp_error($response)) {
        sica_log('Async start FAILED (wp_error)', 'error', array(
            'error' => $response->get_error_message(),
            'duration' => $duration . 's'
        ));
        return $response;
    }

    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);

    // Debug: Log raw response for troubleshooting
    sica_log('Async raw response', 'info', array(
        'http_code' => $response_code,
        'duration' => $duration . 's',
        'body_length' => strlen($response_body),
        'body_preview' => substr($response_body, 0, 500),
        'json_error' => json_last_error_msg()
    ));

    $data = json_decode($response_body, true);

    sica_log('Async start response', 'info', array(
        'http_code' => $response_code,
        'duration' => $duration . 's',
        'ok' => $data['ok'] ?? 'not_set',
        'async' => $data['async'] ?? 'not_set',
        'job_id' => $data['job_id'] ?? 'none',
        'json_decode_success' => $data !== null ? 'yes' : 'NO'
    ));

    if ($response_code !== 200 || !$data || !$data['ok']) {
        $error_msg = $data['error'] ?? 'Unknown error starting async job';
        sica_log('Async start failed', 'error', array(
            'error' => $error_msg,
            'http_code' => $response_code,
            'body_sample' => substr($response_body, 0, 200)
        ));
        return new WP_Error('async_start_failed', $error_msg);
    }

    return array(
        'ok' => true,
        'job_id' => $data['job_id'],
        'poll_url' => $data['poll_url'] ?? null,
        'message' => $data['message'] ?? 'Job queued'
    );
}

/**
 * Poll the status of an async generation job
 *
 * @param string $job_id The job ID to check
 * @return array Job status and results (if complete)
 */
function sica_poll_job_status($job_id) {
    $site_url = get_site_url();
    $api_token = get_option('sica_api_token', '');

    $response = wp_remote_post('https://api.sica.ai/job-status.php', array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'X-Sica-Token' => $api_token
        ),
        'body' => json_encode(array(
            'job_id' => $job_id,
            'site_url' => $site_url
        )),
        'timeout' => 15
    ));

    if (is_wp_error($response)) {
        sica_log('Job status poll failed', 'error', array(
            'job_id' => $job_id,
            'error' => $response->get_error_message()
        ));
        return array(
            'ok' => false,
            'error' => $response->get_error_message()
        );
    }

    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    $data = json_decode($response_body, true);

    if ($response_code !== 200 || !$data || !$data['ok']) {
        return array(
            'ok' => false,
            'error' => $data['error'] ?? 'Failed to get job status'
        );
    }

    return $data;
}

/**
 * Wait for async job completion with polling
 *
 * @param string $job_id The job ID to wait for
 * @param int $max_wait_seconds Maximum time to wait (default 10 minutes)
 * @param int $poll_interval_seconds How often to poll (default 3 seconds)
 * @return array Final job status with results
 */
function sica_wait_for_job_completion($job_id, $max_wait_seconds = 600, $poll_interval_seconds = 3) {
    $start_time = time();
    $last_status = '';

    sica_log('Starting job wait loop', 'info', array(
        'job_id' => $job_id,
        'max_wait' => $max_wait_seconds . 's',
        'poll_interval' => $poll_interval_seconds . 's'
    ));

    while ((time() - $start_time) < $max_wait_seconds) {
        $status = sica_poll_job_status($job_id);

        if (!$status['ok']) {
            sica_log('Job poll returned error', 'error', array(
                'job_id' => $job_id,
                'error' => $status['error'] ?? 'unknown'
            ));
            return $status;
        }

        // Log status changes to Activity Log
        if ($status['status'] !== $last_status) {
            $last_status = $status['status'];
            if (function_exists('sica_add_scheduler_log_entry')) {
                sica_add_scheduler_log_entry(array(
                    'status' => $status['status'] === 'completed' ? 'success' : 'pending',
                    'message' => $status['status_message'] ?? $status['status'],
                    'source' => 'async',
                    'post_title' => $status['title'] ?? ''
                ));
            }
        }

        // Check if job is done
        if ($status['status'] === 'completed') {
            sica_log('Job completed successfully', 'info', array(
                'job_id' => $job_id,
                'elapsed' => (time() - $start_time) . 's'
            ));
            return $status;
        }

        if ($status['status'] === 'failed') {
            sica_log('Job failed', 'error', array(
                'job_id' => $job_id,
                'error' => $status['error_message'] ?? 'unknown'
            ));
            return $status;
        }

        // Still processing - wait and poll again
        sleep($poll_interval_seconds);
    }

    // Timed out
    sica_log('Job wait timed out', 'error', array(
        'job_id' => $job_id,
        'elapsed' => (time() - $start_time) . 's'
    ));

    return array(
        'ok' => false,
        'status' => 'timeout',
        'error' => 'Job did not complete within ' . $max_wait_seconds . ' seconds'
    );
}

// Complete generation tracking
function sica_complete_generation($generation_id, $success, $post_id = null, $error_message = null) {
    $response = wp_remote_post('https://api.sica.ai/subscription.php', array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'X-Sica-Token' => get_option('sica_api_token', '')
        ),
        'body' => json_encode(array(
            'action' => 'complete_generation',
            'generation_id' => $generation_id,
            'success' => $success,
            'post_id' => $post_id,
            'error_message' => $error_message
        )),
        'timeout' => 15
    ));
    
    if (is_wp_error($response)) {
        // error_log('Sica Content AI: Failed to complete generation tracking - ' . $response->get_error_message());
    }
}

/**
 * AJAX handler for email logout/change
 */
function sica_ajax_logout_email() {
    // Verify nonce for security
    if (!check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce', false)) {
        wp_send_json_error('Security verification failed');
        return;
    }
    
    try {
        // Clear all email verification related options
        delete_option('sica_customer_email');
        delete_option('sica_email_verified');
        delete_option('sica_license_status');
        delete_option('sica_license_key');
        delete_option('sica_posts_remaining');
        delete_option('sica_verification_code');
        delete_option('sica_verification_time');
        
        // Clear any cached subscription data
        delete_transient('sica_subscription_status');
        delete_transient('sica_usage_dashboard');
        
        // error_log('[SICA_LOGOUT] User logged out - all email verification data cleared');
        
        wp_send_json_success('Email logout successful');
        
    } catch (Exception $e) {
        // error_log('[SICA_LOGOUT] Logout failed: ' . $e->getMessage());
        wp_send_json_error('Failed to logout: ' . $e->getMessage());
    }
}
add_action('wp_ajax_sica_logout_email', 'sica_ajax_logout_email');



?>