<?php
/**
 * Title and Keyword Generation
 * 
 * Handles AI-powered title and keyword generation functionality
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Debug logging helper for plugin PHP (respects WP_DEBUG)
if (!function_exists('sica_debug_log')) {
    function sica_debug_log($message) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            if (is_array($message) || is_object($message)) {
                // error_log(print_r($message, true));
            } else {
                // error_log($message);
            }
        }
    }
}

/**
 * Generate blog titles callback (AJAX handler)
 */
function generate_blog_titles_callback() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    
    $number_of_titles = isset($_POST['number_of_titles']) ? (int)$_POST['number_of_titles'] : 5;
    $force_bypass_cache = !empty($_POST['bypass_cache']);
    $existing_titles_raw = isset($_POST['current_titles']) ? $_POST['current_titles'] : '';
    
    $keywords_string = get_option('my_auto_blog_keywords', '');
    $business_details = get_business_details();
    
    // Process existing titles for deduplication
    $existing_titles = array();
    if (!empty($existing_titles_raw)) {
        $existing_titles = array_filter(array_map('trim', explode("\n", stripslashes($existing_titles_raw))));
    }
    
    $new_titles = call_chatgpt_to_generate_titles_with_current_data(
        $number_of_titles, 
        $keywords_string, 
        $business_details, 
        $existing_titles,
        null,
        $force_bypass_cache
    );
    
    sica_debug_log('New titles returned from generation function: ' . json_encode($new_titles));
    sica_debug_log('Number of new titles: ' . count($new_titles));
    
    if (!empty($new_titles)) {
        // Merge existing titles with new titles
        $all_titles = array_merge($existing_titles, $new_titles);
        sica_debug_log('Merged titles (existing + new): ' . json_encode($all_titles));
        
        sica_debug_log('Sending success response with merged titles: ' . json_encode($all_titles));
        wp_send_json_success(array('titles' => $all_titles));
    } else {
        // Check if email is verified
        $customer_email = get_option('sica_customer_email', '');
        if (empty($customer_email)) {
            wp_send_json_error('Please verify your email address before generating titles. Go to the Setup tab to verify your email.');
        } else {
            wp_send_json_error('Failed to generate titles. Please check your subscription status or try again in a few minutes.');
        }
    }
}
add_action('wp_ajax_generate_blog_titles', 'generate_blog_titles_callback');

/**
 * Save titles callback (AJAX handler)
 */
function save_titles_callback() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    
    $raw_titles = isset($_POST['titles']) ? $_POST['titles'] : '';
    $titles = sanitize_textarea_field(stripslashes($raw_titles));
    
    sica_debug_log('Raw titles received: ' . $raw_titles);
    sica_debug_log('Cleaned titles for saving: ' . $titles);
    
    // Save titles to WordPress option
    update_option('my_auto_blog_title', $titles);
    
    sica_debug_log('Titles auto-saved successfully');
    wp_send_json_success('Titles saved successfully');
}
add_action('wp_ajax_save_titles', 'save_titles_callback');

/**
 * Generate keywords callback (AJAX handler)
 */
function generate_keywords_callback() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    
    // Get existing keywords for context
    $existing_keywords_raw = isset($_POST['current_keywords']) ? $_POST['current_keywords'] : '';
    $existing_keywords = array();
    if (!empty($existing_keywords_raw)) {
        $existing_keywords = array_filter(array_map('trim', explode("\n", stripslashes($existing_keywords_raw))));
    }
    
    $business_name = get_bloginfo('name');
    $business_category = get_option('my_auto_blog_business_category', '');
    $business_service_area = get_option('my_auto_blog_service_area', '');
    $products_services = get_option('my_auto_blog_products_services', '');
    
    $new_keywords = call_chatgpt_to_generate_keywords(
        $business_name, 
        $business_category, 
        $business_service_area, 
        $products_services,
        $existing_keywords
    );
    
    sica_debug_log('New keywords returned from generation function: ' . json_encode($new_keywords));
    sica_debug_log('Number of new keywords: ' . count($new_keywords));
    
    if (!empty($new_keywords)) {
        // Merge existing keywords with new keywords
        $all_keywords = array_merge($existing_keywords, $new_keywords);
        sica_debug_log('Merged keywords (existing + new): ' . json_encode($all_keywords));
        
        sica_debug_log('Sending success response with merged keywords: ' . json_encode($all_keywords));
        wp_send_json_success(array('keywords' => $all_keywords));
    } else {
        // Check if email is verified
        $customer_email = get_option('sica_customer_email', '');
        if (empty($customer_email)) {
            wp_send_json_error('Please verify your email address before generating keywords. Go to the Setup tab to verify your email.');
        } else {
            wp_send_json_error('Failed to generate keywords. Please check your subscription status or try again in a few minutes.');
        }
    }
}
add_action('wp_ajax_generate_keywords', 'generate_keywords_callback');

/**
 * Call ChatGPT to generate titles with current data
 */
function call_chatgpt_to_generate_titles_with_current_data($number_of_titles, $keywords_string, $business_details, $existing_titles = array(), $randomization_seed = null, $force_bypass_cache = false) {
    sica_debug_log('=== SIMPLIFIED TITLE GENERATION START ===');
    sica_debug_log('Number of titles requested: ' . $number_of_titles);
    
    $customer_email = get_option('sica_customer_email', '');
    $site_url = get_site_url();
    
    if (empty($customer_email)) {
        sica_debug_log('Customer email is missing for title generation.');
        return array();
    }
    
    // Format keywords (filter out irrelevant terms)
    $keywords_arr = explode("\n", $keywords_string);
    $disallowed_terms_kw = sica_get_disallowed_vertical_terms($business_details['business_category'] ?? '');
    $keywords_arr = array_values(array_filter(array_map('trim', $keywords_arr), function($k) use ($disallowed_terms_kw) {
        $kl = strtolower($k);
        foreach ($disallowed_terms_kw as $bad) {
            if ($bad && strpos($kl, strtolower($bad)) !== false) { return false; }
        }
        return true;
    }));
    $keywords_formatted = implode(", ", array_filter($keywords_arr));
    
    // Get current date and build deduplication context
    $current_date = date('Y-m-d');
    $current_year = date('Y');
    $existing_ctx = sica_get_existing_titles_context($existing_titles);
    $avoid_concepts = implode(', ', $existing_ctx['avoid_head_terms']);
    $avoid_titles_sample = implode("\n", array_slice($existing_ctx['titles'], 0, 40));
    $randomizer = substr(sha1(uniqid('', true)), 0, 8);
    
    // High-converting title generation with psychological hooks
    $prompt = "<instructions>
TASK: Generate EXACTLY {$number_of_titles} high-converting blog titles that trigger curiosity, fear-of-missing-out, or problem-awareness.

OUTPUT FORMAT: Return {$number_of_titles} lines. Each line = ONE blog title only. No numbering, no explanations.

<context>
DATE: {$current_date}
BUSINESS: {$business_details['business_name']} - {$business_details['business_category']}
SERVICES: {$business_details['products_services']}
LOCATION: {$business_details['business_service_area']}
TARGET AUDIENCE KEYWORDS: {$keywords_formatted}
</context>

<psychology>
CORE PRINCIPLE: Stop writing for \"keywords\" — write for \"scenarios\" and \"anxieties.\"

Your target reader is a business owner or decision-maker who:
- Has heard about changes in their industry but doesn't know how to adapt
- Suspects competitors are doing something they're not
- Wants specific answers, not generic advice
- Is skeptical of fluff and \"ultimate guides\"

TITLE FORMULAS THAT CONVERT:

1. THE CURIOSITY GAP (use \"quotes\" for mystery):
   - The \"Entity\" Secret: Why Google Recommends Some {$business_details['business_category']}s but Ignores Others
   - The \"Zero-Click\" Problem: Is Your Website Invisible in AI Search Results?

2. NEGATIVE FRAMING (triggers loss aversion):
   - Why Your Competitor's {service} is Outranking You (And How to Fix It)
   - The {$business_details['business_category']} Mistake That's Costing You Customers
   - Stop Making This {service} Error That Drives Clients Away

3. SPECIFIC VS. COMPARISON:
   - {Option A} vs. {Option B}: Which One Actually Works for {location} Businesses?
   - DIY vs. Professional {service}: The Real Cost Breakdown for {location}

4. FEAR/URGENCY HOOKS:
   - Will {industry change} Kill Your {business type}? Here's How to Stay Relevant
   - The {timeframe} Deadline Every {location} {business type} Needs to Know About

5. MYTH-BUSTING:
   - The \"{common belief}\" Myth: What Actually Works for {service} in {current_year}
   - Why Everything You've Heard About {topic} is Wrong

6. HYPER-LOCAL SPECIFICITY:
   - {location}'s Hidden {service} Problem (And the {number} Businesses Solving It)
   - What {location} {business types} Know About {topic} That Others Don't

7. COST/ROI HOOKS:
   - \${low amount} vs. \${high amount}: Why {cheaper option} Outperforms {expensive option}
   - The Math Behind {service}: How {specific metric} Pays for Itself
</psychology>

<requirements>
- EVERY title must use ONE of the formulas above
- Include \"quoted phrases\" in at least 2 titles for curiosity gaps
- Use negative framing in at least 2 titles (\"mistake\", \"problem\", \"killing\", \"costing\")
- Include specific numbers, years, or percentages where natural
- Reference {$business_details['business_service_area']} in 1-2 titles for local relevance
- Vary the emotional triggers (curiosity, fear, comparison, urgency)
- NO generic words: Essential, Comprehensive, Guide, Ultimate, Complete, Top, Best, Tips, Secrets, Key
</requirements>

<avoid_duplicating>
{$avoid_titles_sample}
</avoid_duplicating>

RANDOMIZER: {$randomizer}
</instructions>";

    $request_data = array(
        'site_url' => $site_url,
        'customer_email' => $customer_email,
        'content_type' => 'text',
        'model' => 'gpt-4o',
        'temperature' => 1.1,
        'presence_penalty' => 0.7,
        'frequency_penalty' => 0.4,
        'messages' => array(
            array('role' => 'system', 'content' => 'You are a conversion copywriter who specializes in blog titles that STOP the scroll and DEMAND clicks. You understand that boring "how-to" titles get ignored while psychologically-triggering titles get read.

Your expertise:
- Curiosity gaps using "quoted phrases" that make readers need to know more
- Negative framing that triggers loss aversion ("The Mistake Costing You...")
- Specific numbers and comparisons that feel concrete, not generic
- Local relevance that makes readers think "this is about MY situation"
- Myth-busting angles that challenge conventional wisdom

You NEVER write generic SEO-bait titles. Every title must have a psychological hook.

Return only titles, one per line. No numbering or explanations.'),
            array('role' => 'user', 'content' => $prompt)
        ),
        'max_tokens' => 5000,
        'title' => 'Title Generation'
    );
    
    sica_debug_log('Making AI call with web search for dynamic title generation');
    sica_debug_log('Prompt length: ' . strlen($prompt) . ' characters');
    sica_debug_log('Number of titles requested: ' . $number_of_titles);
    sica_debug_log('Prompt preview: ' . substr($prompt, 0, 200) . '...');
    
    $response = wp_remote_post('https://api.sica.ai/generate.php', array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'X-Sica-Token' => get_option('sica_api_token', '')
        ),
        'body' => json_encode($request_data),
        'timeout' => 120,
    ));
    
    if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
        sica_debug_log('Title generation request failed');
        return array();
    }
    
    $data = json_decode(wp_remote_retrieve_body($response), true);
    if (!$data || empty($data['ok'])) {
        sica_debug_log('Invalid response from title generation');
        return array();
    }
    
    $content = $data['data']['choices'][0]['message']['content'] ?? '';
    if (!$content) {
        sica_debug_log('No content in title generation response');
        return array();
    }
    
    sica_debug_log('Raw AI response length: ' . strlen($content) . ' characters');
    sica_debug_log('Raw AI response: ' . $content);
    
    // Clean and process titles (simplified pipeline)
    $titles = array_filter(array_map('trim', explode("\n", stripslashes($content))));
    $cleaned = clean_blog_titles($titles);
    $variety = enforce_title_variety($cleaned);
    $dedup = remove_internal_duplicates($variety);
    $novel_titles = filter_out_existing_titles($dedup, $existing_ctx);
    
    sica_debug_log('Generated ' . count($novel_titles) . ' novel titles');
    
    return array_slice($novel_titles, 0, $number_of_titles);
}

// Trend research function removed - now using single-call approach with integrated web search

/**
 * Call ChatGPT to generate keywords with existing context
 */
function call_chatgpt_to_generate_keywords($business_name, $business_category, $business_service_area, $products_services, $existing_keywords = array()) {
    sica_debug_log('=== KEYWORD GENERATION START ===');
    sica_debug_log('Existing keywords count: ' . count($existing_keywords));
    
    // Use SaaS proxy instead of direct API key
    $customer_email = get_option('sica_customer_email', '');
    $site_url = get_site_url();
    
    sica_debug_log('Customer email present: ' . (!empty($customer_email) ? 'yes' : 'no'));
    sica_debug_log('Site URL: ' . $site_url);
    
    if (empty($customer_email)) {
        sica_debug_log('Customer email is missing for keyword generation.');
        return array();
    }
    
    // Format existing keywords for prompt
    $existing_list = '';
    if (!empty($existing_keywords)) {
        $existing_list = "\n\nExisting keywords to avoid duplicating:\n" . implode("\n", $existing_keywords);
    }
    
    $full_prompt = "Generate 5 high-value SEO keywords for {$business_name}, a {$business_category} business in {$business_service_area}.

Products/Services: {$products_services}.
Existing keywords: {$existing_list}.

Strategy - Focus on keywords with:
- HIGH commercial intent (people ready to buy/hire)
- LOCAL search potential (include {$business_service_area} when valuable)
- SPECIFIC services this business offers
- BUYER-focused terms (not just informational)

Keyword Types to Generate:
1. Primary service keyword (e.g., 'deck builder')
2. Service + location (e.g., 'deck builder {$business_service_area}')
3. Problem-solving service (e.g., 'deck repair')
4. Specific offering (e.g., 'composite decking') 
5. Competition-beating term (e.g., 'affordable deck builder')

Requirements:
- 1-4 words maximum each
- Focus on what customers search when ready to hire
- Prioritize services this business actually provides
- Include local terms for geographic relevance
- Avoid generic terms like 'construction' or 'services'
- Return one keyword per line, no explanations

Generate keywords customers use when they need this business NOW.";
    
    // Use SaaS proxy for keyword generation
    $request_data = array(
        'site_url' => $site_url,
        'customer_email' => $customer_email,
        'content_type' => 'text',
        'model' => 'gpt-4o-mini',
        'messages' => array(
            array(
                'role' => 'system',
                'content' => 'You are a local business SEO expert focused on HIGH-VALUE commercial keywords. Generate ONLY short keywords (1-4 words) that drive actual customers. Think: What do people search when they\'re ready to HIRE this business? Focus on services, solutions, and local terms. Avoid generic words. Good examples, but be creative and focus on both short and long tail keywords: "emergency deck repair", "deck builder [city]", "composite deck installation".'
            ),
            array(
                'role' => 'user',
                'content' => $full_prompt
            )
        ),
        'max_tokens' => 500,
        'title' => 'Keyword Generation'
    );
    
    $response = wp_remote_post('https://api.sica.ai/generate.php', array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'X-Sica-Token' => get_option('sica_api_token', '')
        ),
        'body' => json_encode($request_data),
        'timeout' => 60,
    ));
    
    if (is_wp_error($response)) {
        sica_debug_log('Error in keyword generation request');
        return array();
    }
    
    $response_code = wp_remote_retrieve_response_code($response);
    if ($response_code !== 200) {
        sica_debug_log('SaaS API error for keyword generation: ' . $response_code);
        return array();
    }
    
    $response_body = wp_remote_retrieve_body($response);
    
    sica_debug_log('SaaS API request to generate keywords was made');
    
    $data = json_decode($response_body, true);
    
    // Handle SaaS proxy response format
    if (!$data || !$data['ok']) {
        $error_msg = $data['error'] ?? 'Unknown SaaS API error';
        sica_debug_log('SaaS API error for keyword generation');
        return array();
    }
    
    // error_log('SaaS response successful, extracting OpenAI data...');
    $openai_data = $data['data'];
    if (isset($openai_data['choices'][0]['message']['content'])) {
        $content = $openai_data['choices'][0]['message']['content'];
        $keywords = array_filter(array_map(function($keyword) {
            return trim(stripslashes($keyword));
        }, explode("\n", $content)));
        
        sica_debug_log('Raw keywords extracted');
        
        // Clean and process keywords
        $cleaned_keywords = clean_keywords($keywords);
        sica_debug_log('Final processed keywords count: ' . count($cleaned_keywords));
        return $cleaned_keywords;
    }
    
    sica_debug_log('No content found in OpenAI response');
    return array();
}

/**
 * Clean and format keywords array
 */
function clean_keywords($keywords) {
    if (empty($keywords)) {
        return array();
    }
    
    $cleaned_keywords = array();
    
    foreach ($keywords as $keyword) {
        $keyword = trim(stripslashes($keyword));
        
        if (empty($keyword)) {
            continue;
        }
        
        // Remove numbering and formatting
        $keyword = preg_replace('/^\d+[\.\)\-\s]*/', '', $keyword);
        $keyword = preg_replace('/^[\-\*\•\s]*/', '', $keyword);
        $keyword = trim($keyword, ' .-"\'');
        
        // Remove quotes if they wrap the entire keyword
        if ((substr($keyword, 0, 1) === '"' && substr($keyword, -1) === '"') ||
            (substr($keyword, 0, 1) === "'" && substr($keyword, -1) === "'")) {
            $keyword = substr($keyword, 1, -1);
        }
        
        // Convert comma-separated to individual keywords
        if (strpos($keyword, ',') !== false) {
            $comma_keywords = explode(',', $keyword);
            foreach ($comma_keywords as $comma_keyword) {
                $comma_keyword = trim($comma_keyword);
                if (!empty($comma_keyword) && str_word_count($comma_keyword) <= 4) {
                    $cleaned_keywords[] = $comma_keyword;
                }
            }
        } else {
            // Only accept keywords with 4 words or less
            if (!empty($keyword) && str_word_count($keyword) <= 4) {
                $cleaned_keywords[] = $keyword;
            }
        }
    }
    
    // Remove duplicates and score for business value
    $unique_keywords = array_unique($cleaned_keywords);
    
    // Score keywords by business value (higher = better)
    $scored_keywords = array();
    foreach ($unique_keywords as $keyword) {
        $score = score_keyword_value($keyword);
        $scored_keywords[] = array('keyword' => $keyword, 'score' => $score);
    }
    
    // Sort by score (highest first) and return top 5
    usort($scored_keywords, function($a, $b) {
        return $b['score'] - $a['score'];
    });
    
    $top_keywords = array();
    for ($i = 0; $i < min(5, count($scored_keywords)); $i++) {
        $top_keywords[] = $scored_keywords[$i]['keyword'];
    }
    
    return $top_keywords;
}

/**
 * Score keyword value for business potential (higher = better)
 */
function score_keyword_value($keyword) {
    $keyword_lower = strtolower($keyword);
    $score = 0;
    
    // High-value commercial intent words (+10 points each)
    $commercial_terms = array('builder', 'contractor', 'installation', 'repair', 'replacement', 'service', 'company', 'expert', 'professional');
    foreach ($commercial_terms as $term) {
        if (strpos($keyword_lower, $term) !== false) {
            $score += 10;
        }
    }
    
    // Problem-solving terms (+8 points each)
    $problem_terms = array('emergency', 'fix', 'broken', 'damaged', 'restoration', 'maintenance');
    foreach ($problem_terms as $term) {
        if (strpos($keyword_lower, $term) !== false) {
            $score += 8;
        }
    }
    
    // Buying modifiers (+6 points each)
    $buying_modifiers = array('affordable', 'cheap', 'best', 'top', 'quality', 'local', 'near me', 'custom');
    foreach ($buying_modifiers as $term) {
        if (strpos($keyword_lower, $term) !== false) {
            $score += 6;
        }
    }
    
    // Specific materials/products (+5 points each)
    $specific_terms = array('composite', 'wood', 'vinyl', 'cedar', 'pressure treated', 'outdoor', 'deck', 'patio');
    foreach ($specific_terms as $term) {
        if (strpos($keyword_lower, $term) !== false) {
            $score += 5;
        }
    }
    
    // Location-based gets bonus (+4 points)
    if (preg_match('/\b(city|town|area|local)\b/', $keyword_lower) || 
        preg_match('/\b[A-Z][a-z]+\s+(city|town|area|local)\b/', $keyword)) {
        $score += 4;
    }
    
    // Penalize generic terms (-5 points each)
    $generic_terms = array('services', 'solutions', 'business', 'general', 'construction', 'work');
    foreach ($generic_terms as $term) {
        if (strpos($keyword_lower, $term) !== false) {
            $score -= 5;
        }
    }
    
    // Bonus for optimal length (2-3 words = +3 points)
    $word_count = str_word_count($keyword);
    if ($word_count >= 2 && $word_count <= 3) {
        $score += 3;
    }
    
    return max(0, $score); // Ensure score is never negative
}

/**
 * Clean blog titles from various formatting issues
 */
function clean_blog_titles($titles) {
    if (empty($titles)) {
        return array();
    }
    
    // Split into array if it's a string
    if (is_string($titles)) {
        $titles_array = explode("\n", $titles);
    } else {
        $titles_array = $titles;
    }
    
    $cleaned_titles = array();
    
    foreach ($titles_array as $title) {
        $title = trim(stripslashes($title));
        
        if (empty($title)) {
            continue;
        }
        
        // Remove numbering and formatting
        $title = preg_replace('/^\d+[\.\)\-\s]*/', '', $title);
        $title = preg_replace('/^[\-\*\•\s]*/', '', $title);
        $title = trim($title, ' .-"\'');
        
        // Remove quotes if they wrap the entire title
        if ((substr($title, 0, 1) === '"' && substr($title, -1) === '"') ||
            (substr($title, 0, 1) === "'" && substr($title, -1) === "'")) {
            $title = substr($title, 1, -1);
        }
        
        if (!empty($title) && strlen($title) > 10) {
            $cleaned_titles[] = $title;
        }
    }
    
    return $cleaned_titles;
}

/**
 * Remove duplicate titles from array
 */
function remove_internal_duplicates($titles) {
    if (empty($titles)) {
        return array();
    }
    
    $unique_titles = array();
    $normalized_titles = array();
    
    foreach ($titles as $title) {
        $normalized = normalize_title_for_comparison($title);
        
        // Check if this normalized version already exists
        $is_duplicate = false;
        foreach ($normalized_titles as $existing_normalized) {
            $similarity = calculate_title_similarity($normalized, $existing_normalized);
            if ($similarity > 0.8) { // 80% similarity threshold
                $is_duplicate = true;
                break;
            }
        }
        
        if (!$is_duplicate) {
            $unique_titles[] = $title;
            $normalized_titles[] = $normalized;
        }
    }
    
    return $unique_titles;
}

/**
 * Normalize title for comparison
 */
function normalize_title_for_comparison($title) {
    $normalized = strtolower($title);
    $normalized = preg_replace('/[^a-z0-9\s]/', '', $normalized);
    $normalized = preg_replace('/\s+/', ' ', $normalized);
    return trim($normalized);
}

/**
 * Calculate similarity between two titles
 */
function calculate_title_similarity($title1, $title2) {
    if (empty($title1) || empty($title2)) {
        return 0;
    }
    
    // Use Levenshtein distance for similarity calculation
    $max_length = max(strlen($title1), strlen($title2));
    if ($max_length === 0) {
        return 1; // Both strings are empty
    }
    
    $distance = levenshtein($title1, $title2);
    $similarity = 1 - ($distance / $max_length);
    
    return max(0, $similarity); // Ensure non-negative
}

/**
 * Generate a replacement title when one is used
 */
function generate_replacement_title() {
    $keywords = get_option('my_auto_blog_keywords', '');
    $business_details = get_business_details();
    
    $titles = call_chatgpt_to_generate_titles_with_current_data(1, $keywords, $business_details);
    
    return !empty($titles) ? $titles[0] : '';
}

/**
 * Generate initial set of titles
 */
function generate_initial_title_set() {
    $keywords = get_option('my_auto_blog_keywords', '');
    $business_details = get_business_details();
    
    return call_chatgpt_to_generate_titles_with_current_data(5, $keywords, $business_details);
}

/**
 * Save keywords callback (AJAX handler)
 */
function save_keywords_callback() {
    // Security: Check user capabilities first
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }

    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');

    if (!isset($_POST['keywords'])) {
        wp_send_json_error('No keywords provided');
    }

    $keywords = sanitize_textarea_field($_POST['keywords']);
    
    // Save the keywords
    update_option('my_auto_blog_keywords', $keywords);

    wp_send_json_success('Keywords saved successfully');
}
add_action('wp_ajax_save_keywords', 'save_keywords_callback');

/**
 * Enforce title variety by removing banned phrases and ensuring unique head terms
 */
function enforce_title_variety($titles) {
    if (empty($titles)) {
        return array();
    }
    
    // Banned generic phrases - these produce boring, low-CTR titles
    $banned_phrases = array(
        'Essential', 'Comprehensive', 'Guide', 'Understanding', 'Key Considerations',
        'Insights', 'Best Practices', 'Current Trends', 'What You Need to Know',
        'Complete Guide', 'Ultimate', 'Everything You Need', 'The Definitive',
        'Secrets', 'Hidden Truths', 'Shocking', 'Amazing', 'Incredible',
        'Top Tips', 'Expert Tips', 'Pro Tips', 'Key Tips', 'Quick Tips',
        'Simple Steps', 'Easy Steps', 'Key Steps', 'Important Steps',
        'Must-Know', 'Need to Know', 'Should Know', 'Everything About',
        'All About', 'Introduction to', 'Beginners Guide', 'Starter Guide',
        'Key Factors', 'Important Factors', 'Key Elements', 'Key Points',
        'In-Depth Look', 'Deep Dive', 'Exploring', 'Examining', 'Navigating',
        'Mastering', 'Unlocking', 'Demystifying', 'Maximizing', 'Optimizing',
        'Enhancing', 'Boosting', 'Improving', 'Leveraging', 'Harnessing',
        'A Look at', 'An Overview of', 'Breaking Down', 'The Importance of'
    );
    
    $filtered_titles = array();
    $used_head_terms = array();
    
    foreach ($titles as $title) {
        $cleaned_title = trim($title);
        
        // Skip titles that contain banned phrases (don't try to fix them)
        $contains_banned = false;
        foreach ($banned_phrases as $phrase) {
            if (preg_match('/\b' . preg_quote($phrase, '/') . '\b/i', $cleaned_title)) {
                $contains_banned = true;
                break;
            }
        }
        
        if ($contains_banned) {
            sica_debug_log("Skipping title with banned phrase: " . $cleaned_title);
            continue; // Skip this title entirely
        }
        
        // Skip if title is too short
        if (strlen($cleaned_title) < 15) {
            continue;
        }
        
        // Extract head term (first 1-2 significant words)
        $head_term = extract_head_term($cleaned_title);
        
        // Skip if head term already used
        if (in_array($head_term, $used_head_terms)) {
            continue;
        }
        
        $filtered_titles[] = $cleaned_title;
        $used_head_terms[] = $head_term;
    }
    
    return $filtered_titles;
}

/**
 * Filter out generic/keyword-echo titles; enforce specificity and qualifiers
 */
function sica_filter_titles_for_specificity($titles, $keywords_arr, $business_details) {
    if (empty($titles)) { return array(); }

    $service_area = strtolower(trim((string)($business_details['business_service_area'] ?? '')));
    $category = strtolower(trim((string)($business_details['business_category'] ?? '')));
    $services_blob = strtolower(trim((string)($business_details['products_services'] ?? '')));

    // Build a quick set of keyword stems to detect pure echo
    $stems = array();
    foreach ((array)$keywords_arr as $kw) {
        $kw = strtolower(trim($kw));
        if ($kw === '') { continue; }
        // Keep meaningful words only
        $parts = preg_split('/\s+/', $kw);
        $parts = array_filter($parts, function($w) {
            return strlen($w) > 3 && !in_array($w, array('with','and','for','near','best','top','the','your','area','city','local'));
        });
        if (!empty($parts)) { $stems[] = implode(' ', $parts); }
    }

    $out = array();
    foreach ($titles as $t) {
        $t_norm = strtolower(trim($t));
        if ($t_norm === '') { continue; }

        // Heuristic 1: avoid exact keyword echo (title equals or is near-identical to a keyword)
        $is_echo = false;
        foreach ($stems as $stem) {
            if (levenshtein($t_norm, $stem) <= 3) { $is_echo = true; break; }
        }
        if ($is_echo) { continue; }

        // Heuristic 2: require at least one qualifier token (who/when/why/how/which)
        if (!preg_match('/\b(why|when|how|what|which|vs\.?|versus|before|after|choose|whether)\b/i', $t)) {
            // If it lacks qualifier, allow only if it contains a concrete modifier (e.g., specific scenario words)
            if (!preg_match('/\b(checklist|criteria|signs|mistakes|factors|trade[- ]?offs|thresholds|timeline|budget|constraints)\b/i', $t)) {
                continue;
            }
        }

        // Heuristic 3: penalize pure GEO+service generic phrasing
        if (preg_match('/\b(affordable|best|top|ultimate|complete)\b/i', $t)) { continue; }
        if (preg_match('/\b(service|services|agency|company) in [a-z\s]+$/i', $t)) { continue; }

        $out[] = $t;
    }
    return $out;
}

/**
 * Extract the main head term from a title for uniqueness checking
 */
function extract_head_term($title) {
    // Convert to lowercase for comparison
    $title_lower = strtolower($title);
    
    // Remove common article words at the beginning
    $title_lower = preg_replace('/^(the|a|an|how|why|when|what|where)\s+/i', '', $title_lower);
    
    // Extract first 1-2 significant words
    $words = explode(' ', $title_lower);
    $significant_words = array();
    
    $skip_words = array('to', 'for', 'in', 'on', 'at', 'by', 'with', 'from', 'of', 'is', 'are', 'was', 'were');
    
    foreach ($words as $word) {
        $word = trim($word, '.,!?:"\'');
        if (!empty($word) && !in_array($word, $skip_words) && strlen($word) > 2) {
            $significant_words[] = $word;
            if (count($significant_words) >= 2) {
                break;
            }
        }
    }
    
    return implode(' ', $significant_words);
}

/**
 * Build context of existing/past titles to avoid repetition
 */
function sica_get_existing_titles_context($existing_titles_param = array()) {
    $titles = array();
    // 1) Titles stored in option queue
    $queued_raw = (string) get_option('my_auto_blog_title', '');
    if (!empty($queued_raw)) {
        $titles = array_merge($titles, array_values(array_filter(array_map('trim', explode("\n", $queued_raw)))));
    }
    // 2) Passed-in titles from UI (e.g., textbox)
    if (!empty($existing_titles_param)) {
        $titles = array_merge($titles, array_values(array_filter(array_map('trim', (array) $existing_titles_param))));
    }
    // 3) Recently published post titles (limit 50)
    $recent = get_posts(array(
        'numberposts' => 50,
        'post_type' => 'post',
        'post_status' => array('publish', 'draft'),
        'orderby' => 'date',
        'order' => 'DESC'
    ));
    foreach ($recent as $p) {
        $titles[] = get_the_title($p->ID);
    }
    // Normalize + unique
    $titles = array_values(array_unique(array_filter(array_map('trim', $titles))));

    // Extract head terms to avoid repeating same concept
    $head_terms = array();
    foreach ($titles as $t) {
        $head = extract_head_term($t);
        if (!empty($head)) { $head_terms[] = $head; }
    }
    $head_terms = array_values(array_unique($head_terms));
    return array(
        'titles' => $titles,
        'avoid_head_terms' => $head_terms
    );
}

/**
 * Filter out titles that overlap with existing titles or concepts
 */
function filter_out_existing_titles($candidate_titles, $existing_ctx) {
    $out = array();
    foreach ($candidate_titles as $t) {
        if (!sica_title_overlaps_existing($t, $existing_ctx)) {
            $out[] = $t;
        }
    }
    return $out;
}

/**
 * Determine if a title overlaps with existing titles/concepts
 */
function sica_title_overlaps_existing($title, $existing_ctx) {
    $norm = normalize_title_for_comparison($title);
    foreach ($existing_ctx['titles'] as $et) {
        $sim = calculate_title_similarity($norm, normalize_title_for_comparison($et));
        if ($sim > 0.82) { // slightly stricter than internal duplicate threshold
            return true;
        }
    }
    $head = extract_head_term($title);
    if (!empty($head) && in_array($head, $existing_ctx['avoid_head_terms'], true)) {
        return true;
    }
    return false;
}

/**
 * Provide disallowed vertical terms to keep research on-category
 */
function sica_get_disallowed_vertical_terms($business_category) {
    $category = strtolower(trim((string) $business_category));
    // Default disallowed set focuses on home-services/real-estate spillover when category is not related
    $home_services_terms = array(
        'real estate', 'mortgage', 'home prices', 'secondary suites', 'renovations',
        'homeowner rebates', 'pst rebate', 'energy-efficient home', 'landlord', 'tenant',
        'zoning bylaw', 'property tax', 'snow removal budget', 'gutter', 'roof moss', 'siding'
    );
    // If the category clearly IS home services, do not disallow those
    $is_home_services = preg_match('/(home|property|cleaning|contractor|remodel|roof|gutter|window|landscap|hvac|plumb|electric|painting)/i', $category);
    if ($is_home_services) {
        return array();
    }
    return $home_services_terms;
}

