jQuery(document).ready(function($) {
    let formChanged = false;

    function setupFormChangeTracking() {
        const form = document.getElementById('content-planner-form');
        if (form) {
            const inputs = form.querySelectorAll('input, textarea, select');
            inputs.forEach(input => {
                input.addEventListener('change', () => {
                    formChanged = true;
                });
            });

            form.addEventListener('submit', () => {
                formChanged = false;
            });
        }
    }

    window.addEventListener('load', setupFormChangeTracking);

    window.addEventListener('beforeunload', (event) => {
        if (formChanged) {
            event.preventDefault();
            event.returnValue = '';
        }
    });

    // Content-based keyword generation
    $('#generate_keywords_content').off('click').on('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        var button = $(this);
        var originalText = button.text();
        button.prop('disabled', true).text('Generating...');

        var currentKeywords = $('#my_auto_blog_keywords').val();
        var currentTitles = $('#my_auto_blog_title').val();
        var businessCategory = $('#business_category').val();
        var serviceArea = $('#business_service_area').val();
        var productsServices = $('#products_services').val();


        // First, quick-save current planner values to ensure server has the latest edits
        $.ajax({
            url: myAutoBlog.ajax_url,
            type: 'POST',
            data: {
                action: 'save_content_planner_quick',
                my_auto_blog_keywords: currentKeywords,
                my_auto_blog_title: currentTitles,
                business_category: businessCategory,
                business_service_area: serviceArea,
                products_services: productsServices,
                _ajax_nonce: myAutoBlog.nonce
            },
            dataType: 'json'
        }).always(function() {
            var data = {
                action: 'generate_keywords',
                _ajax_nonce: myAutoBlog.nonce,
                current_keywords: currentKeywords,
                business_name: $('#business_name').val(),
                business_category: businessCategory,
                business_service_area: serviceArea,
                products_services: productsServices
            };

            $.ajax({
                url: myAutoBlog.ajax_url,
                type: 'POST',
                data: data,
                timeout: 90000, // 90 seconds for keyword generation
                success: function(response) {
                    if (response.success) {
                        var keywords = response.data.keywords;
                        var newValue = keywords.join('\n');
                        var textarea = $('#my_auto_blog_keywords');
                        textarea
                            .val(newValue)
                            .css({ 'background-color': '#90EE90', 'transition': 'none' })
                            .outerHeight();
                        setTimeout(function() {
                            textarea.css({ 'background-color': '', 'transition': 'background-color 0.75s ease-out' });
                        }, 500);
                        saveKeywords(newValue); // auto-save
                        // Keywords updated successfully
                    } else {
                        // Server error occurred
                        alert('Error generating keywords: ' + (response.data?.message || 'Unknown error'));
                    }
                },
                error: function(jqXHR, textStatus, errorThrown) {
                    // Network error occurred
                    alert('Failed to generate keywords');
                },
                complete: function() {
                    button.prop('disabled', false).text('🔍 Generate 5 More Keywords');
                }
            });
        });
    });

    function saveKeywords(keywords) {
        var data = {
            action: 'save_keywords',
            keywords: keywords,
            _ajax_nonce: myAutoBlog.nonce
        };

        $.ajax({
            url: myAutoBlog.ajax_url,
            type: 'POST',
            data: data,
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    // Keywords saved successfully
                } else {
                    // Error saving keywords
                }
            },
            error: function(jqXHR, textStatus, errorThrown) {
                // AJAX error while saving keywords
            }
        });
    }

    function saveTitles(titles) {
        // Remove escaped slashes before saving
        var cleanTitles = titles.replace(/\\'/g, "'").replace(/\\"/g, '"').replace(/\\\\/g, '\\');
        
        var data = {
            action: 'save_titles',
            titles: cleanTitles,
            _ajax_nonce: myAutoBlog.nonce
        };

        $.ajax({
            url: myAutoBlog.ajax_url,
            type: 'POST',
            data: data,
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    // Titles saved successfully
                } else {
                    // Error saving titles
                }
            },
            error: function(jqXHR, textStatus, errorThrown) {
                if (window.SICA_DEBUG) console.error('AJAX error while saving titles:', textStatus, errorThrown);
            }
        });
    }

    // AJAX handler for generating blog titles
    $('#generate_titles_button').click(function() {
        // Title generation progress UI (lightweight, reusable)
        function ensureTitleProgressUI() {
            const existing = $('#titleProgress');
            const html = `
                <div id="titleProgress" style="display:none; margin:10px 0 0; padding:12px; border:1px solid #e5e7eb; border-radius:8px; background:#f9fafb;">
                    <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:8px;">
                        <div style="font-weight:600; color:#374151;">Generating Titles</div>
                        <div id="titleProgressPct" style="font-weight:600; color:#2563eb; background:#e0f2fe; padding:2px 8px; border-radius:999px;">0%</div>
                    </div>
                    <div style="height:6px; background:#e5e7eb; border-radius:999px; overflow:hidden; margin-bottom:10px;">
                        <div id="titleProgressBar" style="height:6px; width:0%; background:linear-gradient(90deg,#2563eb,#06b6d4);"></div>
                    </div>
                    <div id="titleProgressStep" style="font-size:13px; color:#4b5563;">Starting…</div>
                </div>`;
            const btn = $('#generate_titles_button');
            if (existing.length) {
                existing.detach();
                btn.after(existing);
            } else if (btn.length) {
                btn.after(html);
            } else {
                $('#content-planner-form').prepend(html);
            }
        }
        function showTitleProgress() { $('#titleProgress').show(); }
        function hideTitleProgress() { $('#titleProgress').hide(); }
        function setTitleProgress(pct, msg) {
            $('#titleProgressPct').text(Math.round(pct) + '%');
            $('#titleProgressBar').css('width', pct + '%');
            if (msg) { $('#titleProgressStep').text(msg); }
        }

        // Smooth staged simulation
        let tpInterval = null;
        let tpPct = 0;
        let tpStageIdx = 0;
        const tpStages = [
            { name: 'analyze', cap: 20, msg: 'Analyzing business context…' },
            { name: 'research', cap: 60, msg: 'Researching local trends…' },
            { name: 'generate', cap: 85, msg: 'Generating candidate titles…' },
            { name: 'finalize', cap: 98, msg: 'De-duplicating and finalizing…' }
        ];
        function startTitleProgressSim() {
            tpPct = 0; tpStageIdx = 0;
            setTitleProgress(0, tpStages[0].msg);
            tpInterval = setInterval(() => {
                const cap = tpStages[Math.min(tpStageIdx, tpStages.length - 1)].cap;
                const increment = tpPct < 10 ? 1.8 : tpPct < 40 ? 1.2 : tpPct < 70 ? 0.8 : 0.4;
                tpPct = Math.min(tpPct + increment, cap);
                setTitleProgress(tpPct);
                if (tpPct >= cap && tpStageIdx < tpStages.length - 1) {
                    tpStageIdx += 1;
                    setTitleProgress(tpPct, tpStages[tpStageIdx].msg);
                }
            }, 350);
        }
        function stopTitleProgressSim() {
            if (tpInterval) { try { clearInterval(tpInterval); } catch (e) {} tpInterval = null; }
        }

        ensureTitleProgressUI();
        showTitleProgress();
        startTitleProgressSim();

        var keywords = $('#my_auto_blog_keywords').val();
        var currentTitles = $('#my_auto_blog_title').val();
        var businessCategory = $('#business_category').val();
        var serviceArea = $('#business_service_area').val();
        var productsServices = $('#products_services').val();

        // Starting title generation

        // Disable the button to prevent multiple clicks
        $(this).prop('disabled', true).text('Generating...');

        // First, quick-save current planner values to ensure server has the latest edits
        $.ajax({
            url: myAutoBlog.ajax_url,
            type: 'POST',
            data: {
                action: 'save_content_planner_quick',
                my_auto_blog_keywords: keywords,
                my_auto_blog_title: currentTitles,
                business_category: businessCategory,
                business_service_area: serviceArea,
                products_services: productsServices,
                _ajax_nonce: myAutoBlog.nonce
            },
            dataType: 'json'
        }).always(function() {
            // Now proceed with title generation using the saved values
            var data = {
                action: 'generate_blog_titles',
                keywords: keywords,
                current_titles: currentTitles,
                business_category: businessCategory,
                business_service_area: serviceArea,
                products_services: productsServices,
                _ajax_nonce: myAutoBlog.nonce,
                bypass_cache: 1 // force fresh trends when user actively requests more
            };

        // Sending AJAX request for title generation

        $.ajax({
            url: myAutoBlog.ajax_url,
            type: 'POST',
            data: data,
            dataType: 'json',
            timeout: 180000 // up to 3 minutes for two-step title generation
        })
        .done(function(response) {
            // Move progress to completion smoothly
            // Processing AJAX response
            
            if (response.success) {
                setTitleProgress(99, 'Wrapping up…');
                var titles = response.data.titles;
                
                // Backend already merged existing + new titles and deduplicated them
                // Just display the final result directly
                var newValue = titles.join('\n');
                
                // Update textarea and show animation
                var textarea = $('#my_auto_blog_title');
                textarea
                    .val(newValue)
                    // Set initial state with no transition
                    .css({
                        'background-color': '#90EE90',
                        'transition': 'none'
                    })
                    // Force reflow to ensure the green shows
                    .outerHeight();
                    
                // Add transition and trigger fade
                setTimeout(function() {
                    textarea.css({
                        'background-color': '',
                        'transition': 'background-color 0.75s ease-out'
                    });
                }, 500); // Keep green for 500ms before starting fade

                // Auto-save the updated titles
                saveTitles(newValue);
                
                // Titles updated successfully
                stopTitleProgressSim();
                setTitleProgress(100, 'Done');
                setTimeout(hideTitleProgress, 800);
            } else {
                // Server returned error
                alert('Error generating titles: ' + (response.data?.message || 'Unknown error'));
                hideTitleProgress();
                stopTitleProgressSim();
            }
        })
        .fail(function(jqXHR, textStatus, errorThrown) {
            hideTitleProgress();
            stopTitleProgressSim();
            if (window.SICA_DEBUG) console.error('AJAX failure details:', {
                status: jqXHR.status,
                statusText: jqXHR.statusText,
                responseText: jqXHR.responseText,
                textStatus: textStatus,
                errorThrown: errorThrown
            });
            alert('Failed to generate titles. Please check the console for details.');
        })
        .always(function() {
            $('#generate_titles_button').prop('disabled', false).text('Generate more titles using keywords...');
        });
        }); // Close the save_content_planner_quick .always() function
    });

    // Removed single post generation handlers per request

    // Reset formChanged flag when Save button is clicked
    $('input[type="submit"]').click(function() {
        formChanged = false;
    });

    // ========================================
    // EMAIL VERIFICATION HANDLERS (Setup Tab)
    // ========================================
    
    // Send verification code
    $('#send_verification_btn').on('click', function(e) {
        e.preventDefault();
        const button = $(this);
        const email = $('#verification_email').val().trim();
        const marketingConsent = $('#marketing-consent').is(':checked');
        
        if (!email) {
            alert('Please enter your email address');
            return;
        }
        
        if (!isValidEmail(email)) {
            alert('Please enter a valid email address');
            return;
        }
        
        button.prop('disabled', true).text('Sending...');
        
        $.ajax({
            url: myAutoBlog.ajax_url,
            type: 'POST',
            data: {
                action: 'send_verification',
                email: email,
                marketing_consent: marketingConsent,
                _ajax_nonce: myAutoBlog.nonce
            },
            success: function(response) {
                if (response.success) {
                    $('#verification_email_display').text(email);
                    $('#verification_step_1').hide();
                    $('#verification_step_2').show();
                    showVerificationMessage('✅ ' + response.data.message, 'success');
                } else {
                    showVerificationMessage('❌ ' + (response.data || 'Failed to send verification code'), 'error');
                }
            },
            error: function() {
                showVerificationMessage('❌ Network error. Please try again.', 'error');
            },
            complete: function() {
                button.prop('disabled', false).text('📨 Send Verification Code');
            }
        });
    });
    
    // Auto-clear verification code field when clicked
    $('#verification_code').on('focus', function() {
        if ($(this).val() === '') {
            $(this).removeClass('error');
        }
    });
    
    // Only allow numbers in verification code
    $('#verification_code').on('input', function() {
        let value = $(this).val().replace(/[^0-9]/g, '');
        if (value.length > 6) {
            value = value.slice(0, 6);
        }
        $(this).val(value);
        
        // Remove error styling when user starts typing
        $(this).css({
            'border-color': '#ddd',
            'box-shadow': 'none'
        });
    });
    
    // Verify email code
    $('#verify_email_btn').on('click', function(e) {
        e.preventDefault();
        const button = $(this);
        const code = $('#verification_code').val().trim();
        
        if (!code || code.length !== 6) {
            $('#verification_code').addClass('error').focus();
            showVerificationMessage('❌ Please enter the complete 6-digit verification code', 'error');
            return;
        }
        
        if (!/^[0-9]{6}$/.test(code)) {
            $('#verification_code').addClass('error').focus();
            showVerificationMessage('❌ Verification code must be 6 digits', 'error');
            return;
        }
        
        button.prop('disabled', true).text('✅ Verifying...');
        
        $.ajax({
            url: myAutoBlog.ajax_url,
            type: 'POST',
            data: {
                action: 'verify_email',
                code: code,
                _ajax_nonce: myAutoBlog.nonce
            },
            success: function(response) {
                if (response.success) {
                    showVerificationMessage('🎉 ' + response.data.message, 'success');
                    // Reload page after 2 seconds to show subscription dashboard
                    setTimeout(function() {
                        window.location.reload();
                    }, 2000);
                } else {
                    showVerificationMessage('❌ ' + (response.data || 'Invalid verification code'), 'error');
                }
            },
            error: function() {
                showVerificationMessage('❌ Network error. Please try again.', 'error');
            },
            complete: function() {
                button.prop('disabled', false).text('✅ Verify & Activate Free Account');
            }
        });
    });
    
    // Change email address
    $('#change_email_link').on('click', function(e) {
        e.preventDefault();
        $('#verification_step_2').hide();
        $('#verification_step_1').show();
        $('#verification_code').val('');
        $('#verification_status').empty();
    });
    
    // Resend verification code
    $('#resend_code_btn').on('click', function(e) {
        e.preventDefault();
        const button = $(this);
        const email = $('#verification_email_display').text();
        
        button.prop('disabled', true).text('🔄 Sending...');
        
        $.ajax({
            url: myAutoBlog.ajax_url,
            type: 'POST',
            data: {
                action: 'send_verification',
                email: email,
                _ajax_nonce: myAutoBlog.nonce
            },
            success: function(response) {
                if (response.success) {
                    showVerificationMessage('✅ New verification code sent!', 'success');
                } else {
                    showVerificationMessage('❌ ' + (response.data || 'Failed to resend code'), 'error');
                }
            },
            error: function() {
                showVerificationMessage('❌ Network error. Please try again.', 'error');
            },
            complete: function() {
                button.prop('disabled', false).text('🔄 Resend Code');
            }
        });
    });
    
    // Helper functions
    function isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
    
    function showVerificationMessage(message, type) {
        const statusDiv = $('#verification_status');
        const bgColor = type === 'success' ? '#d4edda' : '#f8d7da';
        const textColor = type === 'success' ? '#155724' : '#721c24';
        const borderColor = type === 'success' ? '#c3e6cb' : '#f5c6cb';
        
        // Add error styling to input if verification failed
        if (type === 'error' && message.includes('verification code')) {
            $('#verification_code').css({
                'border-color': '#dc3545',
                'box-shadow': '0 0 0 0.2rem rgba(220, 53, 69, 0.25)'
            });
        }
        
        statusDiv.html(
            '<div style="background: ' + bgColor + '; color: ' + textColor + '; border: 1px solid ' + borderColor + '; padding: 12px; border-radius: 4px; margin-top: 10px;">' +
            message +
            '</div>'
        );
        
        // Auto-hide success messages
        if (type === 'success') {
            setTimeout(function() {
                statusDiv.fadeOut();
            }, 5000);
        }
    }
    
    // Email logout/change function
    window.sicaLogoutEmail = function() {
        if (confirm('Are you sure you want to logout? You\'ll need to verify your email again to access your account.')) {
            // Show loading state
            var button = event.target;
            var originalText = button.innerHTML;
            button.innerHTML = '⏳ Logging out...';
            button.disabled = true;
            
            // Send logout request
            $.ajax({
                url: myAutoBlog.ajax_url,
                type: 'POST',
                data: {
                    action: 'sica_logout_email',
                    _ajax_nonce: myAutoBlog.nonce
                },
                success: function(response) {
                    if (response.success) {
                        // Show success message and reload page
                        alert('✅ Logged out successfully! You can now verify a different email address.');
                        window.location.reload();
                    } else {
                        alert('❌ Logout failed: ' + (response.data || 'Unknown error'));
                        button.innerHTML = originalText;
                        button.disabled = false;
                    }
                },
                error: function() {
                    alert('❌ Network error occurred during logout');
                    button.innerHTML = originalText;
                    button.disabled = false;
                }
            });
        }
    };
});
