<?php
/**
 * Sica Content AI - Diagnostic Script
 * Upload to WordPress root, run once, then DELETE immediately
 * URL: https://your-site.com/sica-diagnostic.php
 */

// Load WordPress
require_once dirname(__FILE__) . '/wp-load.php';

// Security check - only admins
if (!current_user_can('manage_options')) {
    die('Access denied. Please log in as admin first.');
}

echo "<pre style='font-family: monospace; background: #1a1a1a; color: #0f0; padding: 20px; font-size: 12px;'>";
echo "=== SICA CONTENT AI DIAGNOSTIC ===\n";
echo "Time: " . date('Y-m-d H:i:s') . "\n";
echo "Site: " . get_site_url() . "\n\n";

// 1. Check plugin options
echo "=== PLUGIN OPTIONS ===\n";
$api_token = get_option('sica_api_token', '');
$customer_email = get_option('sica_customer_email', '');
$email_verified = get_option('sica_email_verified', false);
$license_key = get_option('sica_license_key', '');

echo "API Token: " . (!empty($api_token) ? "Present (length=" . strlen($api_token) . ")" : "MISSING") . "\n";
echo "Customer Email: " . (!empty($customer_email) ? $customer_email : "MISSING") . "\n";
echo "Email Verified: " . ($email_verified ? "Yes" : "No") . "\n";
echo "License Key: " . (!empty($license_key) ? "Present (starts with " . substr($license_key, 0, 8) . "...)" : "Not set") . "\n\n";

// 2. Test subscription.php
echo "=== TEST: subscription.php ===\n";
$start = microtime(true);
$response = wp_remote_post('https://api.sica.ai/subscription.php', array(
    'headers' => array(
        'Content-Type' => 'application/json',
        'X-Sica-Token' => $api_token
    ),
    'body' => json_encode(array(
        'action' => 'check_usage',
        'site_url' => get_site_url(),
        'customer_email' => $customer_email
    )),
    'timeout' => 15
));
$duration = round(microtime(true) - $start, 2);

if (is_wp_error($response)) {
    echo "ERROR: " . $response->get_error_message() . "\n";
} else {
    $code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    echo "HTTP Code: $code\n";
    echo "Duration: {$duration}s\n";
    echo "Response: " . substr($body, 0, 500) . "\n";
}
echo "\n";

// 3. Test generate.php (quick validation only, not full generation)
echo "=== TEST: generate.php ===\n";
$start = microtime(true);
$response = wp_remote_post('https://api.sica.ai/generate.php', array(
    'headers' => array(
        'Content-Type' => 'application/json',
        'X-Sica-Token' => $api_token
    ),
    'body' => json_encode(array(
        'site_url' => get_site_url(),
        'customer_email' => $customer_email,
        'content_type' => 'text',
        'model' => 'gpt-4o-search-preview',
        'messages' => array(
            array('role' => 'system', 'content' => 'Test'),
            array('role' => 'user', 'content' => 'Say "diagnostic test passed" in 3 words')
        ),
        'max_tokens' => 50,
        'title' => 'Diagnostic Test'
    )),
    'timeout' => 60  // Short timeout for diagnostic
));
$duration = round(microtime(true) - $start, 2);

if (is_wp_error($response)) {
    echo "ERROR: " . $response->get_error_message() . "\n";
    echo "This is likely the issue! wp_remote_post failed.\n";
} else {
    $code = wp_remote_retrieve_response_code($response);
    $body = wp_remote_retrieve_body($response);
    echo "HTTP Code: $code\n";
    echo "Duration: {$duration}s\n";
    echo "Response: " . substr($body, 0, 500) . "\n";
}
echo "\n";

// 4. Check PHP configuration
echo "=== PHP CONFIGURATION ===\n";
echo "PHP Version: " . phpversion() . "\n";
echo "max_execution_time: " . ini_get('max_execution_time') . "s\n";
echo "memory_limit: " . ini_get('memory_limit') . "\n";
echo "allow_url_fopen: " . (ini_get('allow_url_fopen') ? 'Yes' : 'No') . "\n";
echo "cURL enabled: " . (function_exists('curl_version') ? 'Yes' : 'No') . "\n";
if (function_exists('curl_version')) {
    $cv = curl_version();
    echo "cURL version: " . $cv['version'] . "\n";
    echo "SSL version: " . $cv['ssl_version'] . "\n";
}
echo "\n";

// 5. Check WP HTTP
echo "=== WORDPRESS HTTP ===\n";
echo "WP_DEBUG: " . (defined('WP_DEBUG') && WP_DEBUG ? 'Yes' : 'No') . "\n";
echo "WP_DEBUG_LOG: " . (defined('WP_DEBUG_LOG') && WP_DEBUG_LOG ? 'Yes' : 'No') . "\n";
$debug_log = WP_CONTENT_DIR . '/debug.log';
echo "debug.log exists: " . (file_exists($debug_log) ? 'Yes (' . size_format(filesize($debug_log)) . ')' : 'No') . "\n";
echo "\n";

// 6. Check Activity Log
echo "=== RECENT ACTIVITY LOG ===\n";
$logs = get_option('sica_scheduler_log', array());
if (empty($logs)) {
    echo "No activity logs found.\n";
} else {
    $recent = array_slice($logs, -5);
    foreach ($recent as $log) {
        $time = isset($log['timestamp']) ? date('H:i:s', strtotime($log['timestamp'])) : '??:??:??';
        $status = $log['status'] ?? 'unknown';
        $msg = $log['message'] ?? '';
        $source = $log['source'] ?? '';
        echo "[$time] [$source] [$status] $msg\n";
    }
}
echo "\n";

echo "=== DIAGNOSTIC COMPLETE ===\n";
echo "DELETE THIS FILE IMMEDIATELY: sica-diagnostic.php\n";
echo "</pre>";
?>
