<?php
/*
 * AI Post Enhancer
 * Adds AI content enhancement tools directly to WordPress post editor
 * Part of AI Blogger Plugin
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class AI_Post_Enhancer {
    
    public function __construct() {
        // Register AJAX handlers globally (required for AJAX to work)
        add_action('wp_ajax_ai_enhance_content', array($this, 'handle_content_enhancement'));
        add_action('wp_ajax_ai_generate_featured_image', array($this, 'handle_featured_image_generation'));
        
        // Initialize UI components only on post edit screens
        add_action('admin_init', array($this, 'init'));
        
        // Add hooks for enhanced SEO elements output
        add_action('wp_head', array($this, 'output_enhanced_schema'));
        add_action('wp_head', array($this, 'output_enhanced_meta_description'));
    }
    
    public function init() {
        // Only load UI components on post edit screens
        global $pagenow;
        if (!in_array($pagenow, array('post.php', 'post-new.php'))) {
            return;
        }
        
        // Add hooks for editor UI
        add_action('media_buttons', array($this, 'add_editor_button'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_footer', array($this, 'output_inline_js'));
    }
    
    /**
     * Add Content AI Tools button to editor toolbar (Classic Editor)
     */
    public function add_editor_button() {
        global $post;
        if (!$post || !current_user_can('edit_posts')) {
            return;
        }
        
        echo '<div class="ai-blogger-tools-wrapper" style="display: inline-flex; align-items: center; margin-left: 10px; position: relative;">';
        echo '<button type="button" class="button ai-blogger-tools-btn" data-post-id="' . $post->ID . '" style="display: inline-flex; align-items: center; gap: 6px;">';
        echo '<span class="dashicons dashicons-admin-generic" style="font-size: 16px; width: 16px; height: 16px; line-height: 1;"></span>';
        echo '<span style="line-height: 1;">Content AI Tools</span>';
        echo '<span class="dashicons dashicons-arrow-down-alt2" style="font-size: 14px; width: 14px; height: 14px; line-height: 1; margin-left: 2px;"></span>';
        echo '</button>';
        
        // Success message (hidden by default)
        echo '<div class="ai-enhancer-success" style="display: none; margin-left: 15px; color: #46b450; font-weight: 500; font-size: 13px;">';
        echo '<span class="dashicons dashicons-yes-alt" style="margin-right: 5px; font-size: 16px;"></span>';
        echo '<span class="success-text">Content enhanced, proofread, and expanded successfully!</span>';
        echo '</div>';
        
        // Check subscription status for access control
        $has_paid_access = $this->check_paid_access();
        
        // Dropdown menu
        echo '<div class="ai-blogger-dropdown" style="display: none; position: absolute; top: 100%; left: 0; background: white; border: 1px solid #ccc; border-radius: 4px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); z-index: 1000; min-width: 240px; white-space: nowrap;">';
        echo '<ul style="margin: 0; padding: 5px 0; list-style: none;">';
        
        if ($has_paid_access) {
            // Paid users get full access
            echo '<li><a href="#" class="ai-enhance-seo" style="display: block; padding: 8px 15px; text-decoration: none; color: #333;"><span class="dashicons dashicons-chart-line" style="margin-right: 8px;"></span>SEO Enhance Article</a></li>';
            echo '<li><a href="#" class="ai-enhance-content" style="display: block; padding: 8px 15px; text-decoration: none; color: #333;"><span class="dashicons dashicons-edit" style="margin-right: 8px;"></span>Expand Content Only</a></li>';
            echo '<li><a href="#" class="ai-generate-image" style="display: block; padding: 8px 15px; text-decoration: none; color: #333;"><span class="dashicons dashicons-format-image" style="margin-right: 8px;"></span>Generate New Featured Image</a></li>';
        } else {
            // Free users see disabled options and upgrade prompt
            echo '<li><span style="display: block; padding: 8px 15px; color: #999; cursor: not-allowed;"><span class="dashicons dashicons-chart-line" style="margin-right: 8px;"></span>SEO Enhance Article <span style="font-size: 11px;">(Pro)</span></span></li>';
            echo '<li><span style="display: block; padding: 8px 15px; color: #999; cursor: not-allowed;"><span class="dashicons dashicons-edit" style="margin-right: 8px;"></span>Expand Content Only <span style="font-size: 11px;">(Pro)</span></span></li>';
            echo '<li><span style="display: block; padding: 8px 15px; color: #999; cursor: not-allowed;"><span class="dashicons dashicons-format-image" style="margin-right: 8px;"></span>Generate New Featured Image <span style="font-size: 11px;">(Pro)</span></span></li>';
            echo '<li style="border-top: 1px solid #eee; margin-top: 5px; padding-top: 5px;"><a href="https://sica.ai/pricing" target="_blank" style="display: block; padding: 8px 15px; text-decoration: none; color: #0073aa; font-weight: 500;"><span class="dashicons dashicons-star-filled" style="margin-right: 8px;"></span>Upgrade to Access Tools</a></li>';
        }
        
        echo '</ul>';
        echo '</div>';
        echo '</div>';
    }
    
    /**
     * Enqueue scripts for Classic Editor
     */
    public function enqueue_scripts($hook) {
        if (!in_array($hook, array('post.php', 'post-new.php'))) {
            return;
        }
        
        // Ensure jQuery is loaded
        wp_enqueue_script('jquery');
        
        // Add CSS directly to head
        add_action('admin_head', array($this, 'output_admin_css'));
    }
    
    /**
     * Output inline JavaScript to admin footer
     */
    public function output_inline_js() {
        $nonce = wp_create_nonce('ai_enhancer_nonce');
        $ajax_url = admin_url('admin-ajax.php');
        $has_paid_access = $this->check_paid_access() ? 'true' : 'false';
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            var isDropdownOpen = false;
            var isProcessing = false;
            var hasPaidAccess = <?php echo $has_paid_access; ?>;
            
            // Toggle dropdown menu
            $(document).on('click', '.ai-blogger-tools-btn', function(e) {
                e.preventDefault();
                e.stopPropagation();
                
                if (isProcessing) return;
                
                var dropdown = $(this).siblings('.ai-blogger-dropdown');
                
                if (isDropdownOpen) {
                    dropdown.hide();
                    isDropdownOpen = false;
                } else {
                    $('.ai-blogger-dropdown').hide();
                    dropdown.show();
                    isDropdownOpen = true;
                }
            });
            
            // Close dropdown when clicking outside
            $(document).on('click', function(e) {
                if (!$(e.target).closest('.ai-blogger-tools-wrapper').length) {
                    $('.ai-blogger-dropdown').hide();
                    isDropdownOpen = false;
                }
            });
            
            // Handle enhancement actions
            $(document).on('click', '.ai-enhance-seo', function(e) {
                e.preventDefault();
                handleEnhancement('seo_enhance');
            });
            
            $(document).on('click', '.ai-enhance-content', function(e) {
                e.preventDefault();
                handleEnhancement('enrich');
            });
            
            // Handle image generation
            $(document).on('click', '.ai-generate-image', function(e) {
                e.preventDefault();
                handleImageGeneration();
            });
            
            function handleEnhancement(type) {
                if (isProcessing) return;
                
                // Check if user has paid access
                if (!hasPaidAccess) {
                    alert('Content AI Tools require a paid subscription. Please upgrade your plan to access these features.');
                    window.open('https://sica.ai/pricing', '_blank');
                    return;
                }
                
                $('.ai-blogger-dropdown').hide();
                isDropdownOpen = false;
                
                var content = getCurrentEditorContent();
                if (!content || content.trim() === '') {
                    alert('Please add some content to your post before enhancing it.');
                    return;
                }
                
                var postId = $('.ai-blogger-tools-btn').data('post-id');
                if (!postId) {
                    alert('Unable to determine post ID. Please save your post first.');
                    return;
                }
                
                showLoadingState(type);
                
                $.ajax({
                    url: '<?php echo $ajax_url; ?>',
                    type: 'POST',
                    data: {
                        action: 'ai_enhance_content',
                        nonce: '<?php echo $nonce; ?>',
                        post_id: postId,
                        content: content,
                        enhancement_type: type
                    },
                    timeout: 180000,
                    success: function(response) {
                        hideLoadingState();
                        
                        if (response.success) {
                            setEditorContent(response.data.content);
                            showSuccessMessage('Content enhanced, proofread, and expanded successfully!');
                        } else {
                            alert('Error: ' + (response.data || 'Unknown error occurred'));
                        }
                    },
                    error: function(xhr, status, error) {
                        hideLoadingState();
                        if (status === 'timeout') {
                            alert('The enhancement is taking longer than expected. Please try again with shorter content.');
                        } else {
                            alert('Error enhancing content. Please try again.');
                        }
                    }
                });
            }
            
            function handleImageGeneration() {
                if (isProcessing) return;
                
                // Check if user has paid access
                if (!hasPaidAccess) {
                    alert('Content AI Tools require a paid subscription. Please upgrade your plan to access these features.');
                    window.open('https://sica.ai/pricing', '_blank');
                    return;
                }
                
                $('.ai-blogger-dropdown').hide();
                isDropdownOpen = false;
                
                var postId = $('.ai-blogger-tools-btn').data('post-id');
                if (!postId) {
                    alert('Unable to determine post ID. Please save your post first.');
                    return;
                }
                
                // Get the post title for image generation context
                var postTitle = $('#title').val() || document.title || 'Blog Post';
                
                showLoadingState('image');
                
                $.ajax({
                    url: '<?php echo $ajax_url; ?>',
                    type: 'POST',
                    data: {
                        action: 'ai_generate_featured_image',
                        nonce: '<?php echo $nonce; ?>',
                        post_id: postId,
                        post_title: postTitle
                    },
                    timeout: 300000, // 5 minutes for image generation
                    success: function(response) {
                        hideLoadingState();
                        
                        if (response.success) {
                            showSuccessMessage('Featured image generated successfully!');
                            // Refresh the featured image area if visible
                            if ($('#postimagediv').length) {
                                location.reload(); // Simple reload to show new featured image
                            }
                        } else {
                            alert('Error: ' + (response.data || 'Failed to generate image'));
                        }
                    },
                    error: function(xhr, status, error) {
                        hideLoadingState();
                        if (status === 'timeout') {
                            alert('Image generation is taking longer than expected. Please check back in a few minutes.');
                        } else {
                            alert('Error generating image. Please try again.');
                        }
                    }
                });
            }
            
            function getCurrentEditorContent() {
                // Getting current editor content
                
                // Try TinyMCE first (Visual editor)
                if (typeof tinymce !== 'undefined' && tinymce.get('content') && tinymce.get('content').initialized) {
                    // Getting content from TinyMCE
                    return tinymce.get('content').getContent();
                } 
                // Fallback to textarea (Text editor)
                else if ($('#content').length) {
                    // Getting content from textarea
                    return $('#content').val();
                }
                // Try alternative TinyMCE approach
                else if (typeof tinyMCE !== 'undefined') {
                    // Trying alternative TinyMCE approach
                    var ed = tinyMCE.get('content');
                    if (ed && ed.getContent) {
                        return ed.getContent();
                    }
                }
                
                // No editor found, returning empty content
                return '';
            }
            
            function setEditorContent(content) {
                // Setting editor content
                
                // Try TinyMCE first (Visual editor)
                if (typeof tinymce !== 'undefined' && tinymce.get('content') && tinymce.get('content').initialized) {
                    // Using TinyMCE to set content
                    tinymce.get('content').setContent(content);
                    tinymce.get('content').save(); // Save to textarea
                } 
                // Fallback to textarea (Text editor)
                else if ($('#content').length) {
                    // Using textarea to set content
                    $('#content').val(content);
                }
                // Try alternative TinyMCE approach
                else if (typeof tinyMCE !== 'undefined') {
                    // Using alternative TinyMCE approach
                    var ed = tinyMCE.get('content');
                    if (ed && ed.setContent) {
                        ed.setContent(content);
                        ed.save();
                    }
                }
                
                // Trigger change events to ensure WordPress knows content has changed
                $('#content').trigger('change');
                
                // Content set successfully
            }
            
            function showSuccessMessage(message) {
                var $success = $('.ai-enhancer-success');
                $success.find('.success-text').text(message);
                $success.fadeIn();
                
                // Hide success message after 4 seconds
                setTimeout(function() {
                    $success.fadeOut();
                }, 4000);
            }
            
            function showLoadingState(type) {
                isProcessing = true;
                var loadingText = type === 'image' ? 'Generating Image...' : 'Enriching...';
                $('.ai-blogger-tools-btn')
                    .prop('disabled', true)
                    .html('<span class="dashicons dashicons-update-alt" style="font-size: 16px; width: 16px; height: 16px; line-height: 1; animation: ai-spin 1s linear infinite;"></span><span style="line-height: 1;">' + loadingText + '</span>');
            }
            
            function hideLoadingState() {
                isProcessing = false;
                $('.ai-blogger-tools-btn')
                    .prop('disabled', false)
                    .html('<span class="dashicons dashicons-admin-generic" style="font-size: 16px; width: 16px; height: 16px; line-height: 1;"></span><span style="line-height: 1;">Content AI Tools</span><span class="dashicons dashicons-arrow-down-alt2" style="font-size: 14px; width: 14px; height: 14px; line-height: 1; margin-left: 2px;"></span>');
            }
        });
        </script>
        <?php
    }
    
    /**
     * Handle AJAX request for content enhancement
     */
    public function handle_content_enhancement() {
        error_log('AI Post Enhancer: Starting enhancement request');
        // POST data logging disabled to reduce log clutter
        
        // Security checks
        if (!check_ajax_referer('ai_enhancer_nonce', 'nonce', false)) {
            error_log('AI Post Enhancer: Nonce verification failed');
            wp_send_json_error('Security check failed');
        }
        
        if (!current_user_can('edit_posts')) {
            error_log('AI Post Enhancer: User lacks edit_posts capability');
            wp_send_json_error('Insufficient permissions');
        }
        
        // Check subscription access
        if (!$this->check_paid_access()) {
            error_log('AI Post Enhancer: User does not have paid access to Content AI Tools');
            wp_send_json_error('Content AI Tools require a paid subscription. Please upgrade your plan to access these features.');
        }
        
        $post_id = intval($_POST['post_id']);
        $content = wp_kses_post($_POST['content']);
        $enhancement_type = sanitize_text_field($_POST['enhancement_type']);
        
        error_log('AI Post Enhancer: Post ID: ' . $post_id);
        error_log('AI Post Enhancer: Enhancement type: ' . $enhancement_type);
        error_log('AI Post Enhancer: Content length: ' . strlen($content));
        error_log('AI Post Enhancer: Content preview: ' . substr($content, 0, 200) . '...');
        
        if (empty($content)) {
            error_log('AI Post Enhancer: No content provided');
            wp_send_json_error('No content provided');
        }
        
        try {
            $enhanced_content = false;
            
            switch ($enhancement_type) {
                case 'seo_enhance':
                    error_log('AI Post Enhancer: Starting SEO enhancement');
                    $enhanced_content = $this->seo_enhance_article($content, $post_id);
                    break;
                case 'enrich':
                    error_log('AI Post Enhancer: Starting content enrichment');
                    $enhanced_content = $this->enrich_article_content($content, $post_id);
                    break;
                case 'faq':
                    error_log('AI Post Enhancer: Starting FAQ addition');
                    $enhanced_content = $this->add_faq_section($content, $post_id);
                    break;
                case 'seo':
                    error_log('AI Post Enhancer: Starting SEO improvement');
                    $enhanced_content = $this->improve_seo_content($content, $post_id);
                    break;
                default:
                    error_log('AI Post Enhancer: Invalid enhancement type: ' . $enhancement_type);
                    wp_send_json_error('Invalid enhancement type');
            }
            
            if ($enhanced_content) {
                error_log('AI Post Enhancer: Enhancement successful, returning ' . strlen($enhanced_content) . ' characters');
                wp_send_json_success(array(
                    'content' => $enhanced_content,
                    'message' => 'Content enhanced successfully!'
                ));
            } else {
                error_log('AI Post Enhancer: Enhancement failed - no enhanced content returned');
                wp_send_json_error('Failed to enhance content - no content returned');
            }
            
        } catch (Exception $e) {
            error_log('AI Post Enhancer Exception: ' . $e->getMessage());
            error_log('AI Post Enhancer Stack Trace: ' . $e->getTraceAsString());
            wp_send_json_error('Enhancement failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Handle AJAX request for featured image generation
     */
    public function handle_featured_image_generation() {
        error_log('AI Post Enhancer: Starting image generation request');
        // POST data logging disabled to reduce log clutter
        
        // Security checks
        if (!check_ajax_referer('ai_enhancer_nonce', 'nonce', false)) {
            error_log('AI Post Enhancer: Nonce verification failed');
            wp_send_json_error('Security check failed');
        }
        
        if (!current_user_can('edit_posts')) {
            error_log('AI Post Enhancer: User lacks edit_posts capability');
            wp_send_json_error('Insufficient permissions');
        }
        
        // Check subscription access
        if (!$this->check_paid_access()) {
            error_log('AI Post Enhancer: User does not have paid access to Content AI Tools');
            wp_send_json_error('Content AI Tools require a paid subscription. Please upgrade your plan to access these features.');
        }
        
        $post_id = intval($_POST['post_id']);
        $post_title = sanitize_text_field($_POST['post_title']);
        
        error_log('AI Post Enhancer: Post ID: ' . $post_id);
        error_log('AI Post Enhancer: Post title: ' . $post_title);
        
        if (empty($post_title)) {
            error_log('AI Post Enhancer: No post title provided');
            wp_send_json_error('No post title provided');
        }
        
        try {
            // Generate image using the same function as blog creation
            $image_url = generate_image_with_dalle($post_title);
            
            if (!$image_url) {
                throw new Exception('Failed to generate image with DALL-E');
            }
            
            error_log('AI Post Enhancer: Image generated successfully: ' . $image_url);
            
            // Attach image using WordPress media functions
            if (!function_exists('media_sideload_image')) {
                require_once(ABSPATH . 'wp-admin/includes/media.php');
                require_once(ABSPATH . 'wp-admin/includes/file.php');
                require_once(ABSPATH . 'wp-admin/includes/image.php');
            }
            
            $image_id = media_sideload_image($image_url, $post_id, $post_title, 'id');
            if (!is_wp_error($image_id)) {
                set_post_thumbnail($post_id, $image_id);
            } else {
                throw new Exception('Failed to attach image: ' . $image_id->get_error_message());
            }
            
            if ($image_id) {
                error_log('AI Post Enhancer: Image attached successfully with ID: ' . $image_id);
                wp_send_json_success(array(
                    'image_id' => $image_id,
                    'image_url' => wp_get_attachment_url($image_id),
                    'message' => 'Featured image generated and set successfully!'
                ));
            } else {
                throw new Exception('Failed to attach generated image');
            }
            
        } catch (Exception $e) {
            error_log('AI Post Enhancer Image Generation Exception: ' . $e->getMessage());
            error_log('AI Post Enhancer Stack Trace: ' . $e->getTraceAsString());
            wp_send_json_error('Image generation failed: ' . $e->getMessage());
        }
    }
    
    /**
     * Enrich article content with AI
     */
    private function enrich_article_content($content, $post_id) {
        error_log('AI Post Enhancer: enrich_article_content() called');
        
        // API key no longer needed - all requests go through SaaS proxy
        
        // Get post context
        $post = get_post($post_id);
        $title = $post ? $post->post_title : 'Blog Post';
        error_log('AI Post Enhancer: Post title: ' . $title);
        
        // Get business context
        $business_name = get_bloginfo('name');
        $business_category = get_option('my_auto_blog_business_category', '');
        $service_area = get_option('my_auto_blog_service_area', '');
        
        error_log('AI Post Enhancer: Business context - Name: ' . $business_name . ', Category: ' . $business_category . ', Area: ' . $service_area);
        
        // Clean content for analysis
        $clean_content = wp_strip_all_tags($content);
        
        $prompt = "You are an expert content editor and fact-checker. Your task is to enhance and expand existing blog content while preserving its original structure.

**Original Article Title:** \"{$title}\"
**Business Context:** {$business_name} ({$business_category}) in {$service_area}

**Original Content:**
{$content}

**ENHANCEMENT GUIDELINES:**

**CONTENT EXPANSION ONLY:**
- PRESERVE the original title, headings, and overall structure exactly as provided
- EXPAND existing content by adding 2-3 substantial paragraphs within each section
- ADD practical examples, case studies, and actionable insights relevant to {$service_area}
- INCLUDE current industry data, statistics, and trends where appropriate
- PROVIDE expert tips and best practices for {$business_category} professionals

**ACCURACY & QUALITY:**
- Fix any spelling, grammar, or factual errors found
- Ensure all technical information is current and accurate for 2024-2025
- Use authoritative sources and industry-standard practices
- Maintain professional tone appropriate for {$business_category} audience

**STRICT REQUIREMENTS:**
- DO NOT add introductory text before the content
- DO NOT add conclusion summaries, key takeaways, or \"enhancement summaries\"
- DO NOT add contact information or calls-to-action beyond what exists
- DO NOT change the article title or main heading structure
- DO preserve all existing HTML formatting and styling
- ONLY return the enhanced article content itself

**RESEARCH INTEGRATION:**
Use your knowledge and search capabilities to:
- Verify current industry standards and best practices
- Include recent developments in {$business_category}
- Add location-specific insights for {$service_area} when relevant
- Reference authoritative sources using proper inline citations

Return ONLY the enhanced article content with improved depth and accuracy, maintaining the exact same structure and format as the original.";

        // Try to use SaaS proxy first with search capabilities
        try {
            $enhanced_content = $this->generate_content_via_saas_proxy($prompt, $title, $business_name, $business_category, $service_area);
            if (!empty($enhanced_content)) {
                error_log('AI Post Enhancer: Successfully enhanced content via SaaS proxy with search');
                return $enhanced_content;
            }
        } catch (Exception $e) {
            error_log('AI Post Enhancer: SaaS proxy failed, falling back to direct OpenAI: ' . $e->getMessage());
        }

        // If SaaS proxy failed, we cannot proceed - no direct OpenAI calls allowed
        throw new Exception('Content enhancement failed: SaaS proxy unavailable and direct OpenAI calls are not permitted');
    }
    
    /**
     * Generate enhanced content via SaaS proxy with search capabilities
     */
    private function generate_content_via_saas_proxy($prompt, $title, $business_name, $business_category, $service_area) {
        // Use the same SaaS proxy approach as the main content generator
        $site_url = get_site_url();
        $customer_email = get_option('sica_customer_email', get_option('admin_email'));
        
        $request_data = array(
            'site_url' => $site_url,
            'customer_email' => $customer_email,
            'content_type' => 'text',
            'model' => 'gpt-4o-search-preview',
            'messages' => array(
                array(
                    'role' => 'system',
                    'content' => 'You are an expert content editor and fact-checker with access to current web search. Use search to verify facts, get current data, and find relevant examples. Enhance content depth while preserving original structure. Do not add summaries or unnecessary sections.'
                ),
                array(
                    'role' => 'user',
                    'content' => $prompt
                )
            ),
            'temperature' => 0.5,
            'max_tokens' => 4000,
            'title' => $title,
        );

        // Ensure we have a valid API token before making the request
        $api_token = get_option('sica_api_token', '');
        if (empty($api_token)) {
            error_log('AI Post Enhancer: No API token found, attempting reissue...');
            $api_token = function_exists('sica_ensure_api_token') ? sica_ensure_api_token() : '';
            if (empty($api_token)) {
                throw new Exception('No valid API token available. Please check your license key in Account Settings.');
            }
        }

        $saas_endpoint = 'https://api.sica.ai/generate.php';
        
        // Log the request for debugging
        error_log('AI Post Enhancer: Making API request to: ' . $saas_endpoint);
        error_log('AI Post Enhancer: API token present: ' . (!empty($api_token) ? 'yes' : 'no'));
        error_log('AI Post Enhancer: Customer email: ' . $customer_email);
        
        $response = wp_remote_post($saas_endpoint, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-Sica-Token' => $api_token
            ),
            'body' => json_encode($request_data),
            'timeout' => 120,
        ));

        if (is_wp_error($response)) {
            throw new Exception('SaaS proxy request failed: ' . $response->get_error_message());
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        if ($response_code !== 200) {
            error_log('SaaS proxy returned error code ' . $response_code . ': ' . $response_body);
            throw new Exception('SaaS proxy error: ' . $response_code . ' - ' . $response_body);
        }

        $data = json_decode($response_body, true);
        
        if (!$data || !$data['ok']) {
            $error_msg = $data['error'] ?? 'Unknown SaaS API error';
            throw new Exception('SaaS API error: ' . $error_msg);
        }
        
        $openai_data = $data['data'];
        if (!isset($openai_data['choices'][0]['message']['content'])) {
            throw new Exception('Invalid response structure from SaaS API');
        }
        
        $content = $openai_data['choices'][0]['message']['content'];
        
        // Clean up content similar to main generator
        $content = wp_kses_post($content);
        $content = str_replace(['<body>', '</body>', '```html', '```'], '', $content);
        
        return $content;
    }
    
    /**
     * SEO enhance article with headings, structure, schema, and optimization
     */
    private function seo_enhance_article($content, $post_id) {
        error_log('AI Post Enhancer: seo_enhance_article() called');
        
        // API key no longer needed - all requests go through SaaS proxy
        
        // Get post context
        $post = get_post($post_id);
        $title = $post ? $post->post_title : 'Blog Post';
        
        // Get business context
        $business_name = get_bloginfo('name');
        $business_category = get_option('my_auto_blog_business_category', '');
        $service_area = get_option('my_auto_blog_service_area', '');
        $keywords = get_option('my_auto_blog_keywords', '');
        
        // Get existing published content for internal linking context
        $existing_posts = get_posts(array(
            'post_type' => 'post',
            'post_status' => 'publish',
            'posts_per_page' => 10,
            'post__not_in' => array($post_id),
            'orderby' => 'date',
            'order' => 'DESC'
        ));
        
        $internal_links_context = "";
        if (!empty($existing_posts)) {
            $internal_links_context = "\n\n**INTERNAL LINKING OPPORTUNITIES (use 1-2 if relevant):**\n";
            foreach ($existing_posts as $existing_post) {
                $internal_links_context .= "• \"{$existing_post->post_title}\" - " . get_permalink($existing_post->ID) . "\n";
            }
        }
        
        $keywords_list = !empty($keywords) ? implode(", ", array_filter(explode("\n", $keywords))) : "";
        
        // First, analyze the existing content structure
        $content_analysis = $this->analyze_content_structure($content);
        
        $prompt = "You are an expert SEO content strategist and technical writer. Your goal is to INTELLIGENTLY enhance this article by ONLY improving what needs improvement, while preserving all existing good structure and formatting.

**SMART ENHANCEMENT MISSION:**
Analyze and selectively improve \"{$title}\" for {$business_name} ({$business_category}) in {$service_area}.

**TARGET KEYWORDS:** {$keywords_list}

**ORIGINAL CONTENT:**
{$content}

**CONTENT ANALYSIS:**
{$content_analysis}

**INTELLIGENT ENHANCEMENT APPROACH:**

**PHASE 1: PRESERVE WHAT'S ALREADY GOOD**
- Keep all existing H2/H3 headings that are well-written and descriptive
- Preserve existing bullet points, lists, and formatting that work well
- Maintain any existing bold text, emphasis, or call-to-action elements
- Keep good existing content structure and logical flow

**PHASE 2: CREATE ENGAGING, ORGANIC CONTENT**
Based on the content analysis, enhance the article to feel natural and engaging:

**CONTENT STRUCTURE - AVOID FORMULAIC PATTERNS:**
- DON\'T create rigid heading-paragraph-heading-paragraph patterns
- DO create varied section lengths (some short, some detailed)
- DO group related ideas together naturally
- DO use conversational transitions between sections
- DO mix different content types within sections

**NATURAL WRITING STYLE:**
- Write like you're having a conversation with a friend
- Use questions to engage readers (Ever wondered why...?)
- Include relatable scenarios and examples
- Add personality and local flavor for {$service_area}
- Use 'you' to speak directly to readers

**VARIED CONTENT TYPES:**
- Mix short punchy paragraphs with longer detailed ones
- Include real-world examples and case studies
- Add quick tips alongside detailed explanations
- Use storytelling elements where appropriate
- Include comparisons and contrasts naturally

**ORGANIC HEADINGS:**
- Create headings that sound like natural conversation topics
- Vary heading lengths (some short, some descriptive)
- Use intriguing questions as headings when appropriate
- Make headings specific to {$service_area} and {$business_category}

**ENGAGING ELEMENTS:**
- Include specific local examples for {$service_area}
- Add industry insights and 'insider tips'
- Use analogies and comparisons people can relate to
- Include seasonal or timely references when relevant
- Add actionable advice people can use immediately{$internal_links_context}

**STRICT PRESERVATION RULES:**
- DO NOT change existing headings that are already good
- DO NOT reformat content that's already well-structured
- DO NOT remove existing formatting, lists, or emphasis
- DO NOT alter the existing content organization if it works well
- DO focus on ADDING value, not CHANGING what works

**ENHANCEMENT PRIORITIES:**
1. Create engaging, conversational content that people actually want to read
2. Vary content structure to avoid predictable patterns
3. Add personality and local flavor specific to {$service_area}
4. Include real examples and scenarios readers can relate to
5. Mix content types (stories, tips, lists, explanations) organically
6. Fix any spelling, grammar, or technical errors
7. Add keyword optimization naturally within engaging content
8. Include actionable advice and insider knowledge

**OUTPUT REQUIREMENT:**
Return the enhanced article in PROPER HTML FORMAT that:
- Preserves all existing good structure and formatting
- Adds improvements ONLY where the content actually needs them
- Maintains the original organization and flow
- Enhances rather than transforms the content
- Focuses on additive improvements, not wholesale changes

**CRITICAL: OUTPUT ONLY THE HTML CONTENT - NO CODE BLOCKS OR FORMATTING**
- Do NOT wrap output in ```html or any other code block syntax
- Do NOT include any markdown formatting
- Return ONLY the raw HTML content that can be directly inserted into WordPress
- Use <h2> and <h3> tags for headings (NOT ## or ###)
- Use <ul><li> for bullet lists (NOT - or *)
- Use <ol><li> for numbered lists (NOT 1. 2. 3.)
- Use <strong> for bold text (NOT **)
- Use <p> tags for paragraphs
- Use <em> for emphasis where needed
- Ensure all HTML tags are properly closed

**EXAMPLE CORRECT OUTPUT (no code blocks, just raw HTML):**
<h2>Your Main Heading Here</h2>
<p>Your paragraph content here.</p>
<h3>Your Subheading Here</h3>
<ul>
<li>First bullet point</li>
<li>Second bullet point</li>
</ul>

**IMPORTANT: Start your response immediately with the HTML content - no explanations, no code block syntax, no ```html tags**";

        // Use SaaS proxy for SEO enhancement
        $site_url = get_site_url();
        $customer_email = get_option('sica_customer_email', get_option('admin_email'));
        
        $request_data = array(
            'site_url' => $site_url,
            'customer_email' => $customer_email,
            'content_type' => 'text',
            'model' => 'gpt-4o-search-preview',
            'messages' => array(
                array(
                    'role' => 'system',
                    'content' => 'You are an expert SEO content strategist with access to current web search. Use search to find current SEO best practices, trending keywords, and industry standards. Transform plain content into comprehensive, well-structured, SEO-optimized articles that rank well while providing exceptional value. Focus on content hierarchy, keyword optimization, user experience, and local SEO principles.'
                ),
                array(
                    'role' => 'user',
                    'content' => $prompt
                )
            ),
            'temperature' => 0.7,
            'max_tokens' => 4000,
            'title' => $title,
        );

        // Ensure we have a valid API token before making the request
        $api_token = get_option('sica_api_token', '');
        if (empty($api_token)) {
            error_log('AI Post Enhancer: No API token found, attempting reissue...');
            $api_token = function_exists('sica_ensure_api_token') ? sica_ensure_api_token() : '';
            if (empty($api_token)) {
                throw new Exception('No valid API token available. Please check your license key in Account Settings.');
            }
        }

        $saas_endpoint = 'https://api.sica.ai/generate.php';
        
        // Log the request for debugging
        error_log('AI Post Enhancer: Making API request to: ' . $saas_endpoint);
        error_log('AI Post Enhancer: API token present: ' . (!empty($api_token) ? 'yes' : 'no'));
        error_log('AI Post Enhancer: Customer email: ' . $customer_email);
        
        $response = wp_remote_post($saas_endpoint, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-Sica-Token' => $api_token
            ),
            'body' => json_encode($request_data),
            'timeout' => 120,
        ));

        if (is_wp_error($response)) {
            throw new Exception('SaaS proxy request failed: ' . $response->get_error_message());
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        if ($response_code !== 200) {
            error_log('SaaS proxy returned error code ' . $response_code . ': ' . $response_body);
            
            // If we get a 401, try to reissue the token and retry once
            if ($response_code === 401) {
                $new_token = function_exists('sica_ensure_api_token') ? sica_ensure_api_token() : '';
                if (!empty($new_token) && $new_token !== $api_token) {
                    // Retry with the new token
                    $retry_response = wp_remote_post($saas_endpoint, array(
                        'headers' => array(
                            'Content-Type' => 'application/json',
                            'X-Sica-Token' => $new_token
                        ),
                        'body' => json_encode($request_data),
                        'timeout' => 120,
                    ));
                    
                    if (!is_wp_error($retry_response) && wp_remote_retrieve_response_code($retry_response) === 200) {
                        $response = $retry_response;
                        $response_code = 200;
                        $response_body = wp_remote_retrieve_body($retry_response);
                    } else {
                        throw new Exception("Authentication failed even after token refresh. Please check your license key in Account Settings.");
                    }
                } else {
                    throw new Exception("Authentication failed. Please check your license key in Account Settings.");
                }
            } else {
                throw new Exception("SaaS proxy returned error code: $response_code - $response_body");
            }
        }

        $data = json_decode($response_body, true);
        if (!$data || !$data['ok']) {
            $error_msg = $data['error'] ?? 'Unknown SaaS API error';
            throw new Exception('SaaS API error: ' . $error_msg);
        }
        
        $openai_data = $data['data'];
        if (!isset($openai_data['choices'][0]['message']['content'])) {
            throw new Exception('Invalid response structure from SaaS API');
        }

        $enhanced_content = wp_kses_post($openai_data['choices'][0]['message']['content']);
        $enhanced_content = str_replace(['<body>', '</body>', '```html', '```'], '', $enhanced_content);
        
        // Additional SEO enhancements
        $enhanced_content = $this->add_schema_markup($enhanced_content, $post_id);
        $enhanced_content = $this->optimize_meta_elements($enhanced_content, $post_id);
        
        return $enhanced_content;
    }
    
    /**
     * Add schema markup to enhanced content
     */
    private function add_schema_markup($content, $post_id) {
        // Add Article schema markup
        $post = get_post($post_id);
        $business_name = get_bloginfo('name');
        $service_area = get_option('my_auto_blog_service_area', '');
        
        $schema = array(
            '@context' => 'https://schema.org',
            '@type' => 'Article',
            'headline' => $post ? $post->post_title : '',
            'author' => array(
                '@type' => 'Organization',
                'name' => $business_name,
                'address' => array(
                    '@type' => 'PostalAddress',
                    'addressLocality' => $service_area
                )
            ),
            'publisher' => array(
                '@type' => 'Organization',
                'name' => $business_name
            ),
            'datePublished' => $post ? get_the_date('c', $post_id) : date('c'),
            'dateModified' => date('c')
        );
        
        // Store schema as post meta for later output in head
        update_post_meta($post_id, '_ai_enhanced_schema', json_encode($schema));
        
        return $content;
    }
    
    /**
     * Generate and store meta description
     */
    private function optimize_meta_elements($content, $post_id) {
        // Generate meta description from enhanced content
        $text = wp_strip_all_tags($content);
        $sentences = preg_split('/[.!?]+/', $text, -1, PREG_SPLIT_NO_EMPTY);
        $meta_description = '';
        
        if (!empty($sentences)) {
            $first_sentences = array_slice($sentences, 0, 2);
            $meta_description = implode('. ', $first_sentences);
            $meta_description = substr($meta_description, 0, 155);
            if (!preg_match('/[.!?]$/', $meta_description)) {
                $meta_description .= '.';
            }
        }
        
        // Store meta description as post meta
        update_post_meta($post_id, '_ai_enhanced_meta_description', $meta_description);
        
        return $content;
    }

    /**
     * Analyze existing content structure to determine what needs improvement
     */
    private function analyze_content_structure($content) {
        $analysis = [];
        
        // Remove HTML tags for text analysis but keep structure indicators
        $text_content = strip_tags($content);
        $html_content = $content;
        
        // Check for existing headings
        $h2_count = preg_match_all('/<h2[^>]*>/i', $html_content);
        $h3_count = preg_match_all('/<h3[^>]*>/i', $html_content);
        $analysis[] = "Headings: " . ($h2_count + $h3_count > 0 ? "{$h2_count} H2s, {$h3_count} H3s found" : "No headings detected");
        
        // Check for lists
        $ul_count = preg_match_all('/<ul[^>]*>/i', $html_content);
        $ol_count = preg_match_all('/<ol[^>]*>/i', $html_content);
        $analysis[] = "Lists: " . ($ul_count + $ol_count > 0 ? "{$ul_count} bullet lists, {$ol_count} numbered lists" : "No lists found");
        
        // Check for emphasis
        $bold_count = preg_match_all('/<(strong|b)[^>]*>/i', $html_content);
        $analysis[] = "Emphasis: " . ($bold_count > 0 ? "{$bold_count} bold elements found" : "No bold text found");
        
        // Check content length and paragraph structure
        $word_count = str_word_count($text_content);
        $paragraph_count = preg_match_all('/<p[^>]*>/i', $html_content);
        $analysis[] = "Content: {$word_count} words in {$paragraph_count} paragraphs";
        
        // Check for long paragraphs (potential wall-of-text)
        $paragraphs = explode('</p>', $html_content);
        $long_paragraphs = 0;
        foreach ($paragraphs as $para) {
            $para_text = strip_tags($para);
            if (str_word_count($para_text) > 100) {
                $long_paragraphs++;
            }
        }
        if ($long_paragraphs > 0) {
            $analysis[] = "Issues: {$long_paragraphs} paragraphs over 100 words (may need breaking up)";
        }
        
        // Check for existing schema or structured data
        if (strpos($html_content, 'schema.org') !== false) {
            $analysis[] = "Schema: Existing schema markup detected";
        }
        
        // Assessment summary
        $needs_headings = ($h2_count + $h3_count) < 2;
        $needs_formatting = ($ul_count + $ol_count + $bold_count) < 2;
        $needs_structure = $long_paragraphs > 2;
        
        if ($needs_headings || $needs_formatting || $needs_structure) {
            $analysis[] = "RECOMMENDATION: " . 
                ($needs_headings ? "Add headings for structure. " : "") .
                ($needs_formatting ? "Add lists/emphasis for readability. " : "") .
                ($needs_structure ? "Break up long paragraphs. " : "");
        } else {
            $analysis[] = "RECOMMENDATION: Content structure looks good - focus on expanding thin sections and adding value.";
        }
        
        return implode("\n", $analysis);
    }

    /**
     * Add FAQ section to content
     */
    private function add_faq_section($content, $post_id) {
        // Implementation for FAQ generation
        // For now, return enhanced content
        return $this->enrich_article_content($content, $post_id);
    }
    
    /**
     * Improve SEO aspects of content
     */
    private function improve_seo_content($content, $post_id) {
        // Implementation for SEO improvements
        // For now, return enhanced content
        return $this->enrich_article_content($content, $post_id);
    }
    
    /**
     * Output enhanced schema markup in head
     */
    public function output_enhanced_schema() {
        if (!is_single() && !is_page()) {
            return;
        }
        
        global $post;
        if (!$post) {
            return;
        }
        
        $schema_data = get_post_meta($post->ID, '_ai_enhanced_schema', true);
        if (empty($schema_data)) {
            return;
        }
        
        echo '<script type="application/ld+json">' . "\n";
        echo $schema_data . "\n";
        echo '</script>' . "\n";
    }
    
    /**
     * Output enhanced meta description in head
     */
    public function output_enhanced_meta_description() {
        if (!is_single() && !is_page()) {
            return;
        }
        
        global $post;
        if (!$post) {
            return;
        }
        
        $meta_description = get_post_meta($post->ID, '_ai_enhanced_meta_description', true);
        if (empty($meta_description)) {
            return;
        }
        
        echo '<meta name="description" content="' . esc_attr($meta_description) . '">' . "\n";
    }
    
    /**
     * Output CSS directly to admin head
     */
    public function output_admin_css() {
        ?>
        <style type="text/css">
        .ai-blogger-tools-wrapper {
            position: relative;
        }
        
        .ai-blogger-tools-btn {
            background: #0073aa;
            color: white;
            border: none;
            padding: 6px 12px;
            border-radius: 3px;
            cursor: pointer;
            font-size: 13px;
            display: inline-flex;
            align-items: center;
            gap: 6px;
            line-height: 1.2;
            vertical-align: middle;
        }
        
        .ai-blogger-tools-btn:hover {
            background: #005a87;
            color: white;
        }
        
        .ai-blogger-dropdown {
            margin-top: 2px;
            min-width: 240px;
            white-space: nowrap;
        }
        
        .ai-blogger-dropdown ul li a:hover {
            background: #f5f5f5;
        }
        
        .ai-blogger-dropdown .dashicons {
            font-size: 16px;
            width: 16px;
            height: 16px;
        }
        
        .ai-enhancer-loading {
            opacity: 0.6;
            pointer-events: none;
        }
        
        .ai-enhancer-loading::after {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            margin: -10px 0 0 -10px;
            width: 20px;
            height: 20px;
            border: 2px solid #0073aa;
            border-radius: 50%;
            border-top-color: transparent;
            animation: ai-spin 1s linear infinite;
        }
        
        @keyframes ai-spin {
            to { transform: rotate(360deg); }
        }
        </style>
        <?php
    }
    
    /**
     * Check if user has paid access to Content AI Tools
     */
    private function check_paid_access() {
        // Check if subscription functions are available
        if (!function_exists('sica_check_subscription_status')) {
            return false;
        }
        
        $subscription_data = sica_check_subscription_status();
        $customer_email = get_option('sica_customer_email', '');
        $is_email_verified = !empty($customer_email) && get_option('sica_email_verified', false);
        
        // Use robust tier detection with fallback logic
        if ($subscription_data && $subscription_data['ok'] && $is_email_verified) {
            $tier = $subscription_data['tier'] ?? 'lite';
        } else {
            // Fallback to local license data when server is unavailable
            $license_status = get_option('sica_license_status', 'inactive');
            $license_type = get_option('sica_license_type', 'free');
            $tier = $license_status === 'active' && $license_type === 'paid' ? 'starter' : 'lite';
        }
        
        // Only 'lite' tier is free - all other tiers have paid access
        return ($tier !== 'lite');
    }
} 