/**
 * Sica Content AI - Welcome Screen JavaScript
 * Handles interactive welcome flow with simulated demo and real API integration
 */

class SicaWelcome {
    constructor() {
        this.currentStep = 1;
        this.demoInProgress = false;
        this.emailVerified = window.sicaWelcome?.isEmailVerified || false;
        this.progressTracker = null;
        
        this.init();
    }
    
    init() {
        this.bindEvents();
        this.updateFormState();
        this.initializeProgressTracker();
    }
    
    bindEvents() {
        // Key service input
        const keyServiceInput = document.getElementById('key-service');
        const noIdeaCheckbox = document.getElementById('no-idea-checkbox');
        const generateBtn = document.getElementById('generate-demo-btn');
        
        if (keyServiceInput) {
            keyServiceInput.addEventListener('input', () => this.updateFormState());
            keyServiceInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter' && !generateBtn.disabled) {
                    this.startDemo();
                }
            });
        }
        
        if (noIdeaCheckbox) {
            noIdeaCheckbox.addEventListener('change', () => this.handleNoIdeaCheckbox());
        }
        
        if (generateBtn) {
            generateBtn.addEventListener('click', () => this.startDemo());
        }
        
        // Email verification events
        this.bindEmailEvents();
        
        // Help system
        this.bindHelpEvents();
        
        // Preview actions
        this.bindPreviewActions();
    }
    
    bindEmailEvents() {
        const sendVerificationBtn = document.getElementById('send-verification-btn');
        const verifyEmailBtn = document.getElementById('verify-email-btn');
        const resendCodeBtn = document.getElementById('resend-code-btn');
        const changeEmailLink = document.getElementById('change-email-link');
        const closeModalBtn = document.getElementById('close-email-modal');
        
        if (sendVerificationBtn) {
            sendVerificationBtn.addEventListener('click', () => this.sendVerificationCode());
        }
        
        if (verifyEmailBtn) {
            verifyEmailBtn.addEventListener('click', () => this.verifyEmail());
        }
        
        if (resendCodeBtn) {
            resendCodeBtn.addEventListener('click', () => this.resendVerificationCode());
        }
        
        if (changeEmailLink) {
            changeEmailLink.addEventListener('click', (e) => {
                e.preventDefault();
                this.changeEmail();
            });
        }
        
        if (closeModalBtn) {
            closeModalBtn.addEventListener('click', () => this.closeEmailModal());
        }
        
        // Verification code input formatting
        const verificationCodeInput = document.getElementById('verification-code');
        if (verificationCodeInput) {
            verificationCodeInput.addEventListener('input', (e) => {
                // Only allow numbers and limit to 6 digits
                e.target.value = e.target.value.replace(/\D/g, '').substring(0, 6);
                
                // Auto-verify when 6 digits entered
                if (e.target.value.length === 6) {
                    setTimeout(() => this.verifyEmail(), 500);
                }
            });
        }
    }
    
    bindHelpEvents() {
        const helpButton = document.getElementById('help-button');
        const helpPopover = document.getElementById('help-popover');
        
        if (helpButton && helpPopover) {
            helpButton.addEventListener('click', () => {
                const isVisible = helpPopover.style.display === 'block';
                helpPopover.style.display = isVisible ? 'none' : 'block';
            });
            
            // Close help when clicking outside
            document.addEventListener('click', (e) => {
                if (!helpButton.contains(e.target) && !helpPopover.contains(e.target)) {
                    helpPopover.style.display = 'none';
                }
            });
        }
    }
    
    bindPreviewActions() {
        const publishBtn = document.getElementById('publish-demo-btn');
        const retryBtn = document.getElementById('retry-demo-btn');
        
        if (publishBtn) {
            publishBtn.addEventListener('click', () => this.initiateRealPost());
        }
        
        if (retryBtn) {
            retryBtn.addEventListener('click', () => this.retryDemo());
        }
    }
    
    updateFormState() {
        const keyServiceInput = document.getElementById('key-service');
        const generateBtn = document.getElementById('generate-demo-btn');
        
        if (keyServiceInput && generateBtn) {
            const hasValue = keyServiceInput.value.trim().length > 0;
            generateBtn.disabled = !hasValue || this.demoInProgress;
            
            if (hasValue) {
                generateBtn.classList.add('ready');
            } else {
                generateBtn.classList.remove('ready');
            }
        }
    }
    
    handleNoIdeaCheckbox() {
        const checkbox = document.getElementById('no-idea-checkbox');
        const keyServiceInput = document.getElementById('key-service');
        
        if (checkbox?.checked && keyServiceInput) {
            // Suggest a service based on detected industry
            const suggestions = this.getServiceSuggestions(window.sicaWelcome?.detectedIndustry || '');
            const suggestion = suggestions[Math.floor(Math.random() * suggestions.length)];
            keyServiceInput.value = suggestion;
            this.updateFormState();
        }
    }
    
    getServiceSuggestions(industry) {
        const suggestions = {
            'plumbing': ['Emergency plumbing repairs', 'Drain cleaning services', 'Water heater installation'],
            'real estate': ['Home buying consultation', 'Property valuation services', 'First-time buyer guidance'],
            'digital marketing': ['SEO optimization services', 'Social media management', 'PPC advertising campaigns'],
            'law': ['Legal consultation services', 'Contract review', 'Personal injury representation'],
            'medical': ['Health consultations', 'Preventive care services', 'Specialty treatments'],
            'automotive': ['Auto repair services', 'Vehicle maintenance', 'Brake inspection'],
            'dental': ['Dental cleanings', 'Cosmetic dentistry', 'Emergency dental care'],
            'hvac': ['AC repair services', 'Heating system maintenance', 'HVAC installation']
        };
        
        // Try to match industry
        const industryLower = industry.toLowerCase();
        for (const [key, values] of Object.entries(suggestions)) {
            if (industryLower.includes(key)) {
                return values;
            }
        }
        
        // Default suggestions
        return [
            'Professional consultation',
            'Expert services',
            'Customer support',
            'Quality solutions'
        ];
    }
    
    initializeProgressTracker() {
        // Initialize progress tracker for real API calls
        if (window.ArticleProgressTracker) {
            this.progressTracker = new window.ArticleProgressTracker('progress-container');
        }
    }
    
    async startDemo() {
        if (this.demoInProgress) return;
        
        this.demoInProgress = true;
        const keyService = document.getElementById('key-service')?.value.trim() || '';
        
        // Update UI
        this.showPreviewCard();
        this.updateStep(1, true);
        
        // Start simulated demo stages
        try {
            await this.runDemoStages(keyService);
        } catch (error) {
            console.error('Demo failed:', error);
            this.showDemoError('Demo generation failed. Please try again.');
        } finally {
            this.demoInProgress = false;
        }
    }
    
    async runDemoStages(keyService) {
        const stages = [
            { stage: 1, delay: 1000 },
            { stage: 2, delay: 1500 },
            { stage: 3, delay: 2000 },
            { stage: 4, delay: 1500 }
        ];
        
        for (const { stage, delay } of stages) {
            await this.delay(delay);
            const response = await this.makeAjaxRequest('generate_demo', {
                key_service: keyService,
                stage: stage
            });
            
            if (response.success) {
                this.updateDemoProgress(response.data);
            } else {
                throw new Error(response.data || 'Demo stage failed');
            }
        }
    }
    
    updateDemoProgress(data) {
        const statusEl = document.getElementById('preview-status');
        const titleEl = document.getElementById('generated-title');
        const contentEl = document.getElementById('generated-intro');
        const imageEl = document.getElementById('generated-image');
        const imagePlaceholder = document.getElementById('image-placeholder');
        const previewContent = document.getElementById('preview-content');
        const previewActions = document.getElementById('preview-actions');
        
        if (statusEl) {
            statusEl.textContent = data.message;
        }
        
        if (data.stage >= 2 && data.title && titleEl) {
            titleEl.textContent = data.title;
            if (previewContent) previewContent.style.display = 'block';
        }
        
        if (data.stage >= 3 && data.content && contentEl) {
            contentEl.textContent = data.content;
        }
        
        if (data.stage >= 4 && data.image) {
            if (imageEl && imagePlaceholder) {
                imageEl.src = data.image;
                imageEl.style.display = 'block';
                imagePlaceholder.style.display = 'none';
            }
            
            if (statusEl) {
                statusEl.textContent = '✅ Demo Complete!';
                statusEl.style.background = '#28a745';
            }
            
            if (previewActions) {
                previewActions.style.display = 'flex';
            }
        }
    }
    
    showPreviewCard() {
        const previewCard = document.getElementById('preview-card');
        if (previewCard) {
            previewCard.style.display = 'block';
            previewCard.scrollIntoView({ behavior: 'smooth', block: 'center' });
        }
    }
    
    showDemoError(message) {
        const statusEl = document.getElementById('preview-status');
        if (statusEl) {
            statusEl.textContent = '❌ ' + message;
            statusEl.style.background = '#dc3545';
        }
    }
    
    initiateRealPost() {
        if (this.emailVerified) {
            this.createRealPost();
        } else {
            this.showEmailModal();
        }
    }
    
    showEmailModal() {
        const modal = document.getElementById('email-modal');
        if (modal) {
            modal.style.display = 'flex';
            
            // Focus on email input
            const emailInput = document.getElementById('verification-email');
            if (emailInput) {
                setTimeout(() => emailInput.focus(), 300);
            }
        }
    }
    
    closeEmailModal() {
        const modal = document.getElementById('email-modal');
        if (modal) {
            modal.style.display = 'none';
        }
    }
    
    async sendVerificationCode() {
        const emailInput = document.getElementById('verification-email');
        const sendBtn = document.getElementById('send-verification-btn');
        
        if (!emailInput || !sendBtn) return;
        
        const email = emailInput.value.trim();
        if (!this.isValidEmail(email)) {
            this.showEmailError('Please enter a valid email address.');
            return;
        }
        
        sendBtn.disabled = true;
        sendBtn.innerHTML = '<span class="spinner"></span> Sending...';
        
        try {
            const response = await this.makeAjaxRequest('send_verification', { 
                email,
                consent_source: 'plugin_setup'
            });
            
            if (response.success) {
                this.showEmailStep(2);
                document.getElementById('verification-email-display').textContent = email;
            } else {
                this.showEmailError(response.data || 'Failed to send verification code.');
            }
        } catch (error) {
            this.showEmailError('Network error. Please try again.');
        } finally {
            sendBtn.disabled = false;
            sendBtn.innerHTML = '📨 Send Verification Code';
        }
    }
    
    async verifyEmail() {
        const codeInput = document.getElementById('verification-code');
        const verifyBtn = document.getElementById('verify-email-btn');
        
        if (!codeInput || !verifyBtn) return;
        
        const code = codeInput.value.trim();
        if (code.length !== 6) {
            this.showEmailError('Please enter the complete 6-digit code.');
            return;
        }
        
        verifyBtn.disabled = true;
        verifyBtn.innerHTML = '<span class="spinner"></span> Verifying...';
        
        try {
            const response = await this.makeAjaxRequest('verify_email', { code });
            
            if (response.success) {
                this.emailVerified = true;
                this.closeEmailModal();
                this.createRealPost();
            } else {
                this.showEmailError(response.data || 'Invalid verification code.');
            }
        } catch (error) {
            this.showEmailError('Verification failed. Please try again.');
        } finally {
            verifyBtn.disabled = false;
            verifyBtn.innerHTML = '✅ Verify & Create Post';
        }
    }
    
    async resendVerificationCode() {
        const email = document.getElementById('verification-email-display')?.textContent;
        if (!email) return;
        
        const resendBtn = document.getElementById('resend-code-btn');
        if (resendBtn) {
            resendBtn.disabled = true;
            resendBtn.innerHTML = '<span class="spinner"></span> Sending...';
        }
        
        try {
            await this.makeAjaxRequest('send_verification', { email });
            this.showEmailSuccess('New verification code sent!');
        } catch (error) {
            this.showEmailError('Failed to resend code.');
        } finally {
            if (resendBtn) {
                resendBtn.disabled = false;
                resendBtn.innerHTML = '🔄 Resend Code';
            }
        }
    }
    
    changeEmail() {
        this.showEmailStep(1);
        const emailInput = document.getElementById('verification-email');
        if (emailInput) {
            emailInput.focus();
        }
    }
    
    showEmailStep(step) {
        const step1 = document.getElementById('email-step-1');
        const step2 = document.getElementById('email-step-2');
        
        if (step === 1) {
            if (step1) step1.style.display = 'block';
            if (step2) step2.style.display = 'none';
        } else {
            if (step1) step1.style.display = 'none';
            if (step2) step2.style.display = 'block';
        }
    }
    
    showEmailError(message) {
        // Remove existing error messages
        const existingError = document.querySelector('.email-error');
        if (existingError) existingError.remove();
        
        // Create new error message
        const errorEl = document.createElement('div');
        errorEl.className = 'email-error';
        errorEl.style.cssText = 'background: #f8d7da; color: #721c24; padding: 8px 12px; border-radius: 4px; margin: 8px 0; font-size: 14px;';
        errorEl.textContent = message;
        
        // Insert after email form
        const emailForm = document.getElementById('email-form');
        if (emailForm) {
            emailForm.appendChild(errorEl);
        }
    }
    
    showEmailSuccess(message) {
        // Remove existing messages
        const existingMsg = document.querySelector('.email-success');
        if (existingMsg) existingMsg.remove();
        
        // Create success message
        const successEl = document.createElement('div');
        successEl.className = 'email-success';
        successEl.style.cssText = 'background: #d4edda; color: #155724; padding: 8px 12px; border-radius: 4px; margin: 8px 0; font-size: 14px;';
        successEl.textContent = message;
        
        const emailForm = document.getElementById('email-form');
        if (emailForm) {
            emailForm.appendChild(successEl);
        }
    }
    
    async createRealPost() {
        const keyService = document.getElementById('key-service')?.value.trim() || '';
        const title = document.getElementById('generated-title')?.textContent || '';
        
        // Show real progress component
        const progressContainer = document.getElementById('progress-container');
        const previewContent = document.getElementById('preview-content');
        const previewActions = document.getElementById('preview-actions');
        
        if (progressContainer) progressContainer.style.display = 'block';
        if (previewContent) previewContent.style.display = 'none';
        if (previewActions) previewActions.style.display = 'none';
        
        // Initialize real progress tracking
        if (this.progressTracker) {
            this.progressTracker.updateProgress('analyze', 10, 'Starting real article generation...');
        }
        
        try {
            const response = await this.makeAjaxRequest('create_real_post', {
                key_service: keyService,
                title: title
            });
            
            if (response.success) {
                this.showCelebration();
                this.updateStep(2, true);
                this.completeWelcome();
            } else {
                throw new Error(response.data || 'Failed to create post');
            }
        } catch (error) {
            console.error('Real post creation failed:', error);
            if (this.progressTracker) {
                this.progressTracker.error('Failed to create post. Please try again.');
            }
        }
    }
    
    showCelebration() {
        // Simple celebration - could be enhanced with confetti library
        const celebration = document.getElementById('celebration-confetti');
        if (celebration) {
            celebration.innerHTML = `
                <div style="position: fixed; top: 50%; left: 50%; transform: translate(-50%, -50%); 
                            background: white; padding: 40px; border-radius: 16px; text-align: center; 
                            box-shadow: 0 16px 64px rgba(0,0,0,0.2); z-index: 10000;">
                    <div style="font-size: 48px; margin-bottom: 16px;">🎉</div>
                    <h3 style="margin: 0 0 8px 0; color: #28a745;">Your site just got richer!</h3>
                    <p style="margin: 0; color: #6c757d;">Check Search Console in a few days to see it index.</p>
                    <button onclick="this.parentElement.parentElement.style.display='none'" 
                            style="margin-top: 20px; background: #28a745; color: white; border: none; 
                                   padding: 12px 24px; border-radius: 8px; cursor: pointer;">
                        Continue to Dashboard
                    </button>
                </div>
                <div style="position: fixed; top: 0; left: 0; width: 100%; height: 100%; 
                            background: rgba(0,0,0,0.5); backdrop-filter: blur(4px);"></div>
            `;
            celebration.style.display = 'block';
        }
    }
    
    retryDemo() {
        // Reset demo state
        const previewCard = document.getElementById('preview-card');
        const previewContent = document.getElementById('preview-content');
        const previewActions = document.getElementById('preview-actions');
        const progressContainer = document.getElementById('progress-container');
        
        if (previewCard) previewCard.style.display = 'none';
        if (previewContent) previewContent.style.display = 'none';
        if (previewActions) previewActions.style.display = 'none';
        if (progressContainer) progressContainer.style.display = 'none';
        
        // Clear form
        const keyServiceInput = document.getElementById('key-service');
        if (keyServiceInput) {
            keyServiceInput.value = '';
            keyServiceInput.focus();
        }
        
        this.updateFormState();
        this.demoInProgress = false;
    }
    
    updateStep(step, completed = false) {
        const steps = document.querySelectorAll('.step');
        steps.forEach((stepEl, index) => {
            stepEl.classList.remove('active', 'completed');
            
            if (index + 1 < step || (index + 1 === step && completed)) {
                stepEl.classList.add('completed');
            } else if (index + 1 === step) {
                stepEl.classList.add('active');
            }
        });
        
        this.currentStep = step;
    }
    
    async completeWelcome() {
        try {
            await this.makeAjaxRequest('complete_welcome', {});
            
            // Redirect to main dashboard after delay
            setTimeout(() => {
                window.location.href = '/wp-admin/admin.php?page=ai-blogger';
            }, 3000);
        } catch (error) {
            console.error('Failed to complete welcome:', error);
        }
    }
    
    async makeAjaxRequest(actionType, data = {}) {
        const formData = new FormData();
        formData.append('action', 'sica_welcome');
        formData.append('action_type', actionType);
        formData.append('nonce', window.sicaWelcome?.nonce || '');
        
        for (const [key, value] of Object.entries(data)) {
            formData.append(key, value);
        }
        
        const response = await fetch(window.sicaWelcome?.ajaxUrl || '/wp-admin/admin-ajax.php', {
            method: 'POST',
            body: formData
        });
        
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        
        return await response.json();
    }
    
    isValidEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
    
    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    if (document.body.classList.contains('sica-welcome-body')) {
        new SicaWelcome();
    }
});

// Handle modal clicks outside content
document.addEventListener('click', (e) => {
    const emailModal = document.getElementById('email-modal');
    const modalContent = document.querySelector('.email-modal-content');
    
    if (emailModal && e.target === emailModal && !modalContent?.contains(e.target)) {
        emailModal.style.display = 'none';
    }
});

// Keyboard shortcuts
document.addEventListener('keydown', (e) => {
    // Escape to close modals
    if (e.key === 'Escape') {
        const emailModal = document.getElementById('email-modal');
        const helpPopover = document.getElementById('help-popover');
        
        if (emailModal?.style.display === 'flex') {
            emailModal.style.display = 'none';
        }
        
        if (helpPopover?.style.display === 'block') {
            helpPopover.style.display = 'none';
        }
    }
});