<?php
/**
 * Admin Interface Management
 * 
 * Handles WordPress admin interface, menus, tabs, and asset loading
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Include category fix utility
require_once plugin_dir_path(__FILE__) . 'category-fix-utility.php';

// Include PLG planner tab
require_once plugin_dir_path(__FILE__) . 'plg-planner.php';

// Note: This function has been consolidated into my_auto_blog_enqueue_assets below

/**
 * Add settings link on the plugin page
 */
function ai_blog_writer_settings_link($links) {
    $settings_link = '<a href="admin.php?page=ai-blogger">Settings</a>';
    array_unshift($links, $settings_link);
    return $links;
}

/**
 * Enqueue scripts and styles for admin interface
 * Consolidated function that handles all admin assets
 */
function my_auto_blog_enqueue_assets($hook) {
    // Always load menu icon styles in admin
    wp_enqueue_style('ai-blogger-menu-styles', plugins_url('../styles.css', __FILE__), array(), null, 'all');
    
    // Only load other assets on plugin page
    if ('toplevel_page_ai-blogger' !== $hook) {
        return;
    }
    
    // Enqueue JavaScript with dependencies
    wp_enqueue_script('ai-blogger-scripts', plugins_url('../scripts.js', __FILE__), array('jquery'), null, true);
    wp_localize_script('ai-blogger-scripts', 'myAutoBlog', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce'    => wp_create_nonce('my_auto_blog_nonce'),
    ));
    
    // Enable JS translations for admin script
    if (function_exists('wp_set_script_translations')) {
        wp_set_script_translations('ai-blogger-scripts', 'sica-content-ai', plugin_dir_path(__FILE__) . '../languages');
    }
}
add_action('admin_enqueue_scripts', 'my_auto_blog_enqueue_assets');

/**
 * Add the main menu page
 */
function my_auto_blog_menu() {
    // Use custom SVG icon for the main menu (resolve from plugin root)
    $plugin_main_file = dirname(__DIR__) . '/sica-content-ai.php';
    $icon_url = plugins_url('assets/sica-content-ai-wordpress-plugin-icon-20px.svg', $plugin_main_file);
    add_menu_page(
        esc_html__('Content AI', 'sica-content-ai'),
        esc_html__('Content AI', 'sica-content-ai'),
        'manage_options',
        'ai-blogger',
        'my_auto_blog_settings_page',
        $icon_url,
        30
    );
}
add_action('admin_menu', 'my_auto_blog_menu');

/**
 * Render the main settings page with tabs
 */
function my_auto_blog_settings_page() {
    // Get current tab, default to 'planner' (first tab for PLG)
    $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'planner';
    
    // Determine tier (free users cannot access Scheduler)
    $is_free_tier = true;
    if (function_exists('sica_check_subscription_status')) {
        $status = sica_check_subscription_status();
        $customer_email = get_option('sica_customer_email', '');
        $is_email_verified = !empty($customer_email) && get_option('sica_email_verified', false);
        
        // Use robust tier detection with fallback logic
        if ($status && $status['ok'] && $is_email_verified) {
            $tier = $status['tier'] ?? 'lite';
            $is_free_tier = ($tier === 'lite');
        } else {
            // Fallback to local license data when server is unavailable
            $license_status = get_option('sica_license_status', 'inactive');
            $license_type = get_option('sica_license_type', 'free');
            $tier = $license_status === 'active' && $license_type === 'paid' ? 'starter' : 'lite';
            $is_free_tier = ($tier === 'lite');
        }
    }
    
    // Reordered tabs with PLG-focused "Overview" first
    $tabs = [
        'planner' => esc_html__('Overview', 'sica-content-ai'),
        'content_planner' => esc_html__('Content Planner', 'sica-content-ai'),
        'scheduler' => esc_html__('Scheduler', 'sica-content-ai'), 
        'prompts' => esc_html__('AI Settings', 'sica-content-ai'),
        'setup' => esc_html__('Account Settings', 'sica-content-ai')
    ];

    echo '<div class="wrap">';
    echo '<h2>' . esc_html__('Content AI', 'sica-content-ai') . '</h2>';
    echo '<h2 class="nav-tab-wrapper">';
    foreach ($tabs as $tab => $name) {
        $class = ($active_tab === $tab) ? ' nav-tab-active' : '';
        echo "<a href='?page=ai-blogger&tab=$tab' class='nav-tab$class'>$name</a>";
    }
    echo '</h2>';

    // Map tab names to function names (backward compatibility + new PLG tab)
    $tab_function_map = [
        'planner' => 'my_auto_blog_plg_planner_tab',
        'content_planner' => 'my_auto_blog_content_planner_tab', // backward compatibility
        'scheduler' => 'my_auto_blog_scheduler_tab',
        'prompts' => 'my_auto_blog_prompts_tab', 
        'setup' => 'my_auto_blog_setup_tab'
    ];
    
    // Note: All tabs are accessible for WordPress.org compliance
    // Upgrade prompts are shown within the scheduler interface instead
    $function_name = $tab_function_map[$active_tab] ?? 'my_auto_blog_plg_planner_tab';
    if (function_exists($function_name)) {
        $function_name();
    } else {
        echo '<div style="padding: 20px;"><p>Tab content coming soon...</p></div>';
    }

    echo '</div>';
}

// Note: Styles are now handled by my_auto_blog_enqueue_assets function above

/**
 * Display admin notices
 */
function my_admin_notices() {
    $screen = get_current_screen();
    if ($screen && $screen->id !== 'toplevel_page_ai-blogger') {
        return;
    }

    // Check for updates
    if (isset($_GET['settings-updated']) && $_GET['settings-updated'] === 'true') {
        ?>
        <div class="notice notice-success is-dismissible">
            <p>Settings saved successfully!</p>
        </div>
        <?php
    }

    // Check for specific update messages
    if (isset($_GET['prompts-updated']) && $_GET['prompts-updated'] === 'true') {
        ?>
        <div class="notice notice-success is-dismissible">
            <p>AI prompts updated successfully!</p>
        </div>
        <?php
    }

    if (isset($_GET['prompts-fixed']) && $_GET['prompts-fixed'] === 'true') {
        ?>
        <div class="notice notice-success is-dismissible">
            <p>Prompt formatting issues have been fixed!</p>
        </div>
        <?php
    }
}
add_action('admin_notices', 'my_admin_notices');