<?php
/**
 * Category Fix Utility
 * 
 * One-time utility to fix uncategorized posts and apply proper categories
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Fix uncategorized posts by re-running category assignment
 */
function fix_uncategorized_posts() {
    // Get all posts in 'Uncategorized' category
    $uncategorized_posts = get_posts(array(
        'category_name' => 'uncategorized',
        'post_type' => 'post',
        'posts_per_page' => -1,
        'post_status' => 'any'
    ));
    
    $business_category = get_option('my_auto_blog_business_category', '');
    $fixed_count = 0;
    
    foreach ($uncategorized_posts as $post) {
        $title = $post->post_title;
        $content = $post->post_content;
        
        error_log("Fixing category for post: '$title'");
        
        // Re-run intelligent category assignment
        $category_id = assign_intelligent_category($title, $content, $business_category);
        
        if ($category_id) {
            // Remove from uncategorized and assign new category
            $uncategorized_cat = get_category_by_slug('uncategorized');
            $current_categories = wp_get_post_categories($post->ID);
            
            // Remove uncategorized if it exists
            if ($uncategorized_cat) {
                $current_categories = array_diff($current_categories, array($uncategorized_cat->term_id));
            }
            
            // Add new category
            $current_categories[] = $category_id;
            
            wp_set_post_categories($post->ID, array_unique($current_categories));
            
            $fixed_count++;
            error_log("Fixed post '$title' - assigned to category ID: $category_id");
        } else {
            error_log("Could not determine category for post: '$title'");
        }
    }
    
    return $fixed_count;
}

/**
 * AJAX handler to fix uncategorized posts
 */
function fix_uncategorized_posts_ajax() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    
    $fixed_count = fix_uncategorized_posts();
    
    wp_send_json_success(array(
        'fixed_count' => $fixed_count,
        'message' => "Fixed $fixed_count uncategorized posts"
    ));
}
add_action('wp_ajax_fix_uncategorized_posts', 'fix_uncategorized_posts_ajax');

/**
 * Test category assignment for different business types
 */
function test_category_assignment($test_business_category) {
    $categories = get_business_adaptive_categories($test_business_category);
    return array_keys($categories);
}

/**
 * Add fix button to admin interface
 */
function add_category_fix_button() {
    // Get count of uncategorized posts
    $uncategorized_posts = get_posts(array(
        'category_name' => 'uncategorized',
        'post_type' => 'post',
        'posts_per_page' => -1,
        'post_status' => 'any'
    ));
    
    $count = count($uncategorized_posts);
    
    if ($count > 0) {
        echo '<div style="background: #fff3cd; border: 1px solid #ffeaa7; border-radius: 8px; padding: 15px; margin: 20px 0;">';
        echo '<h4 style="margin-top: 0; color: #856404;">📂 Category Fix Available</h4>';
        echo "<p>Found <strong>$count uncategorized posts</strong> that can be automatically categorized.</p>";
        echo '<button type="button" class="button button-secondary" onclick="fixUncategorizedPosts()" id="fix-categories-btn">';
        echo '🔧 Fix Categories Automatically';
        echo '</button>';
        echo '<div id="fix-categories-result" style="margin-top: 10px;"></div>';
        echo '</div>';
    }
}

/**
 * Add JavaScript for category fix functionality
 */
function add_category_fix_js() {
    ?>
    <script>
    function fixUncategorizedPosts() {
        const button = document.getElementById('fix-categories-btn');
        const result = document.getElementById('fix-categories-result');
        
        button.disabled = true;
        button.textContent = '🔧 Fixing Categories...';
        
        fetch(ajaxurl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'fix_uncategorized_posts',
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                result.innerHTML = '<div style="color: #155724; background: #d4edda; border: 1px solid #c3e6cb; padding: 10px; border-radius: 4px;">' +
                    '✅ ' + data.data.message +
                    '</div>';
                button.style.display = 'none';
                
                // Refresh page after 2 seconds to show updated categories
                setTimeout(() => {
                    location.reload();
                }, 2000);
            } else {
                result.innerHTML = '<div style="color: #721c24; background: #f8d7da; border: 1px solid #f5c6cb; padding: 10px; border-radius: 4px;">' +
                    '❌ Error: ' + data.data +
                    '</div>';
                button.disabled = false;
                button.textContent = '🔧 Fix Categories Automatically';
            }
        })
        .catch(error => {
            result.innerHTML = '<div style="color: #721c24; background: #f8d7da; border: 1px solid #f5c6cb; padding: 10px; border-radius: 4px;">' +
                '❌ Network error: ' + error.message +
                '</div>';
            button.disabled = false;
            button.textContent = '🔧 Fix Categories Automatically';
        });
    }
    </script>
    <?php
}
