<?php
/**
 * Image Generation with DALL-E
 * 
 * Handles AI image generation using OpenAI's DALL-E models and image optimization
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Save base64 image data to WordPress media library
 */
function save_base64_image_to_media_library($base64_data, $title = '') {
    // Decode base64 data
    $image_data = base64_decode($base64_data);
    if (!$image_data) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Failed to decode base64 image data'); }
        return false;
    }
    
    // Generate filename
    $filename = 'ai-generated-' . sanitize_title($title) . '-' . time() . '.png';
    
    // Get WordPress upload directory
    $upload_dir = wp_upload_dir();
    $file_path = $upload_dir['path'] . '/' . $filename;
    $file_url = $upload_dir['url'] . '/' . $filename;
    
    // Save file
    if (file_put_contents($file_path, $image_data) === false) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Failed to save image file'); }
        return false;
    }
    
    // Add to WordPress media library
    $attachment = array(
        'guid' => $file_url,
        'post_mime_type' => 'image/png',
        'post_title' => $title ? 'AI Generated: ' . $title : 'AI Generated Image',
        'post_content' => '',
        'post_status' => 'inherit'
    );
    
    $attach_id = wp_insert_attachment($attachment, $file_path);
    
    if (is_wp_error($attach_id)) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Failed to create attachment'); }
        return false;
    }
    
    // Optimize and generate metadata (convert to WebP if possible)
    require_once(ABSPATH . 'wp-admin/includes/image.php');
    if (function_exists('optimize_image_attachment')) {
        optimize_image_attachment($attach_id, 1600, 82);
    } else {
        $attach_data = wp_generate_attachment_metadata($attach_id, $file_path);
        wp_update_attachment_metadata($attach_id, $attach_data);
    }
    
    return wp_get_attachment_url($attach_id);
}

/**
 * Main image generation function - uses user-selected model from AI Settings
 */
function generate_image_with_dalle($title) {
    // Determine tier-based default: DALL-E 3 for free tier (cheaper), GPT-Image-1 for paid
    $is_free_tier = true;
    if (function_exists('sica_check_subscription_status')) {
        $status = sica_check_subscription_status();
        $tier = $status['tier'] ?? 'lite';
        $is_free_tier = ($tier === 'lite');
    }
    // Master and premium tiers get gpt-image-1, free tier gets dall-e-3
    $tier_default = $is_free_tier ? 'dall-e-3' : 'gpt-image-1';
    
    // Get user's preferred image model from AI Settings with tier-appropriate default
    $selected_model = get_option('sica_image_model', $tier_default);
    
    if ($selected_model === 'dall-e-3') {
        // Use DALL-E 3 as primary choice
        return generate_image_with_dalle_3($title);
    } else {
        // Use gpt-image-1 as primary choice (default)
        $image_url = generate_image_with_gpt4o($title);
        if ($image_url) {
            return $image_url;
        }
        // Fallback to DALL-E 3 if gpt-image-1 fails
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('GPT-Image-1 generation failed, using DALL-E 3 fallback'); }
        return generate_image_with_dalle_3($title);
    }
}

/**
 * Enhanced DALL-E 3 image generation with improved prompts
 */
function generate_image_with_dalle_3($title) {
    // Use SaaS proxy for DALL-E 3 as well
    $customer_email = get_option('sica_customer_email', '');
    $site_url = get_site_url();

    // Get business context
    $business_name = get_bloginfo('name');
    $business_category = get_option('my_auto_blog_business_category', '');
    $business_service_area = get_option('my_auto_blog_service_area', '');
    
    // Check if user has custom image prompt override
    $custom_image_prompt = get_option('my_auto_blog_image_style_prompt');
    
    if ($custom_image_prompt !== false && !empty(trim($custom_image_prompt))) {
        // User has custom prompt - replace variables in their custom template
        $prompt = replace_image_prompt_variables($custom_image_prompt, $title, $business_category, $business_service_area);
    } else {
        // Use dynamic style-based template system
        $prompt = generate_dynamic_image_prompt($title, $business_category, $business_service_area);
    }

    if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Using SaaS proxy with dall-e-3 for image generation'); }

    // Convert DALL-E format to chat completion format for SaaS proxy compatibility
    $request_data = array(
        'site_url' => $site_url,
        'customer_email' => $customer_email,
        'content_type' => 'image',
        'model' => 'dall-e-3',
        'messages' => array(
            array(
                'role' => 'user',
                'content' => $prompt
            )
        ),
        'title' => $title,
        'dalle_options' => array(
            'n' => 1,
            'size' => '1792x1024',
            'quality' => 'hd',
            'style' => 'natural',
            'response_format' => 'url'
        )
    );

    $response = wp_remote_post('https://api.sica.ai/generate.php', array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'X-Sica-Token' => get_option('sica_api_token', '')
        ),
        'body' => json_encode($request_data),
        'timeout' => 300,
    ));

    if (is_wp_error($response)) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Error in DALL-E request'); }
        return false;
    }

    $response_code = wp_remote_retrieve_response_code($response);
    if ($response_code !== 200) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SaaS API error for DALL-E 3'); }
        return false;
    }

    $data = json_decode(wp_remote_retrieve_body($response), true);
    
    // Check SaaS response structure
    if (!$data || !$data['ok']) {
        $error_msg = $data['error'] ?? 'Unknown SaaS API error';
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SaaS API error for DALL-E 3'); }
        return false;
    }
    
    $openai_data = $data['data'];
    
    // Handle DALL-E response format
    if (isset($openai_data['data'][0]['url'])) {
        return $openai_data['data'][0]['url'];
    }

    if (defined('WP_DEBUG') && WP_DEBUG) { error_log('No image URL found in DALL-E response from SaaS proxy'); }
    return false;
}

/**
 * Advanced image generation using gpt-image-1 model
 */
function generate_image_with_gpt4o($title) {
    // Use SaaS proxy instead of direct OpenAI calls
    $customer_email = get_option('sica_customer_email', '');
    $site_url = get_site_url();

    // Get business context
    $business_name = get_bloginfo('name');
    $business_category = get_option('my_auto_blog_business_category', '');
    $business_service_area = get_option('my_auto_blog_service_area', '');
    
    // Check if user has custom image prompt override
    $custom_image_prompt = get_option('my_auto_blog_image_style_prompt');
    
    if ($custom_image_prompt !== false && !empty(trim($custom_image_prompt))) {
        // User has custom prompt - replace variables in their custom template
        $prompt = replace_image_prompt_variables($custom_image_prompt, $title, $business_category, $business_service_area);
    } else {
        // Use dynamic style-based template system
        $prompt = generate_dynamic_image_prompt($title, $business_category, $business_service_area);
    }

    // GPT image generation via SaaS proxy
    if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Using SaaS proxy with gpt-image-1 for image generation'); }
    
    $request_data = array(
        'site_url' => $site_url,
        'customer_email' => $customer_email,
        'content_type' => 'image',
        'model' => 'gpt-image-1',
        'messages' => array(
            array(
                'role' => 'user',
                'content' => $prompt
            )
        ),
        'title' => $title
    );

    $response = wp_remote_post('https://api.sica.ai/generate.php', array(
        'headers' => array(
            'Content-Type' => 'application/json',
            'X-Sica-Token' => get_option('sica_api_token', '')
        ),
        'body' => json_encode($request_data),
        'timeout' => 300,
    ));

    if (is_wp_error($response)) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Error in GPT image generation request'); }
        return generate_image_with_dalle_3($title);
    }

    $response_code = wp_remote_retrieve_response_code($response);
    if ($response_code !== 200) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SaaS API error for GPT image generation'); }
        return generate_image_with_dalle_3($title);
    }

    $data = json_decode(wp_remote_retrieve_body($response), true);
    
    // Check SaaS response structure first
    if (!$data || !$data['ok']) {
        $error_msg = $data['error'] ?? 'Unknown SaaS API error';
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SaaS API error for image generation'); }
        return generate_image_with_dalle_3($title);
    }
    
    $openai_data = $data['data'];
    
    // gpt-image-1 returns images in Images API format with base64 data
    if (isset($openai_data['data'][0]['b64_json'])) {
        // The Images API returns base64 data - save to WordPress media library
        $base64_data = $openai_data['data'][0]['b64_json'];
        
        // Convert base64 to image file and upload to WordPress
        $image_url = save_base64_image_to_media_library($base64_data, $title);
        if ($image_url) {
            return $image_url;
        }
    }
    
    // Legacy chat completion format check (fallback)
    if (isset($openai_data['choices'][0]['message']['content'])) {
        $content = $openai_data['choices'][0]['message']['content'];
        
        // Look for image data in the content array
        if (is_array($content)) {
            foreach ($content as $item) {
                if (isset($item['type']) && $item['type'] === 'image_url') {
                    // If SaaS responded with a hosted URL, use it directly
                    if (!empty($item['image_url']['url']) && filter_var($item['image_url']['url'], FILTER_VALIDATE_URL)) {
                        return $item['image_url']['url'];
                    }
                    // If SaaS responded with base64 data in URL field, convert
                    if (!empty($item['image_url']['url'])) {
                        $image_url = convert_base64_to_url($item['image_url']['url'], $title);
                        if ($image_url) {
                            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('GPT image generation successful'); }
                            return $image_url;
                        }
                    }
                }
            }
        }
    }

    if (isset($openai_data['error'])) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('GPT image generation API error'); }
    } else {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('No image data found in GPT generation response'); }
    }
    
    // Fall back to DALL-E 3 if GPT image generation fails
    error_log('Falling back to DALL-E 3 for image generation.');
    return generate_image_with_dalle_3($title);
}

/**
 * Helper function to convert base64 image to WordPress URL
 */
function convert_base64_to_url($base64_image, $title) {
    // Create a unique filename based on title and timestamp
    $sanitized_title = sanitize_file_name($title);
    $timestamp = time();
    $filename = 'gpt-image-' . substr($sanitized_title, 0, 30) . '-' . $timestamp . '.png';
    
    // Decode base64 image
    $image_data = base64_decode($base64_image);
        if ($image_data === false) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Failed to decode base64 image data'); }
        return false;
    }
    
    // Get WordPress upload directory
    $upload_dir = wp_upload_dir();
    if ($upload_dir['error']) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('WordPress upload directory error'); }
        return false;
    }
    
    $file_path = $upload_dir['path'] . '/' . $filename;
    
    // Save the image file
    $result = file_put_contents($file_path, $image_data);
    if ($result === false) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Failed to save GPT-generated image'); }
        return false;
    }
    
    // Return the URL
    $file_url = $upload_dir['url'] . '/' . $filename;
    if (defined('WP_DEBUG') && WP_DEBUG) { error_log('GPT-generated image saved successfully'); }
    
    return $file_url;
}

/**
 * Generate dynamic image prompt based on style choice and context
 */
function generate_dynamic_image_prompt($title, $business_category = '', $business_service_area = '') {
    $image_style = get_option('my_auto_blog_image_style', 'professional_photography');
    
    // Get the style template
    $template = get_style_template($image_style);
    
    // Replace variables with actual values
    $prompt = replace_image_prompt_variables($template, $title, $business_category, $business_service_area);
    
    return $prompt;
}

/**
 * Replace variables in image prompt template with actual business context
 */
function replace_image_prompt_variables($template, $title, $business_category, $business_service_area) {
    // Prepare replacement values
    $business_name = get_bloginfo('name');
    $products_services = get_option('my_auto_blog_products_services', '');
    
    // Clean and prepare values
    $title_clean = wp_strip_all_tags($title);
    $business_category_clean = !empty($business_category) ? $business_category : 'business';
    $service_area_clean = !empty($business_service_area) ? $business_service_area : '';
    $products_services_clean = !empty($products_services) ? $products_services : '';
    
    // Define variable replacements
    $replacements = [
        '{title}' => $title_clean,
        '{business_name}' => $business_name,
        '{business_category}' => $business_category_clean,
        '{service_area}' => $service_area_clean,
        '{products_services}' => $products_services_clean,
    ];
    
    // Perform replacements
    $prompt = $template;
    foreach ($replacements as $variable => $value) {
        $prompt = str_replace($variable, $value, $prompt);
    }
    
    // Clean up any leftover variable markers or empty sections
    $prompt = preg_replace('/\{[^}]+\}/', '', $prompt); // Remove any unfilled variables
    $prompt = preg_replace('/\s+/', ' ', $prompt); // Clean up multiple spaces
    $prompt = trim($prompt);
    
    return $prompt;
}

/**
 * AJAX handler to reset image prompt and reload
 */
function ajax_reset_image_prompt_and_reload() {
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    delete_option('my_auto_blog_image_style_prompt');
    wp_send_json_success('Image prompt reset successfully');
}
add_action('wp_ajax_reset_image_prompt_and_reload', 'ajax_reset_image_prompt_and_reload');