<?php
/**
 * PLG-Focused Planner Tab
 * 
 * Product-Led Growth optimized interface focused on: show status, create value, convert
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Render the PLG-focused planner tab
 */
function my_auto_blog_plg_planner_tab() {
    // Debug: Check if required functions exist
    if (!function_exists('sica_check_subscription_status')) {
        echo '<div style="padding: 20px; background: #f8d7da; border: 1px solid #f5c6cb; border-radius: 4px;">';
        echo '<strong>Error:</strong> Subscription system not loaded. Please check that subscription-functions.php is included.';
        echo '</div>';
        return;
    }
    
    // Get subscription data
    $subscription_data = sica_check_subscription_status();
    $customer_email = get_option('sica_customer_email', '');
    $is_email_verified = !empty($customer_email) && get_option('sica_email_verified', false);
    
    // Debug: Log subscription data
    if (defined('WP_DEBUG') && WP_DEBUG) { error_log('PLG Planner - Subscription data loaded'); }
    
    // Determine plan details with robust fallback logic (consistent with Account Settings)
    if ($subscription_data && $subscription_data['ok'] && $is_email_verified) {
        $plan_name = $subscription_data['tier'] ?? 'lite';
        $posts_limit = (int)($subscription_data['posts_limit'] ?? 2);
        $posts_used = (int)($subscription_data['posts_used'] ?? 0);
        $posts_remaining = ($posts_limit === -1) ? -1 : max(0, $posts_limit - $posts_used); // -1 = unlimited
        $reset_date = $subscription_data['billing_period_end'] ?? 'End of month';
    } else {
        // Fallback to local license data when server is unavailable or subscription check fails
        $license_status = get_option('sica_license_status', 'inactive');
        $license_type = get_option('sica_license_type', 'free');
        $plan_name = $license_status === 'active' && $license_type === 'paid' ? 'starter' : 'lite';
        
        // Set appropriate limits based on plan
        if ($plan_name === 'starter') {
            $posts_limit = 12;  // Starter plan gets 12 posts/month
            $posts_used = 0;
            $posts_remaining = 12;
        } else {
            $posts_limit = 2;   // Lite plan gets 2 posts/month
            $posts_used = 0;
            $posts_remaining = 2;
        }
        
        $reset_date = 'End of month';
        if (defined('WP_DEBUG') && WP_DEBUG) { 
            error_log('PLG Planner - Using local fallback data due to server issues: plan=' . $plan_name . ', limit=' . $posts_limit); 
        }
    }
    
    $plan_display_name = ucfirst($plan_name) === 'Lite' ? 'Content AI Lite' : 'Content AI ' . ucfirst($plan_name);
    
    // Format reset date
    if ($reset_date !== 'End of month') {
        $reset_date = date('M j', strtotime($reset_date));
    }
    
    ?>
    <style>
    /* Layout */
    .sica-grid { 
      display: grid; 
      grid-template-columns: minmax(0, 2fr) minmax(280px, 1fr); 
      gap: 24px; 
      align-items: start; 
    }
    .sica-main { min-width: 0; }                 /* allow content to shrink */
    .sica-side { position: sticky; top: 72px; }  /* keep plan/upgrade visible */

    /* Cards (reuse for all panels) */
    .sica-card {
      background:#fff; border:1px solid #e5e7eb; border-radius:10px;
      padding:20px; box-shadow:0 1px 2px rgba(0,0,0,.04);
    }
    .sica-card h3 { margin:0 0 12px; font-size:16px; }

    /* Usage meter example */
    .sica-meter { height:8px; background:#f3f4f6; border-radius:999px; overflow:hidden; }
    .sica-meter > span { display:block; height:100%; background:#10b981; width:0; transition:width .3s; }

    /* Segmented control (scheduler frequencies) */
    .sica-segment { display:flex; gap:8px; flex-wrap:wrap; }
    .sica-segment input[type="radio"] { display:none; }
    .sica-segment .seg {
      padding:6px 10px; border:1px solid #e5e7eb; border-radius:6px; font-size:12px; cursor:pointer; color:#374151; background:#fff;
    }
    .sica-segment .seg.active { background:#eef2ff; color:#4f46e5; border-color:#c7d2fe; }
    .sica-badge { font-size:11px; padding:2px 8px; border-radius:999px; border:1px solid #e5e7eb; color:#374151; background:#f9fafb; }
    .sica-badge.on { color:#065f46; border-color:#a7f3d0; background:#ecfdf5; }
    .sica-text-link { background:none; border:none; color:#6b7280; font-size:12px; cursor:pointer; padding:0; }
    .sica-save-status { font-size:12px; color:#6b7280; opacity:0; transition:opacity .2s; }
    .sica-save-status.show { opacity:1; }
    .sica-save-status.success { color:#065f46; }
    .sica-save-status.error { color:#b91c1c; }

     /* Slim info bar (free tier upgrade nudge) */
     .sica-info-bar {
       margin: 0 0 16px 0;
       background: #f9fafb;
       border: 1px solid #e5e7eb;
       border-radius: 8px;
       padding: 10px 14px;
       display: flex;
       align-items: center;
       justify-content: space-between;
       gap: 12px;
     }
     .sica-info-bar-text { color: #374151; font-size: 13px; }
     .sica-info-actions { display: flex; gap: 8px; }
     .sica-info-actions .link-btn {
       background: white;
       color: #4f46e5;
       border: 1px solid #e5e7eb;
       padding: 6px 10px;
       border-radius: 6px;
       font-size: 12px;
       font-weight: 600;
       text-decoration: none;
       cursor: pointer;
     }
     .sica-info-actions .link-btn:hover { background: #f3f4f6; text-decoration: none; }

    /* Responsive */
    @media (max-width: 1100px){
      .sica-grid { grid-template-columns: 1fr; }
      .sica-side { position: static; }
    }

    .plg-header-bar {
        display: flex;
        align-items: center;
        justify-content: space-between;
        background: white;
        border: 1px solid #ddd;
        border-radius: 8px;
        padding: 20px;
        margin-bottom: 30px;
        box-shadow: 0 1px 3px rgba(0,0,0,0.1);
    }
    /* Compact header for paid tiers to reduce visual weight */
    .plg-header-bar.is-paid { padding-top: 14px; padding-bottom: 14px; margin-bottom: 16px; }
    .plg-header-bar.is-paid .plg-usage-text { display: none; }
    
    .plg-plan-info {
        display: flex;
        align-items: center;
        gap: 15px;
    }
    
    .plg-plan-badge {
        background: #4f46e5;
        color: white;
        padding: 6px 12px;
        border-radius: 20px;
        font-size: 13px;
        font-weight: 600;
    }
    
    .plg-plan-badge.starter {
        background: #059669;
    }
    
    .plg-plan-badge.growth {
        background: #dc2626;
    }
    
    .plg-usage-meter {
        text-align: center;
        flex: 1;
    }
    
    .plg-usage-circle {
        width: 80px;
        height: 80px;
        border-radius: 50%;
        border: 6px solid #e5e7eb;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto 8px;
        position: relative;
        background: conic-gradient(#4f46e5 0deg, #4f46e5 <?php echo $posts_limit > 0 ? min(100, ($posts_used / $posts_limit) * 360) : 0; ?>deg, #e5e7eb <?php echo $posts_limit > 0 ? min(100, ($posts_used / $posts_limit) * 360) : 0; ?>deg);
    }
    
    .plg-usage-circle-inner {
        width: 60px;
        height: 60px;
        border-radius: 50%;
        background: white;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: bold;
        color: #1f2937;
    }
    
    .plg-usage-text {
        color: #6b7280;
        font-size: 13px;
    }
    
    .plg-action-buttons {
        display: flex;
        gap: 10px;
        align-items: center;
    }
    
    .plg-btn-primary {
        background: #4f46e5;
        color: white;
        border: none;
        padding: 12px 24px;
        border-radius: 6px;
        font-weight: 600;
        text-decoration: none;
        cursor: pointer;
        font-size: 14px;
    }
    
    .plg-btn-secondary {
        background: white;
        color: #4f46e5;
        border: 1px solid #4f46e5;
        padding: 12px 20px;
        border-radius: 6px;
        font-weight: 600;
        text-decoration: none;
        cursor: pointer;
        font-size: 14px;
    }
    
    .plg-btn-primary:hover {
        opacity: 0.9;
        color: white !important;
        text-decoration: none;
    }
    
    .plg-btn-secondary:hover {
        opacity: 0.9;
        color: #4f46e5 !important;
        text-decoration: none;
    }
    
    /* Old grid styles removed - now using CSS Grid */
    
    .plg-card {
        background: white;
        border: 1px solid #ddd;
        border-radius: 8px;
        padding: 24px;
        box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        width: 100%;
        box-sizing: border-box;
        min-height: fit-content;
        word-wrap: break-word;
        overflow-wrap: break-word;
    }
    
    .plg-card h3 {
        margin: 0 0 16px 0;
        color: #1f2937;
        font-size: 18px;
        font-weight: 600;
    }
    
    .plg-container {
        max-width: 1200px;
        margin: 16px auto 0; /* add breathing room below tabs */
        padding: 0 20px;
    }
    
    .plg-right-column {
        display: flex;
        flex-direction: column;
        gap: 0;
        height: fit-content;
        min-width: 0;
    }
    
    .plg-generate-panel {
        text-align: center;
    }
    
    .plg-generate-btn {
        background: #059669;
        color: white;
        border: none;
        padding: 20px 40px;
        border-radius: 8px;
        font-size: 18px;
        font-weight: 600;
        cursor: pointer;
        margin-bottom: 16px;
        width: 100%;
        max-width: 300px;
    }
    
    .plg-generate-btn:disabled {
        background: #9ca3af;
        cursor: not-allowed;
    }
    
    /* Compact Generate Button for Header */
    .plg-generate-btn-compact {
        background: #059669;
        color: white;
        border: none;
        padding: 8px 16px;
        border-radius: 6px;
        font-size: 14px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.2s ease;
        width: auto;
        max-width: none;
        margin-bottom: 0;
    }
    
    .plg-generate-btn-compact:hover {
        background: #047857;
        transform: translateY(-1px);
    }
    
    .plg-generate-btn-compact:disabled {
        background: #e5e7eb;
        color: #9ca3af;
        cursor: not-allowed;
        transform: none;
    }
    
    .plg-generate-subtext {
        color: #6b7280;
        font-size: 14px;
        margin-bottom: 20px;
    }
    
    .plg-toggles {
        display: flex;
        gap: 20px;
        justify-content: center;
        align-items: center;
        flex-wrap: wrap;
    }
    
    .plg-toggle {
        display: flex;
        align-items: center;
        gap: 8px;
    }
    
    .plg-locked-card {
        opacity: 0.6;
        position: relative;
    }
    
    .plg-lock-overlay {
        position: absolute;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        background: rgba(255,255,255,0.95);
        padding: 20px;
        border-radius: 8px;
        text-align: center;
        border: 2px solid #f59e0b;
    }
    
    .plg-lock-icon {
        font-size: 24px;
        margin-bottom: 8px;
    }
    
    .plg-keywords-chips {
        display: flex;
        flex-wrap: wrap;
        gap: 8px;
        margin-bottom: 16px;
    }
    
    .plg-keyword-chip {
        background: #ede9fe;
        color: #5b21b6;
        padding: 6px 12px;
        border-radius: 20px;
        font-size: 13px;
        display: flex;
        align-items: center;
        gap: 6px;
    }
    
    .plg-add-keyword {
        background: #f3f4f6;
        color: #6b7280;
        border: 2px dashed #d1d5db;
        cursor: pointer;
    }
    
    .plg-add-keyword:not(.disabled):hover {
        background: #e5e7eb;
    }
    
    .plg-add-keyword.disabled {
        opacity: 0.5;
        cursor: not-allowed;
    }
    
    .plg-warning-banner {
        background: #fef3c7;
        border: 1px solid #f59e0b;
        border-radius: 6px;
        padding: 12px 16px;
        margin-bottom: 20px;
        display: flex;
        align-items: center;
        gap: 10px;
    }
    
    .plg-error-banner {
        background: #fef2f2;
        border: 1px solid #f87171;
        border-radius: 6px;
        padding: 12px 16px;
        margin-bottom: 20px;
        display: flex;
        align-items: center;
        gap: 10px;
    }
    
    /* Modern Progress Component Styles */
    .plg-progress-post {
        padding: 12px;
        background: linear-gradient(135deg, #e3f2fd 0%, #f0f8ff 100%);
        border-radius: 8px;
        margin-bottom: 8px;
        border-left: 3px solid #4f46e5;
        animation: slideInFromTop 0.3s ease-out;
    }
    
    @keyframes slideInFromTop {
        from {
            opacity: 0;
            transform: translateY(-20px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
    
    @keyframes slideOutToTop {
        from {
            opacity: 1;
            transform: translateY(0);
        }
        to {
            opacity: 0;
            transform: translateY(-20px);
        }
    }
    
    .plg-progress-header {
        display: flex;
        align-items: center;
        gap: 10px;
        margin-bottom: 12px;
    }
    
    .plg-progress-icon {
        width: 20px;
        height: 20px;
        border-radius: 50%;
        background: #4f46e5;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 12px;
        animation: pulse 2s infinite;
    }
    
    @keyframes pulse {
        0%, 100% { transform: scale(1); }
        50% { transform: scale(1.1); }
    }
    
    .plg-progress-title {
        font-weight: 500;
        font-size: 14px;
        color: #1f2937;
        flex: 1;
    }
    
    .plg-progress-percentage {
        font-size: 12px;
        font-weight: 600;
        color: #4f46e5;
        background: white;
        padding: 2px 8px;
        border-radius: 12px;
    }
    
    .plg-progress-bar {
        background: rgba(255,255,255,0.8);
        height: 6px;
        border-radius: 3px;
        overflow: hidden;
        margin-bottom: 8px;
    }
    
    .plg-progress-fill {
        background: linear-gradient(90deg, #4f46e5, #7c3aed);
        height: 100%;
        transition: width 0.8s ease;
        border-radius: 3px;
    }
    
    .plg-progress-step {
        font-size: 12px;
        color: #6b7280;
        display: flex;
        align-items: center;
        gap: 6px;
    }
    
    .plg-progress-step-icon {
        font-size: 14px;
    }
    
    .plg-progress-cancel {
        background: none;
        border: none;
        color: #6b7280;
        font-size: 11px;
        cursor: pointer;
        padding: 2px 6px;
        border-radius: 3px;
    }
    
    .plg-progress-cancel:hover {
        background: rgba(0,0,0,0.1);
    }
    
    /* Old mobile styles removed - handled by CSS Grid */
        
        .plg-right-column {
            margin-top: 0;
        }
    }
    
    @media (max-width: 768px) {
        .plg-header-bar {
            flex-direction: column;
            gap: 20px;
            text-align: center;
        }
        
        /* Old mobile styles for grid removed */
        
        .plg-action-buttons {
            flex-direction: column;
            width: 100%;
        }
        
        .plg-btn-primary, .plg-btn-secondary {
            width: 100%;
        }
        
        .plg-card {
            padding: 20px;
        }
    }
    </style>
    
    <div class="plg-container">
        
        <?php if ($posts_remaining <= 1 && $posts_remaining > 0): ?>
        <div class="plg-warning-banner">
            <span>⚠️</span>
            <div>
                <strong>You're near the limit.</strong> 
                <?php if ($plan_name === 'lite'): ?>
                    Unlock 12 posts/month and scheduling with Starter plan.
                    <a href="#" onclick="openUpgradeModal()" style="margin-left: 10px; font-weight: 600;">Upgrade now →</a>
                <?php elseif ($plan_name === 'starter'): ?>
                    Upgrade to Growth for 30 posts/month and bulk tools.
                    <a href="#" onclick="openUpgradeModal()" style="margin-left: 10px; font-weight: 600;">Upgrade now →</a>
                <?php else: ?>
                    You'll need to wait until <?php echo $reset_date; ?> for more posts.
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>
        
        <?php if ($posts_remaining <= 0 && $posts_remaining !== -1): ?>
        <div class="plg-error-banner">
            <span>🚫</span>
            <div>
                <strong>You've used <?php echo $posts_used; ?>/<?php echo $posts_limit; ?> posts this month.</strong> 
                <?php if ($plan_name === 'lite'): ?>
                    Upgrade to keep publishing or wait until <?php echo $reset_date; ?>.
                    <a href="#" onclick="openUpgradeModal()" style="margin-left: 10px; font-weight: 600;">Upgrade now →</a>
                <?php elseif ($plan_name === 'starter'): ?>
                    Upgrade to Growth for more posts or wait until <?php echo $reset_date; ?>.
                    <a href="#" onclick="openUpgradeModal()" style="margin-left: 10px; font-weight: 600;">Upgrade now →</a>
                <?php else: ?>
                    Wait until <?php echo $reset_date; ?> for your posts to reset.
                <?php endif; ?>
            </div>
        </div>
        <?php endif; ?>
        
        
        <?php if ($plan_name === 'lite'): ?>
        <div class="sica-info-bar">
            <div class="sica-info-bar-text">
                Unlock 12 posts/month, auto-scheduling, and unlimited keywords.
            </div>
            <div class="sica-info-actions">
                <a href="#" class="link-btn" onclick="openUpgradeModal();return false;">See plans</a>
                <button class="link-btn" onclick="showLicenseEntry()">Enter key</button>
            </div>
        </div>
        <?php endif; ?>

        <!-- Main Content Grid -->
        <div class="sica-grid">
            
            <!-- LEFT: Create Content (2/3) -->
            <div class="sica-main">
                <?php if (!$is_email_verified): ?>
                <div class="sica-card" style="margin-bottom:16px; border-left:4px solid #f59e0b; background:#fff7ed;">
                    <div style="display:flex; align-items:center; justify-content:space-between; gap:12px;">
                        <div>
                            <h3 style="margin:0; display:flex; align-items:center; gap:8px;">
                                📧 Verify your email to unlock free AI blogs
                            </h3>
                            <p style="margin:.25rem 0; color:#92400e; font-size:13px;">
                                Get access to 2 free AI-powered blogs per month. Quick verification protects your site and prevents abuse.
                            </p>
                        </div>
                    </div>
                    <div style="margin-top:10px;">
                        <div style="display:flex; gap:8px; align-items:center;">
                            <input type="email" id="verification-email" placeholder="your@email.com"
                                   autocomplete="off" autocapitalize="none" spellcheck="false" 
                                   data-lpignore="true" data-form-type="other"
                                   style="flex:1; padding:8px 12px; border:1px solid #f59e0b; border-radius:6px; background:#fff; color:#111827;" />
                            <button onclick="sendVerificationCode()"
                                    style="background:#f59e0b; color:#fff; border:none; padding:8px 12px; border-radius:6px; cursor:pointer;">
                                Send code
                            </button>
                        </div>
                        
                        <!-- Email Purpose Notice -->
                    </div>
                    <ul style="margin:.5rem 0 0; padding-left:1.2rem; color:#92400e; font-size:12px;">
                        <li>No credit card required</li>
                        <li>Emails only used for verification and post notifications you enable</li>
                        <li>Change or remove at any time in Notifications</li>
                    </ul>
                </div>
                <?php endif; ?>

                <div class="sica-card">
                <!-- Card Header with Generate Button -->
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
                    <h3 style="margin: 0;">Create Content</h3>
                    <?php if ($posts_remaining > 0 || $posts_remaining === -1): ?>
                        <button class="plg-generate-btn plg-generate-btn-compact" onclick="toggleGeneratePanel()" <?php echo !$is_email_verified ? 'disabled' : ''; ?>>
                            Generate content now...
                        </button>
                    <?php else: ?>
                        <button class="plg-generate-btn plg-generate-btn-compact" disabled style="background: #e5e7eb; color: #9ca3af; cursor: not-allowed;">
                            Upgrade
                        </button>
                    <?php endif; ?>
                </div>
                
                <!-- Posts Container (Recent + Upcoming) -->
                <div id="posts-container">
                    <!-- Inline Generate Panel (hidden by default) -->
                    <div id="inline-generate-panel" style="display:none; margin-bottom:16px; padding:12px; border:1px solid #e5e7eb; border-radius:8px; background:#f9fafb;">
                        <div style="display:flex; gap:8px; align-items:center; margin-bottom:8px;">
                            <input id="inline-generate-input" type="text" placeholder="Enter a blog title…" style="flex:1; padding:8px 10px; border:1px solid #d1d5db; border-radius:6px;">
                            <button class="button button-primary" onclick="submitInlineGenerate(false)">Generate</button>
                            <button class="button" onclick="toggleGeneratePanel()" style="background:#fff; border:1px solid #e5e7eb; color:#374151;">Cancel</button>
                        </div>
                        <?php 
                        $backlog_titles = array_slice(array_filter(explode("\n", get_option('my_auto_blog_title', ''))), 0, 10);
                        if (!empty($backlog_titles)): ?>
                        <div style="display:flex; gap:8px; align-items:center; margin-top:8px;">
                            <select id="inline-generate-select" style="flex:1; padding:8px 10px; border:1px solid #d1d5db; border-radius:6px;">
                                <option value="">Or pick from your backlog…</option>
                                <?php foreach ($backlog_titles as $t): ?>
                                    <option value="<?php echo esc_attr($t); ?>"><?php echo esc_html(mb_strimwidth($t, 0, 90, '…')); ?></option>
                                <?php endforeach; ?>
                            </select>
                            <button class="button" onclick="submitInlineGenerate(true)" style="background:#fff; border:1px solid #e5e7eb; color:#374151;">Generate selected</button>
                        </div>
                        <?php endif; ?>
                        <div id="inline-generate-error" style="color:#dc2626; font-size:12px; margin-top:6px; display:none;"></div>
                    </div>
                    <?php 
                    // Get upcoming titles from queue
                    $upcoming_titles = array_filter(explode("\n", get_option('my_auto_blog_title', '')));
                    // Server-side guarantee: if queue low, top-up then re-fetch
                    if (count($upcoming_titles) < 5) {
                        if (function_exists('auto_generate_titles_if_needed')) {
                            auto_generate_titles_if_needed();
                            $upcoming_titles = array_filter(explode("\n", get_option('my_auto_blog_title', '')));
                        }
                    }
                    $recent_posts = get_recent_ai_posts(3);
                    
                    // Show upcoming blog posts first
                    if (!empty($upcoming_titles)): 
                    ?>
                    <div style="margin-bottom: 20px;">
                        <h4 style="color: #6b7280; font-size: 14px; margin: 0 0 12px 0;">Upcoming Blog Posts</h4>
                        <div id="upcoming-posts-list">
                            <?php foreach (array_slice($upcoming_titles, 0, 5) as $index => $title): ?>
                            <div class="post-item upcoming" data-title="<?php echo esc_attr($title); ?>" style="display: flex; align-items: center; gap: 10px; padding: 12px; background: #f9fafb; border-radius: 8px; margin-bottom: 8px; border-left: 3px solid #6b7280;">
                                <div style="flex: 1; min-width: 0;">
                                    <div style="font-weight: 500; font-size: 14px; line-height: 1.4; color: #374151; word-wrap: break-word; overflow-wrap: break-word;"><?php echo esc_html($title); ?></div>
                                    <div style="font-size: 12px; color: #6b7280; margin-top: 2px;">
                                        <?php echo $index === 0 ? 'Next in queue' : 'Queued'; ?>
                                        <?php if ($posts_remaining > 0 || $posts_remaining === -1): ?>
                                            • Ready to generate
                                        <?php else: ?>
                                            • Requires upgrade
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div style="display:flex; gap:6px;">
                                    <button onclick="generateSpecificTitle('<?php echo esc_js($title); ?>')" style="color:#4f46e5; background:#f0f0ff; border:1px solid #e5e7eb; font-size: 12px; padding: 4px 8px; border-radius: 4px; cursor: pointer;">Generate now</button>
                                    <button onclick="deletePlannedTitle('<?php echo esc_js($title); ?>')" style="color:#6b7280; background:#ffffff; border:1px solid #e5e7eb; font-size: 12px; padding: 4px 8px; border-radius: 4px; cursor: pointer;">Delete this title</button>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        
                        <!-- Link to Content Planner -->
                        <div style="text-align: center; margin-top: 16px; padding-top: 16px;">
                            <a href="#" onclick="switchToContentPlannerTab(); return false;" style="color: #6b7280; font-size: 12px; text-decoration: none;">
                                📝 Manage titles in Content Planner →
                            </a>
                        </div>
                    </div>
                    <?php else: ?>
                    <!-- Empty state: Generate blog titles -->
                    <div style="margin-bottom: 20px; text-align: center; padding: 32px 20px; background: #f9fafb; border-radius: 8px; border: 2px dashed #d1d5db;">
                        <div style="font-size: 18px; margin-bottom: 8px;">📝</div>
                        <h4 style="color: #374151; margin: 0 0 8px 0; font-size: 16px;">No blog titles yet</h4>
                        <p style="color: #6b7280; margin: 0 0 20px 0; font-size: 14px;">Generate some blog titles to get started</p>
                        <button class="button button-primary" onclick="autoGenerateTitles()" style="margin-bottom: 12px;">
                            Generate blog titles
                        </button>
                        <div style="font-size: 12px; color: #6b7280;">
                            Or <a href="#" onclick="switchToContentPlannerTab(); return false;" style="color: #4f46e5;">set up manually in Content Planner →</a>
                        </div>
                    </div>
                    <?php endif; ?>
                    
                </div>
                
                <!-- Usage Info (shown only on Free/Lite to avoid duplication for paid) -->
                <?php if ($plan_name === 'lite'): ?>
                    <?php if ($posts_remaining > 0): ?>
                        <p style="color: #6b7280; font-size: 12px; margin: 0; text-align: center; padding: 12px; background: #f9fafb; border-radius: 6px;">
                            <?php echo $posts_remaining; ?> post<?php echo $posts_remaining !== 1 ? 's' : ''; ?> remaining this month
                        </p>
                    <?php elseif ($posts_remaining === -1): ?>
                        <p style="color: #059669; font-size: 12px; margin: 0; text-align: center; padding: 12px; background: #ecfdf5; border-radius: 6px;">
                            🚀 Unlimited posts available
                        </p>
                    <?php else: ?>
                        <p style="color: #dc2626; font-size: 12px; margin: 0; text-align: center; padding: 12px; background: #fef2f2; border-radius: 6px;">
                            You've used all <?php echo $posts_limit; ?> posts this month • <a href="#" onclick="openUpgradeModal()" style="color: #dc2626; font-weight: 500;">Upgrade to continue</a>
                        </p>
                    <?php endif; ?>
                <?php endif; ?>
                    
                    <?php /* email verification banner moved to top */ ?>
                </div>
                
                <!-- Keywords Section -->
                <div class="sica-card" style="margin-top: 20px;">
                    <?php 
                    $keywords = get_current_keywords();
                    $keyword_limit = ($plan_name === 'lite') ? 3 : -1; // Free: 3 keywords, Paid: unlimited
                    $is_keyword_limit_reached = ($keyword_limit > 0 && count($keywords) >= $keyword_limit);
                    $keywords_to_show = ($keyword_limit > 0) ? array_slice($keywords, 0, $keyword_limit) : $keywords;
                    ?>
                    <h3>Target Keywords (<?php echo count($keywords); ?><?php echo ($keyword_limit > 0) ? '/'.$keyword_limit : ''; ?>)</h3>
                    <div class="plg-keywords-chips">
                        <?php foreach ($keywords_to_show as $keyword): ?>
                        <div class="plg-keyword-chip">
                            <?php echo esc_html($keyword); ?>
                            <span onclick="removeKeyword('<?php echo esc_js($keyword); ?>')" style="cursor: pointer; margin-left: 4px;">×</span>
                        </div>
                        <?php endforeach; ?>
                        
                        <?php if (!$is_keyword_limit_reached): ?>
                        <div class="plg-keyword-chip plg-add-keyword" onclick="addKeyword()">
                            + Add keyword
                        </div>
                        <?php else: ?>
                        <div class="plg-keyword-chip plg-add-keyword disabled" title="Upgrade for unlimited keywords">
                            + Add keyword 🔒
                        </div>
                        <?php endif; ?>
                    </div>
                    <?php if ($is_keyword_limit_reached): ?>
                    <div style="font-size: 13px; color: #6b7280; margin-top: 8px;">
                        Upgrade for unlimited keywords
                        <a href="#" onclick="openUpgradeModal()" style="color: #4f46e5; font-weight: 600; margin-left: 4px;">See plans →</a>
                    </div>
                    <?php endif; ?>
                </div>
                
                <!-- Recently Created Posts -->
                <?php if (!empty($recent_posts)): ?>
                <div class="sica-card" style="margin-top: 20px;">
                    <h3>Recently Created</h3>
                    <div id="recent-posts-list">
                        <?php foreach ($recent_posts as $post): ?>
                        <div class="post-item completed" style="display: flex; align-items: center; gap: 10px; padding: 12px; background: #f0fdf4; border-radius: 8px; margin-bottom: 8px; border-left: 3px solid #059669;">
                            <?php if (!empty($post['thumb'])): ?>
                                <img src="<?php echo esc_url($post['thumb']); ?>" alt="" width="54" height="40" style="width:54px; height:40px; object-fit:cover; border-radius:4px; flex:0 0 auto;" />
                            <?php else: ?>
                                <div style="width:54px; height:40px; background:#e5e7eb; border-radius:4px; flex:0 0 auto;"></div>
                            <?php endif; ?>
                            <div style="flex: 1; min-width: 0;">
                                <div style="font-weight: 500; font-size: 14px; line-height: 1.4; word-wrap: break-word; overflow-wrap: break-word;"><?php echo esc_html($post['title']); ?></div>
                                <div style="font-size: 12px; color: #6b7280; margin-top: 2px;">
                                    <?php echo $post['date']; ?> • <?php echo $post['status']; ?>
                                </div>
                            </div>
                            <div style="display: flex; gap: 6px;">
                                <a href="<?php echo esc_url(get_permalink($post['id'])); ?>" target="_blank" style="color: #059669; text-decoration: none; font-size: 12px; padding: 4px 8px; border-radius: 4px; background: #dcfce7;">View ↗</a>
                                <a href="<?php echo esc_url(get_edit_post_link($post['id'])); ?>" style="color: #4f46e5; text-decoration: none; font-size: 12px; padding: 4px 8px; border-radius: 4px; background: #eef2ff;">Edit</a>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- RIGHT: Plan / Upgrade / Settings (1/3, sticky) -->
            <aside class="sica-side">
                <!-- Plan/Usage -->
                <div class="sica-card" style="margin-bottom:16px;">
                    <h3><?php echo $plan_display_name; ?></h3>
                    <p style="margin:.25rem 0;"><?php echo esc_html($posts_used); ?> of <?php echo esc_html($posts_limit); ?> used</p>
                    <div class="sica-meter"><span style="width:<?php echo $posts_limit > 0 ? min(100, ($posts_used/$posts_limit)*100) : 0; ?>%"></span></div>
                    <p style="margin:.5rem 0 0; color:#6b7280;">Resets: <?php echo esc_html($reset_date); ?></p>
                    <div style="display:flex; flex-direction:column; gap:8px; margin-top:12px;">
                        <?php if ($plan_name === 'lite'): ?>
                            <a class="button" href="https://sica.ai/pricing" target="_blank" rel="noopener" style="width:100%; text-align:center; background:#ffffff; color:#374151; border:1px solid #e5e7eb;">Upgrade to 12 posts/month</a>
                            <button class="button" onclick="showLicenseEntry()" style="width:100%; text-align:center; background:#ffffff; color:#374151; border:1px solid #e5e7eb;">Enter license key</button>
                        <?php elseif ($plan_name === 'starter'): ?>
                            <a class="button" href="https://sica.ai/pricing" target="_blank" rel="noopener" style="width:100%; text-align:center; background:#ffffff; color:#374151; border:1px solid #e5e7eb;">Upgrade to 30 posts/month</a>
                            <a class="button" href="<?php echo esc_url( admin_url('admin.php?page=ai-blogger&tab=setup#account-settings') ); ?>" style="width:100%; text-align:center; background:#ffffff; color:#374151; border:1px solid #e5e7eb;">Manage Account</a>
                        <?php else: ?>
                            <a class="button" href="<?php echo esc_url( admin_url('admin.php?page=ai-blogger&tab=setup#account-settings') ); ?>" style="width:100%; text-align:center; background:#ffffff; color:#374151; border:1px solid #e5e7eb;">Manage Account</a>
                            <a class="button" href="https://sica.ai/pricing" target="_blank" rel="noopener" style="width:100%; text-align:center; background:#ffffff; color:#374151; border:1px solid #e5e7eb;">View Plans</a>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Scheduler -->
                <?php if ($plan_name === 'lite'): ?>
                <div class="sica-card" style="margin-bottom:16px;">
                    <h3>Scheduling</h3>
                    <p style="margin:.25rem 0;">Auto-publish is available on Starter.</p>
                    <a class="button" href="https://sica.ai/pricing" target="_blank" rel="noopener">See plans</a>
                </div>
                <?php else: ?>
                <?php 
                // Get current scheduler settings
                $cron_active = get_option('my_auto_blog_cron_active', 0);
                $current_schedule = get_option('my_auto_blog_schedule', 'weekly');
                // Compute next scheduled run time
                $next_timestamp = wp_next_scheduled('my_auto_blog_cron_event');
                $upcoming_titles = array_filter(explode("\n", get_option('my_auto_blog_title', '')));
                
                if ($cron_active && $next_timestamp) {
                    // Format in site timezone
                    $next_run_text = wp_date('M j, g:i A', $next_timestamp);
                } elseif ($cron_active) {
                    if (empty($upcoming_titles)) {
                        $next_run_text = 'Add titles to queue first';
                    } else {
                        $next_run_text = 'Scheduling…';
                    }
                } else {
                    $next_run_text = 'Paused';
                }
                // Additional scheduler context
                $start_date = get_option('my_auto_blog_start_date', date('Y-m-d'));
                $start_time = get_option('my_auto_blog_start_time', '09:00');
                $timezone = get_option('timezone_string', 'UTC');
                ?>
                <div class="sica-card" style="margin-bottom:16px;">
                    <div style="display:flex; align-items:center; justify-content:space-between; gap:12px;">
                        <h3 style="margin:0;">Auto-generate posts</h3>
                        <div style="display:flex; align-items:center; gap:8px;">
                            <span class="sica-badge <?php echo $cron_active ? 'on' : ''; ?>"><?php echo $cron_active ? 'Running' : 'Paused'; ?></span>
                            <?php if ($cron_active): ?>
                                <button class="sica-text-link" onclick="toggleAutoSchedule(false)">Pause</button>
                            <?php else: ?>
                                <button class="sica-text-link" onclick="toggleAutoSchedule(true)">Start</button>
                            <?php endif; ?>
                        </div>
                    </div>

                    <div style="color:#6b7280; font-size:12px; margin:6px 0 12px 0; display:flex; align-items:center; gap:8px;">
                        <span><?php echo $cron_active ? 'Running • Next: ' . esc_html($next_run_text) : 'Paused'; ?></span>
                        <span aria-hidden="true">•</span>
                        <a href="#" onclick="switchToSchedulerTab(); return false;" class="sica-text-link" style="padding:0; color:#4f46e5;">Manage</a>
                    </div>

                    <?php $effective_schedule = $current_schedule ?: 'every_three_days'; ?>
                    <div style="font-size:12px; color:#6b7280;">Frequency</div>
                    <div class="sica-segment" style="margin: 8px 0 6px 0;">
                        <label class="seg <?php echo $effective_schedule==='daily'?'active':''; ?>">
                            <input type="radio" name="schedule-frequency" value="daily" <?php checked($effective_schedule, 'daily'); ?> onchange="onScheduleChange(this)">Daily
                        </label>
                        <label class="seg <?php echo $effective_schedule==='every_three_days'?'active':''; ?>">
                            <input type="radio" name="schedule-frequency" value="every_three_days" <?php checked($effective_schedule, 'every_three_days'); ?> onchange="onScheduleChange(this)">Every 3 days
                        </label>
                        <label class="seg <?php echo $effective_schedule==='weekly'?'active':''; ?>">
                            <input type="radio" name="schedule-frequency" value="weekly" <?php checked($effective_schedule, 'weekly'); ?> onchange="onScheduleChange(this)">Weekly
                        </label>
                        <label class="seg <?php echo $effective_schedule==='monthly'?'active':''; ?>">
                            <input type="radio" name="schedule-frequency" value="monthly" <?php checked($effective_schedule, 'monthly'); ?> onchange="onScheduleChange(this)">Monthly
                        </label>
                    </div>
                    <div id="schedule-save-status" class="sica-save-status" aria-live="polite"></div>

                    <!-- Details dropdown removed for cleaner UI -->
                </div>
                <?php endif; ?>
                
                <!-- Publish Settings -->
                <div class="sica-card" style="margin-bottom:16px;">
                    <h3>Publish Settings</h3>
                    <div style="margin-top: 12px;">
                        <div style="margin-bottom: 16px;">
                            <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                                <input type="checkbox" id="publish-immediately" <?php echo get_option('publish_immediately', 0) ? 'checked' : ''; ?> style="margin: 0;">
                                <span>Publish immediately</span>
                            </label>
                        </div>
                        <div>
                            <label for="post-author" style="display: block; margin-bottom: 4px; font-weight: 500;">Author:</label>
                            <select id="post-author" style="width: 100%; padding: 8px 12px; border: 1px solid #d1d5db; border-radius: 4px;">
                                <?php
                                $users = get_users(['role__in' => ['administrator', 'editor', 'author']]);
                                $current_author = get_option('my_auto_blog_author', 1);
                                foreach ($users as $user) {
                                    $selected = ($user->ID == $current_author) ? 'selected' : '';
                                    echo "<option value='{$user->ID}' {$selected}>{$user->display_name}</option>";
                                }
                                ?>
                            </select>
                        </div>
                    </div>
                </div>
                
                <!-- Notifications -->
                <div class="sica-card">
                    <h3>Notifications</h3>
                    <p style="margin:.25rem 0;">Get notified when posts are created</p>
                    <div style="margin-top: 12px;">
                        <div id="notification-chips" style="display:flex; flex-wrap:wrap; gap:6px; padding:8px; border:1px solid #d1d5db; border-radius:6px; background:#fff;">
                            <!-- Chips injected by JS -->
                            <input id="notification-input" type="email" placeholder="Add email…" 
                                   autocomplete="off" autocapitalize="none" spellcheck="false" 
                                   data-lpignore="true" data-form-type="other"
                                   style="flex:1; min-width:180px; border:none; outline:none; padding:6px; font-size:13px;" />
                        </div>
                        <div id="notification-error" style="color:#dc2626; font-size:12px; margin-top:6px; display:none;"></div>
                        <div style="display:flex; align-items:center; gap:8px; margin-top:10px;">
                            <span id="notification-hint" style="font-size:12px; color:#6b7280;">Press Enter to add; paste multiple to auto-add</span>
                            <span id="notification-save-status" style="font-size:12px; color:#6b7280;"></span>
                        </div>
                    </div>
                </div>
            </aside>
        </div> <!-- Close sica-grid -->
    
    </div> <!-- Close plg-container -->
    
    <!-- Upgrade Modal (Hidden) -->
    <div id="upgrade-modal" style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 10000;">
        <div style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); background: white; border-radius: 8px; padding: 30px; max-width: 600px; width: 90%;">
            <h2 style="margin: 0 0 20px 0; text-align: center;">Upgrade Your Plan</h2>
            
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 20px;">
                <!-- Starter Plan -->
                <div style="border: 2px solid #059669; border-radius: 8px; padding: 20px; text-align: center;">
                    <h3 style="color: #059669; margin: 0 0 10px 0;">Starter</h3>
                    <div style="font-size: 24px; font-weight: bold; margin-bottom: 15px;">$49/month</div>
                    <ul style="text-align: left; margin: 0; padding-left: 20px; list-style: none;">
                        <li style="margin-bottom: 8px;">✅ 12 posts/month</li>
                        <li style="margin-bottom: 8px;">✅ Auto-scheduling</li>
                        <li style="margin-bottom: 8px;">✅ Unlimited keywords</li>
                        <li style="margin-bottom: 8px;">✅ Non-watermarked images</li>
                        <li style="margin-bottom: 8px;">✅ Priority email support</li>
                    </ul>
                    <button onclick="upgradeToStarter()" style="background: #059669; color: white; border: none; padding: 12px 24px; border-radius: 6px; cursor: pointer; width: 100%; margin-top: 15px;">
                        Upgrade on sica.ai
                    </button>
                </div>
                
                <!-- Growth Plan -->
                <div style="border: 2px solid #dc2626; border-radius: 8px; padding: 20px; text-align: center;">
                    <h3 style="color: #dc2626; margin: 0 0 10px 0;">Growth</h3>
                    <div style="font-size: 24px; font-weight: bold; margin-bottom: 15px;">$99/month</div>
                    <ul style="text-align: left; margin: 0; padding-left: 20px; list-style: none;">
                        <li style="margin-bottom: 8px;">✅ 30 posts/month</li>
                        <li style="margin-bottom: 8px;">✅ Bulk title generation</li>
                        <li style="margin-bottom: 8px;">✅ Search indexing ping</li>
                        <li style="margin-bottom: 8px;">✅ Slack/Teams alerts</li>
                        <li style="margin-bottom: 8px;">✅ Everything in Starter</li>
                    </ul>
                    <button onclick="upgradeToGrowth()" style="background: #dc2626; color: white; border: none; padding: 12px 24px; border-radius: 6px; cursor: pointer; width: 100%; margin-top: 15px;">
                        Upgrade on sica.ai
                    </button>
                </div>
            </div>
            
            <div style="text-align: center; color: #6b7280; font-size: 14px; margin-bottom: 15px;">
                More indexed pages. More local visibility.
            </div>
            
            <button onclick="closeUpgradeModal()" style="background: #6b7280; color: white; border: none; padding: 8px 16px; border-radius: 4px; cursor: pointer; display: block; margin: 0 auto;">
                Close
            </button>
        </div>
    </div>
    
    <script>
    function openUpgradeModal() {
        document.getElementById('upgrade-modal').style.display = 'block';
    }
    
    function closeUpgradeModal() {
        document.getElementById('upgrade-modal').style.display = 'none';
    }
    
    function upgradeToStarter() {
        window.open('https://sica.ai/pricing?plan=starter', '_blank');
    }
    
    function upgradeToGrowth() {
        window.open('https://sica.ai/pricing?plan=growth', '_blank');
    }
    
    function generateCustomTitle() {
        if (!<?php echo $is_email_verified ? 'true' : 'false'; ?>) {
            alert('Please verify your email first');
            return;
        }
        // Deprecated prompt flow in favor of inline panel
        toggleGeneratePanel();
    }

    function generateSpecificTitle(title) {
        if (!<?php echo $is_email_verified ? 'true' : 'false'; ?>) {
            alert('Please verify your email first');
            return;
        }
        if (!title) return;
        generateFromTitle(title, true);
    }

    function generateFromTitle(title, removeFromQueue) {
        const button = document.querySelector('.plg-generate-btn');
        const originalText = button ? button.textContent : '';
        if (button) {
            button.disabled = true;
            button.textContent = 'Generating...';
            button.style.background = '#e8eaed';
        }
        // Starting content generation
        // Disable per-title generate buttons while a job is running
        setUpcomingGenerateDisabled(true);
        createProgressPlaceholder();

        const publishImmediately = document.getElementById('publish-immediately').checked;
        const authorId = document.getElementById('post-author').value;

        const body = new URLSearchParams({
            action: 'generate_blog_post_async',
            publish_immediately: publishImmediately ? '1' : '0',
            author_id: authorId,
            custom_title: title,
            remove_from_queue: removeFromQueue ? '1' : '0',
            _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
        });

        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body
        })
        .then(response => response.json())
        .then(data => {
            // Generation job queued
            if (data.success && data.data && data.data.job_id) {
                startPollingGeneration(data.data.job_id, title, publishImmediately, removeFromQueue, button, originalText);
            } else {
                errorProgress(data.data || 'Failed to generate post');
                setUpcomingGenerateDisabled(false);
                setTimeout(() => {
                    if (button) {
                        button.disabled = false;
                        button.textContent = originalText;
                        button.style.background = '#059669';
                    }
                }, 2000);
            }
        })
        .catch(error => {
            console.error('SICA: async enqueue network error', error);
            errorProgress('Network error: ' + error.message);
            setUpcomingGenerateDisabled(false);
            setTimeout(() => {
                if (button) {
                    button.disabled = false;
                    button.textContent = originalText;
                    button.style.background = '#059669';
                }
            }, 2000);
        });
    }

    function setUpcomingGenerateDisabled(disabled) {
        const genButtons = document.querySelectorAll('#upcoming-posts-list button[onclick^="generateSpecificTitle"]');
        genButtons.forEach(btn => {
            btn.disabled = !!disabled;
            if (disabled) {
                if (!btn.dataset.originalText) btn.dataset.originalText = btn.textContent;
                btn.textContent = 'Busy…';
                btn.style.opacity = '0.6';
                btn.style.cursor = 'not-allowed';
            } else {
                if (btn.dataset.originalText) {
                    btn.textContent = btn.dataset.originalText;
                    delete btn.dataset.originalText;
                }
                btn.style.opacity = '';
                btn.style.cursor = '';
            }
        });
    }

    function startPollingGeneration(jobId, requestedTitle, wasPublished, removeFromQueue, button, originalText) {
        let attempts = 0;
        const maxAttempts = 180; // 6 minutes (180 × 2s) - reduced timeout for faster failure detection
        const startTime = Date.now();
        
        const poll = () => {
            attempts++;
            const elapsed = Math.round((Date.now() - startTime) / 1000);
            
            fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: new URLSearchParams({
                    action: 'get_generation_status',
                    job_id: jobId,
                    _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
                })
            })
            .then(r => r.json())
            .then(resp => {
                if (!resp.success || !resp.data) {
                    if (attempts < maxAttempts) {
                        setTimeout(poll, 2000);
                    } else {
                        // Timeout reached - notify server to cleanup and refund
                        handleGenerationTimeout(jobId, requestedTitle, button, originalText, elapsed);
                    }
                    return;
                }
                
                // Check for immediate error state from failed AJAX response
                if (resp.data && resp.data.state === 'error') {
                    errorProgress(resp.data.error || 'Generation failed');
                    setUpcomingGenerateDisabled(false);
                    if (button) { button.disabled = false; button.textContent = originalText; button.style.background = '#059669'; }
                    return;
                }
                const { state, result, error } = resp.data;
                // Checking generation status
                
                if (state === 'pending') {
                    // For high-priority jobs, poll more frequently in the first 30 seconds
                    const pollInterval = elapsed < 30 ? 1000 : 2000;
                    
                    // Fallback: if job state isn't updating but post exists, mark as done
                    if (attempts > 15 && (attempts % 5 === 0)) { // start checking after ~30s, then every ~10s
                        // Fallback check by title
                        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                            body: new URLSearchParams({
                                action: 'find_post_by_title',
                                title: requestedTitle,
                                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
                            })
                        })
                        .then(r => r.json())
                        .then(check => {
                            if (check && check.success && check.data && check.data.found) {
                                // Fallback found post
                                const res = check.data.result || {};
                                completeProgress(res.post_title || requestedTitle, res.post_url || '', wasPublished, res.thumb_url || '');
                                if (removeFromQueue) {
                                    removeUpcomingFromUI(requestedTitle);
                                    removeFromBacklogSelector(requestedTitle);
                                }
                                setUpcomingGenerateDisabled(false);
                                if (button) { button.disabled = false; button.textContent = 'Generate another'; button.style.background = '#059669'; }
                                try { updateUsageUIAfterGeneration(); } catch(e){}
                                return; // stop polling
                            }
                            if (attempts < maxAttempts) setTimeout(poll, pollInterval); else handleGenerationTimeout(jobId, requestedTitle, button, originalText, elapsed);
                        })
                        .catch(() => {
                            if (attempts < maxAttempts) setTimeout(poll, pollInterval); else handleGenerationTimeout(jobId, requestedTitle, button, originalText, elapsed);
                        });
                        return;
                    }
                    if (attempts < maxAttempts) {
                        setTimeout(poll, pollInterval);
                    } else {
                        // Timeout reached - notify server to cleanup and refund
                        handleGenerationTimeout(jobId, requestedTitle, button, originalText, elapsed);
                    }
                    return;
                }
                
                if (state === 'processing') {
                    // Job is being processed
                    // Poll more frequently while processing, less frequently for longer jobs
                    const pollInterval = elapsed < 60 ? 1500 : 3000; 
                    if (attempts < maxAttempts) {
                        setTimeout(poll, pollInterval);
                    } else {
                        handleGenerationTimeout(jobId, requestedTitle, button, originalText, elapsed);
                    }
                    return;
                }
                
                if (state === 'error') {
                    errorProgress(error || 'Generation failed');
                    setUpcomingGenerateDisabled(false);
                    if (button) { button.disabled = false; button.textContent = originalText; button.style.background = '#059669'; }
                    return;
                }
                // done
                const postTitleResp = (result && (result.post_title || result.title)) || requestedTitle || 'Blog Post Generated';
                const postUrlResp = result && (result.post_url || result.url);
                const thumbUrlResp = result && result.thumb_url;
                completeProgress(postTitleResp, postUrlResp, wasPublished, thumbUrlResp);
                if (removeFromQueue) {
                    removeUpcomingFromUI(requestedTitle);
                    removeFromBacklogSelector(requestedTitle);
                }
                setTimeout(() => {
                    if (button) {
                        button.disabled = false;
                        button.textContent = 'Generate another';
                        button.style.background = '#059669';
                    }
                }, 800);
                try { updateUsageUIAfterGeneration(); } catch(e){}
                setUpcomingGenerateDisabled(false);
            })
            .catch(() => {
                if (attempts < maxAttempts) {
                    setTimeout(poll, 2500); // Slightly longer delay for network errors
                } else {
                    handleGenerationTimeout(jobId, requestedTitle, button, originalText, elapsed);
                }
            });
        };
        poll();
    }

    function handleGenerationTimeout(jobId, requestedTitle, button, originalText, elapsedSeconds) {
        console.warn(`Generation timeout after ${elapsedSeconds}s for job ${jobId}`);
        
        // Show user-friendly timeout message
        errorProgress(`Generation took too long (${Math.round(elapsedSeconds/60)} minutes). Your credit has been refunded.`);
        
        // Notify server to cleanup and refund credit
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: new URLSearchParams({
                action: 'cancel_generation',
                job_id: jobId,
                reason: 'timeout',
                elapsed_seconds: elapsedSeconds,
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(r => r.json())
        .then(resp => {
            if (resp.success && resp.data && resp.data.refunded) {
                // Credit refunded for timed out generation
                // Update usage UI to reflect the refund
                try { updateUsageUIAfterGeneration(-1); } catch(e) {} // -1 indicates refund
            }
        })
        .catch(err => {
            console.error('Failed to notify server of timeout:', err);
        });
        
        // Reset UI state
        setUpcomingGenerateDisabled(false);
        if (button) {
            button.disabled = false;
            button.textContent = originalText;
            button.style.background = '#059669';
        }
    }

    function deletePlannedTitle(title) {
        if (!title) return;
        if (!confirm('Delete this planned title?')) return;
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: new URLSearchParams({
                action: 'delete_planned_title',
                title: title,
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Remove from DOM
                const items = document.querySelectorAll('.post-item.upcoming');
                items.forEach(item => {
                    if (item.getAttribute('data-title') === title) {
                        item.remove();
                    }
                });
            } else {
                alert(data.data || 'Failed to delete title');
            }
        })
        .catch(err => alert('Network error: ' + err.message));
    }
    
    function switchToContentPlannerTab() {
        // Find and click the Content Planner tab
        const contentPlannerTab = document.querySelector('a[href="#content-planner"]');
        if (contentPlannerTab) {
            contentPlannerTab.click();
        } else {
            // Fallback: try to find by tab text
            const tabs = document.querySelectorAll('.nav-tab');
            tabs.forEach(tab => {
                if (tab.textContent.includes('Content Planner')) {
                    tab.click();
                }
            });
        }
    }

    function switchToSchedulerTab() {
        // Tabs are query-based (?page=ai-blogger&tab=scheduler). Try to find by href.
        const schedulerTab = Array.from(document.querySelectorAll('.nav-tab')).find(tab =>
            tab.getAttribute('href') && tab.getAttribute('href').includes('tab=scheduler')
        );
        if (schedulerTab) {
            schedulerTab.click();
            return;
        }
        // Fallback: try to find by visible text
        const tabs = document.querySelectorAll('.nav-tab');
        for (const tab of tabs) {
            if (tab.textContent.trim().toLowerCase() === 'scheduler') {
                tab.click();
                return;
            }
        }
        // Absolute fallback: navigate directly
        window.location.href = '?page=ai-blogger&tab=scheduler';
    }
    
    function toggleAutoSchedule(enable) {
        // Get selected frequency
        const selectedFrequency = document.querySelector('input[name="schedule-frequency"]:checked');
        if (!selectedFrequency) {
            alert('Please select a scheduling frequency first');
            return;
        }
        
        const frequency = selectedFrequency.value;
        const action = enable ? 'enable' : 'disable';
        
        // Show loading state
        const button = event.target;
        const originalText = button.textContent;
        button.disabled = true;
        button.textContent = enable ? 'Enabling...' : 'Disabling...';
        
        // Make AJAX call to save settings
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'toggle_auto_schedule',
                enable: enable ? '1' : '0',
                frequency: frequency,
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Reload page to update UI
                location.reload();
            } else {
                alert('Error: ' + (data.data || 'Failed to update schedule settings'));
                // Reset button
                button.disabled = false;
                button.textContent = originalText;
            }
        })
        .catch(error => {
            alert('Network error: ' + error.message);
            // Reset button
            button.disabled = false;
            button.textContent = originalText;
        });
    }

    function onScheduleChange(input) {
        // Update segmented control active class
        document.querySelectorAll('.sica-segment .seg').forEach(seg => seg.classList.remove('active'));
        const label = input.closest('.seg');
        if (label) label.classList.add('active');

        // If auto-schedule is running, auto-save frequency change
        const isRunning = <?php echo get_option('my_auto_blog_cron_active', 0) ? 'true' : 'false'; ?>;
        if (isRunning) {
            const statusEl = document.getElementById('schedule-save-status');
            if (statusEl) {
                statusEl.textContent = 'Saving…';
                statusEl.className = 'sica-save-status show';
            }
            fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: new URLSearchParams({
                    action: 'toggle_auto_schedule',
                    enable: '1',
                    frequency: input.value,
                    _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
                })
            })
            .then(r => r.json())
            .then(resp => {
                if (statusEl) {
                    statusEl.textContent = 'Saved';
                    statusEl.className = 'sica-save-status show success';
                    setTimeout(() => { statusEl.className = 'sica-save-status'; statusEl.textContent = ''; }, 1500);
                }
            })
            .catch(() => {
                if (statusEl) {
                    statusEl.textContent = 'Failed to save';
                    statusEl.className = 'sica-save-status show error';
                    setTimeout(() => { statusEl.className = 'sica-save-status'; statusEl.textContent = ''; }, 2000);
                }
            });
        }
    }
    
    function autoGenerateTitles() {
        if (!<?php echo $is_email_verified ? 'true' : 'false'; ?>) {
            alert('Please verify your email first');
            return;
        }
        
        const button = event.target;
        const originalText = button.textContent;
        button.disabled = true;
        button.textContent = 'Generating titles...';
        
        // Make AJAX call to auto-generate titles
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: new URLSearchParams({
                action: 'auto_generate_titles',
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Reload page to show new titles
                location.reload();
            } else {
                alert('Error generating titles: ' + (data.data || 'Please try again'));
                button.disabled = false;
                button.textContent = originalText;
            }
        })
        .catch(error => {
            alert('Network error: ' + error.message);
            button.disabled = false;
            button.textContent = originalText;
        });
    }
    
    function fixSchedule() {
        const button = event.target;
        const originalText = button.textContent;
        button.disabled = true;
        button.textContent = 'Fixing...';
        
        // Make AJAX call to fix schedule
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: new URLSearchParams({
                action: 'fix_schedule',
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Reload page to show updated schedule
                location.reload();
            } else {
                alert('Error fixing schedule: ' + (data.data || 'Please try again'));
                button.disabled = false;
                button.textContent = originalText;
            }
        })
        .catch(error => {
            alert('Network error: ' + error.message);
            button.disabled = false;
            button.textContent = originalText;
        });
    }
    
    function createProgressPlaceholder() {
        const postsContainer = document.getElementById('posts-container');
        const upcomingPostsList = document.getElementById('upcoming-posts-list');
        const recentPostsList = document.getElementById('recent-posts-list');
        
        // Ensure no stale progress placeholder exists (from prior timeout/error)
        const existing = document.getElementById('progress-placeholder');
        if (existing) {
            try { existing.remove(); } catch(e) {}
        }

        // Create progress placeholder
        const progressPlaceholder = document.createElement('div');
        progressPlaceholder.id = 'progress-placeholder';
        progressPlaceholder.className = 'plg-progress-post';
        progressPlaceholder.innerHTML = `
            <div class="plg-progress-header">
                <div class="plg-progress-icon">⚡</div>
                <div class="plg-progress-title">Generating New Blog Post...</div>
                <div class="plg-progress-percentage">0%</div>
            </div>
            <div class="plg-progress-bar">
                <div class="plg-progress-fill" style="width: 0%;"></div>
            </div>
            <div class="plg-progress-step">
                <span class="plg-progress-step-icon">🤖</span>
                <span id="progress-step-text">Starting AI generation...</span>
                <span id="progress-elapsed" style="margin-left:auto;color:#6b7280;font-size:12px;">Elapsed: 0s</span>
                <button class="plg-progress-cancel" onclick="cancelGeneration()">Cancel</button>
            </div>
        `;
        
        // Insert at the very top of posts container
        if (upcomingPostsList) {
            // Insert before upcoming posts section
            const upcomingSection = upcomingPostsList.closest('div[style*="margin-bottom: 20px"]');
            postsContainer.insertBefore(progressPlaceholder, upcomingSection);
        } else if (recentPostsList) {
            // Insert before recent posts section
            const recentSection = recentPostsList.closest('div[style*="margin-bottom: 20px"]');
            postsContainer.insertBefore(progressPlaceholder, recentSection);
        } else {
            // Insert at beginning
            postsContainer.insertBefore(progressPlaceholder, postsContainer.firstChild);
        }
        // Progress placeholder created
        
        // Start the progress simulation
        startProgressSimulation();
    }

            // Ensure the queue is topped up on load if it's running low
            (function ensureTitleQueueToppedUp(){
                try {
                    const upcomingList = document.getElementById('upcoming-posts-list');
                    const currentCount = upcomingList ? upcomingList.querySelectorAll('.post-item.upcoming').length : 0;
                    if (currentCount < 5) {
                        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                            body: new URLSearchParams({
                                action: 'auto_top_up_titles_if_needed',
                                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
                            })
                        }).then(r => r.json()).then(resp => {
                            // If titles were topped up, reload to display new queue
                            if (resp && resp.success && resp.data && resp.data.topped_up) {
                                location.reload();
                            }
                        }).catch(() => {});
                    }
                } catch(e) {}
            })();

    // Inline Generate Panel logic
    function toggleGeneratePanel() {
        const panel = document.getElementById('inline-generate-panel');
        if (!panel) return;
        const isOpen = panel.style.display !== 'none';
        panel.style.display = isOpen ? 'none' : 'block';
        if (!isOpen) {
            const input = document.getElementById('inline-generate-input');
            if (input) input.focus();
        }
    }

    function submitInlineGenerate(fromBacklog) {
        const errorEl = document.getElementById('inline-generate-error');
        if (errorEl) { errorEl.style.display = 'none'; errorEl.textContent = ''; }
        let title = '';
        if (fromBacklog) {
            const select = document.getElementById('inline-generate-select');
            title = select ? select.value : '';
        } else {
            const input = document.getElementById('inline-generate-input');
            title = input ? input.value.trim() : '';
        }
        if (!title) {
            if (errorEl) { errorEl.textContent = 'Please enter or select a title.'; errorEl.style.display = 'block'; }
            return;
        }
        const removeFromQueue = !!fromBacklog;
        toggleGeneratePanel();
        generateFromTitle(title, removeFromQueue);
    }

    // Remove a title from the Upcoming UI list
    function removeUpcomingFromUI(title) {
        const items = document.querySelectorAll('#upcoming-posts-list .post-item.upcoming');
        let removed = false;
        items.forEach(item => {
            const t = item.getAttribute('data-title') || '';
            if (!removed && t === title) {
                item.parentNode.removeChild(item);
                removed = true;
            }
        });
    }

    // Remove a title from the inline backlog selector if present
    function removeFromBacklogSelector(title) {
        const select = document.getElementById('inline-generate-select');
        if (!select) return;
        const opts = Array.from(select.options);
        for (const opt of opts) {
            if (opt.value === title) {
                opt.remove();
                break;
            }
        }
    }
    
    let progressInterval;
    let currentProgress = 0;
    
    function startProgressSimulation(targetDurationMs = 80000) {
        const milestones = [
            { at: 8,  icon: '🧠', text: 'Analyzing business context...' },
            { at: 22, icon: '🔍', text: 'Researching trending topics...' },
            { at: 36, icon: '✍️', text: 'Generating article content...' },
            { at: 50, icon: '📝', text: 'Optimizing for SEO...' },
            { at: 65, icon: '🎨', text: 'Creating featured image (may take a few mins)...' },
            { at: 92, icon: '✨', text: 'Creating featured image (still working)...' }
        ];
        let milestoneIndex = 0;
        currentProgress = 0;
        updateProgress(0, '⚡', 'Starting AI generation...');

        const start = performance.now();
        // Start elapsed timer display
        startElapsedTimer();

        // Piecewise linear phases for intuitive pacing
        // Each item: { to: progress%, share: fraction of total duration used to reach this segment }
        // Edit these two arrays to tune speed feel without touching code below
        const phases = [
            { to: 20, share: 0.25 },
            { to: 60, share: 0.40 },
            { to: 95, share: 0.35 }
        ];
        // Precompute absolute times per phase based on targetDurationMs
        const phaseTimes = phases.map(p => p.share * targetDurationMs);
        const phaseEnds = [];
        let acc = 0;
        for (let i = 0; i < phaseTimes.length; i++) {
            acc += phaseTimes[i];
            phaseEnds.push(acc);
        }

        function tick(now) {
            const elapsed = now - start;
            const cappedElapsed = Math.min(elapsed, targetDurationMs);

            // Determine current phase
            let phaseIndex = 0;
            while (phaseIndex < phaseEnds.length && cappedElapsed > phaseEnds[phaseIndex]) {
                phaseIndex++;
            }

            // Compute progress linearly within the phase
            let progress = 0;
            if (phaseIndex === 0) {
                const tIn = cappedElapsed;
                const d = phaseEnds[0];
                progress = (tIn / d) * phases[0].to;
            } else if (phaseIndex < phases.length) {
                const prevEnd = phaseEnds[phaseIndex - 1];
                const tIn = cappedElapsed - prevEnd;
                const d = phaseEnds[phaseIndex] - prevEnd;
                const fromPct = phases[phaseIndex - 1].to;
                const toPct = phases[phaseIndex].to;
                const local = d > 0 ? (tIn / d) : 1;
                progress = fromPct + local * (toPct - fromPct);
            } else {
                progress = phases[phases.length - 1].to;
            }

            currentProgress = Math.round(Math.max(0, Math.min(95, progress)));

            const m = milestones[milestoneIndex];
            if (m && currentProgress >= m.at) {
                updateProgress(currentProgress, m.icon, m.text);
                milestoneIndex++;
            } else {
                updateProgress(currentProgress, undefined, undefined);
            }

            if (cappedElapsed < targetDurationMs) {
                progressInterval = requestAnimationFrame(tick);
            }
        }
        progressInterval = requestAnimationFrame(tick);
    }
    
    function updateProgress(percentage, icon, text) {
        const progressFill = document.querySelector('#progress-placeholder .plg-progress-fill');
        const progressPercentage = document.querySelector('#progress-placeholder .plg-progress-percentage');
        const progressIcon = document.querySelector('#progress-placeholder .plg-progress-step-icon');
        const progressText = document.getElementById('progress-step-text');
        
        if (progressFill) progressFill.style.width = percentage + '%';
        if (progressPercentage) progressPercentage.textContent = percentage + '%';
        if (progressIcon && icon !== undefined) progressIcon.textContent = icon;
        if (progressText && text !== undefined) progressText.textContent = text;
        
        currentProgress = percentage;
    }

    // Update usage UI ("X of Y used" and meter) after generation or cancellation
    function updateUsageUIAfterGeneration(increment = 1) {
        // Update count text
        const usageTextP = document.querySelector('.sica-side .sica-card p');
        if (usageTextP) {
            const match = usageTextP.textContent.match(/(\d+) of (\d+) used/);
            if (match) {
                let used = parseInt(match[1], 10);
                const total = parseInt(match[2], 10);
                if (!isNaN(used) && !isNaN(total)) {
                    used += increment; // Can be +1 for generation, -1 for refund
                    used = Math.max(0, Math.min(used, total)); // Clamp between 0 and total
                    usageTextP.textContent = `${used} of ${total} used`;
                    // Update meter width
                    const meterFill = document.querySelector('.sica-meter span');
                    if (meterFill) {
                        const pct = Math.min(100, (used / total) * 100);
                        meterFill.style.width = pct + '%';
                    }
                }
            }
        }
    }
    
    function stopProgressSimulation() {
        if (typeof progressInterval === 'number') {
            try { cancelAnimationFrame(progressInterval); } catch(e){}
            try { clearInterval(progressInterval); } catch(e){}
        }
        progressInterval = null;
        stopElapsedTimer();
    }

    function completeProgress(postTitle, postUrl, wasPublished, thumbUrl) {
        stopProgressSimulation();
        
        // Final progress update
        updateProgress(100, '✅', 'Post generated successfully!');
        
        // Transform to completed post after 1 second
        setTimeout(() => {
            const progressPlaceholder = document.getElementById('progress-placeholder');
            if (progressPlaceholder) {
                const statusText = wasPublished ? 'Published' : 'Draft';
                const viewButton = postUrl ? `<a href="${postUrl}" target="_blank" style="color: #059669; text-decoration: none; font-size: 12px; padding: 4px 8px; border-radius: 4px; background: #dcfce7;">View ↗</a>` : '';
                // Derive edit link if we can map URL to edit path
                const editButton = postUrl ? `<a href="${postUrl.replace(/\/?$/, '/edit')}" style="color: #4f46e5; text-decoration: none; font-size: 12px; padding: 4px 8px; border-radius: 4px; background: #eef2ff;">Edit</a>` : '';
                const thumbHtml = thumbUrl
                    ? `<img src="${thumbUrl}" alt="" width="54" height="40" style="width:54px; height:40px; object-fit:cover; border-radius:4px; flex:0 0 auto;" />`
                    : '<div style="width:54px; height:40px; background:#e5e7eb; border-radius:4px; flex:0 0 auto;"></div>';
                progressPlaceholder.innerHTML = `
                    <div class="post-item completed" style="display: flex; align-items: center; gap: 10px; padding: 12px; background: #f0fdf4; border-radius: 8px; margin-bottom: 8px; border-left: 3px solid #059669;">
                        ${thumbHtml}
                        <div style="flex: 1; min-width: 0;">
                            <div style="font-weight: 500; font-size: 14px; line-height: 1.4; word-wrap: break-word; overflow-wrap: break-word;">${postTitle}</div>
                            <div style="font-size: 12px; color: #6b7280; margin-top: 2px;">Just now • ${statusText}</div>
                        </div>
                        <div style="display: flex; gap: 6px;">${viewButton}${editButton ? ' ' + editButton : ''}</div>
                    </div>
                `;
                progressPlaceholder.className = '';
                progressPlaceholder.style.background = '';
                progressPlaceholder.style.borderLeftColor = '';
                progressPlaceholder.id = 'new-post-item';
            }
        }, 1000);
    }

    // Elapsed timer utilities for planner progress UI
    let elapsedTimerHandle = null;
    let elapsedStartMs = 0;
    function formatElapsedMs(ms) {
        const totalSeconds = Math.floor(ms / 1000);
        const minutes = Math.floor(totalSeconds / 60);
        const seconds = totalSeconds % 60;
        return minutes + 'm ' + (seconds < 10 ? '0' + seconds : seconds) + 's';
    }
    function startElapsedTimer() {
        elapsedStartMs = Date.now();
        const el = document.getElementById('progress-elapsed');
        if (!el) return;
        if (elapsedTimerHandle) { try { clearInterval(elapsedTimerHandle); } catch(e){} }
        el.textContent = 'Elapsed: 0s';
        elapsedTimerHandle = setInterval(() => {
            const diff = Date.now() - elapsedStartMs;
            const el2 = document.getElementById('progress-elapsed');
            if (!el2) return;
            el2.textContent = 'Elapsed: ' + formatElapsedMs(diff);
        }, 1000);
    }
    function stopElapsedTimer() {
        if (elapsedTimerHandle) {
            try { clearInterval(elapsedTimerHandle); } catch(e){}
            elapsedTimerHandle = null;
        }
    }
    
    function errorProgress(errorMessage) {
        stopProgressSimulation();
        
        const progressPlaceholder = document.getElementById('progress-placeholder');
        if (progressPlaceholder) {
            progressPlaceholder.innerHTML = `
                <div style="display: flex; align-items: center; gap: 10px;">
                    <span style="color: #dc2626; font-size: 16px;">❌</span>
                    <div style="flex: 1;">
                        <div style="font-weight: 500; font-size: 14px; color: #dc2626;">Generation Failed</div>
                        <div style="font-size: 12px; color: #6b7280; margin-top: 2px;">${errorMessage}</div>
                    </div>
                    <button onclick="removeProgressPlaceholder()" style="color: #6b7280; background: none; border: none; font-size: 12px; cursor: pointer;">Remove</button>
                </div>
            `;
            progressPlaceholder.style.background = '#fef2f2';
            progressPlaceholder.style.borderLeftColor = '#dc2626';
        }
        // Re-enable per-title generate buttons
        setUpcomingGenerateDisabled(false);
    }
    
    function cancelGeneration() {
        if (confirm('Cancel blog post generation?')) {
            stopProgressSimulation();
            removeProgressPlaceholder();
            
            // Reset generate button
            const button = document.querySelector('.plg-generate-btn');
            if (button) {
                button.disabled = false;
                button.textContent = '🚀 Generate a post now';
                button.style.background = '#059669';
            }
            setUpcomingGenerateDisabled(false);
        }
    }
    
    function removeProgressPlaceholder() {
        const progressPlaceholder = document.getElementById('progress-placeholder');
        if (progressPlaceholder) {
            progressPlaceholder.style.animation = 'slideOutToTop 0.3s ease-in forwards';
            setTimeout(() => {
                progressPlaceholder.remove();
            }, 300);
        }
    }
    
    function addKeyword() {
        const keyword = prompt('Enter a keyword:');
        if (keyword && keyword.trim()) {
            // Save keyword via AJAX
            // Adding keyword
        }
    }
    
    function removeKeyword(keyword) {
        if (confirm('Remove "' + keyword + '"?')) {
            // Remove keyword via AJAX
            // Removing keyword
        }
    }
    
    function sendVerificationCode() {
        const email = document.getElementById('verification-email').value.trim();
        const button = document.querySelector('button[onclick="sendVerificationCode()"]');
        
        if (!email) {
            alert('Please enter your email address');
            return;
        }
        
        // Simple email validation
        const emailPattern = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailPattern.test(email)) {
            alert('Please enter a valid email address');
            return;
        }
        
        // Disable button and show loading state
        if (button) {
            button.disabled = true;
            button.textContent = 'Sending...';
        }
        
        // Send AJAX request
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: new URLSearchParams({
                action: 'send_verification',
                email: email,
                marketing_consent: false, // No marketing consent checkbox on Overview tab
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('✅ ' + data.data.message);
                // Reload page to show verified state
                location.reload();
            } else {
                alert('❌ ' + (data.data || 'Failed to send verification code'));
            }
        })
        .catch(error => {
            alert('❌ Network error. Please try again.');
            console.error('Verification error:', error);
        })
        .finally(() => {
            // Re-enable button
            if (button) {
                button.disabled = false;
                button.textContent = 'Send code';
            }
        });
    }
    
    function saveNotificationEmails() {
        const emails = Array.from(document.querySelectorAll('#notification-chips .chip'))
            .map(chip => chip.getAttribute('data-email'))
            .join(',');
        const statusEl = document.getElementById('notification-save-status');
        if (statusEl) {
            statusEl.textContent = 'Saving…';
        }
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: new URLSearchParams({
                action: 'save_notification_emails',
                emails: emails,
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(response => response.json())
        .then(data => {
            if (statusEl) {
                statusEl.textContent = data.success ? 'Saved' : 'Failed to save';
                setTimeout(() => { statusEl.textContent = ''; }, 1500);
            }
            if (!data.success) {
                alert('Error saving emails: ' + (data.data || 'Please try again'));
            }
        })
        .catch(error => {
            if (statusEl) {
                statusEl.textContent = 'Network error';
                setTimeout(() => { statusEl.textContent = ''; }, 2000);
            }
        });
    }

    // Notifications: chips input behavior
    (function initNotificationChips() {
        const stored = '<?php echo esc_js(get_option('my_auto_blog_notify_emails', '')); ?>';
        const initialEmails = stored ? stored.split(',').map(e => e.trim()).filter(Boolean) : [];
        const chipsContainer = document.getElementById('notification-chips');
        const input = document.getElementById('notification-input');
        const errorEl = document.getElementById('notification-error');

        function isValidEmail(email) {
            // Simple validation; server re-validates
            return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
        }

        function renderChip(email) {
            const chip = document.createElement('span');
            chip.className = 'chip';
            chip.setAttribute('data-email', email);
            chip.style.cssText = 'display:inline-flex; align-items:center; gap:6px; padding:4px 8px; background:#f3f4f6; color:#374151; border:1px solid #e5e7eb; border-radius:999px; font-size:12px;';
            chip.innerHTML = `<span>${email}</span><button type="button" aria-label="Remove" style="border:none;background:transparent;color:#6b7280;cursor:pointer;padding:0;line-height:1;">×</button>`;
            chip.querySelector('button').addEventListener('click', () => {
                chipsContainer.removeChild(chip);
                saveNotificationEmails();
                input.focus();
            });
            chipsContainer.insertBefore(chip, input);
        }

        function addEmail(email) {
            const normalized = email.trim();
            if (!normalized) return;
            if (!isValidEmail(normalized)) {
                errorEl.textContent = `Invalid email: ${normalized}`;
                errorEl.style.display = 'block';
                return;
            }
            errorEl.style.display = 'none';
            // prevent duplicates
            const existing = Array.from(chipsContainer.querySelectorAll('.chip'))
                .some(ch => ch.getAttribute('data-email').toLowerCase() === normalized.toLowerCase());
            if (existing) return;
            renderChip(normalized);
            saveNotificationEmails();
        }

        // Seed initial chips
        initialEmails.forEach(addEmail);

        function tokenizeAndAdd(text) {
            const parts = text.split(/[\n,;\s]+/).map(p => p.trim()).filter(Boolean);
            parts.forEach(addEmail);
        }

        input.addEventListener('keydown', (e) => {
            if (e.key === 'Enter' || e.key === ',') {
                e.preventDefault();
                addEmail(input.value);
                input.value = '';
            } else if (e.key === 'Backspace' && !input.value) {
                // Remove last chip
                const chips = chipsContainer.querySelectorAll('.chip');
                if (chips.length) {
                    chipsContainer.removeChild(chips[chips.length - 1]);
                    saveNotificationEmails();
                }
            }
        });

        input.addEventListener('paste', (e) => {
            e.preventDefault();
            const text = (e.clipboardData || window.clipboardData).getData('text');
            tokenizeAndAdd(text);
            saveNotificationEmails();
        });
    })();
    
    function showLicenseEntry() {
        const licenseKey = prompt('Enter your license key:');
        if (!licenseKey || !licenseKey.trim()) {
            return;
        }
        
        // Show loading
        showNotification('⏳ Activating license...', 'info');
        
        // Make AJAX call to activate license
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'activate_license_key',
                license_key: licenseKey.trim(),
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showSuccessModal('✅ License activated successfully!');
            } else {
                showNotification('❌ License activation failed: ' + (data.data || 'Invalid license key'), 'error');
            }
        })
        .catch(error => {
            showNotification('❌ Network error: ' + error.message, 'error');
        });
    }
    
    function showSuccessModal(message) {
        // Create modal overlay
        const overlay = document.createElement('div');
        overlay.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.8);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 999999;
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Arial, sans-serif;
        `;
        
        // Create modal content
        const modal = document.createElement('div');
        modal.style.cssText = `
            background: white;
            padding: 40px;
            border-radius: 12px;
            text-align: center;
            max-width: 400px;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
            animation: modalSlideIn 0.3s ease-out;
        `;
        
        // Add CSS animation
        const style = document.createElement('style');
        style.textContent = `
            @keyframes modalSlideIn {
                from { transform: translateY(-50px); opacity: 0; }
                to { transform: translateY(0); opacity: 1; }
            }
        `;
        document.head.appendChild(style);
        
        modal.innerHTML = `
            <div style="font-size: 24px; margin-bottom: 20px;">${message}</div>
            <div style="font-size: 18px; color: #666; margin-bottom: 10px;">Updating your plan...</div>
            <div id="countdown" style="font-size: 16px; color: #198754; font-weight: bold;">Reloading in 3...</div>
        `;
        
        overlay.appendChild(modal);
        document.body.appendChild(overlay);
        
        // Countdown timer
        let count = 3;
        const countdownEl = document.getElementById('countdown');
        
        const timer = setInterval(() => {
            count--;
            if (count > 0) {
                countdownEl.textContent = `Reloading in ${count}...`;
            } else {
                countdownEl.textContent = 'Reloading now!';
                clearInterval(timer);
                // Force reload
                window.location.reload(true);
            }
        }, 1000);
    }
    
    function showNotification(message, type = 'info') {
        // Remove existing notifications
        const existing = document.querySelector('.plg-notification');
        if (existing) {
            existing.remove();
        }
        
        // Create notification
        const notification = document.createElement('div');
        notification.className = 'plg-notification';
        notification.innerHTML = message;
        
        // Style based on type
        const styles = {
            success: 'background: #d4edda; color: #155724; border: 1px solid #c3e6cb;',
            error: 'background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb;',
            info: 'background: #d1ecf1; color: #0c5460; border: 1px solid #bee5eb;'
        };
        
        notification.style.cssText = styles[type] + 
            'position: fixed; top: 20px; right: 20px; padding: 12px 20px; border-radius: 6px; ' +
            'font-weight: 500; z-index: 10001; box-shadow: 0 2px 8px rgba(0,0,0,0.15);';
        
        document.body.appendChild(notification);
        
        // Auto-remove after 5 seconds
        setTimeout(() => {
            if (notification.parentNode) {
                notification.remove();
            }
        }, 5000);
    }
    
    // Close modal when clicking outside
    document.getElementById('upgrade-modal').addEventListener('click', function(e) {
        if (e.target === this) {
            closeUpgradeModal();
        }
    });
    </script>
    
    <?php
}

/**
 * Get current keywords array
 */
function get_current_keywords() {
    $keywords = get_option('my_auto_blog_keywords', '');
    if (empty($keywords)) {
        return array();
    }
    
    $keywords_array = explode("\n", $keywords);
    return array_filter(array_map('trim', $keywords_array));
}

/**
 * Auto-generate titles when queue drops below 5
 */
function auto_generate_titles_if_needed() {
    $titles = get_option('my_auto_blog_title', '');
    $titles_array = array_filter(array_map('trim', explode("\n", (string) $titles)));
    
    // Only generate if we have fewer than 5 titles
    if (count($titles_array) < 5) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Auto-generating 5 more titles for queue'); }
        
        // Get business details for title generation
        $keywords_string = get_option('my_auto_blog_keywords', '');
        $business_details = array(
            'business_name' => get_bloginfo('name'),
            'business_category' => get_option('my_auto_blog_business_category', ''),
            'business_service_area' => get_option('my_auto_blog_service_area', ''),
            'products_services' => get_option('my_auto_blog_products_services', '')
        );
        
        // Generate 5 new titles
        $new_titles = call_chatgpt_to_generate_titles_with_current_data(
            5, 
            $keywords_string, 
            $business_details, 
            $titles_array
        );
        
        if (!empty($new_titles)) {
            // Add new titles to existing ones
            $all_titles = array_merge($titles_array, $new_titles);
            update_option('my_auto_blog_title', implode("\n", $all_titles));
            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Auto-generated more titles for queue'); }
            return true;
        } else {
            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Failed to auto-generate new titles'); }
            return false;
        }
    }
    
    return false; // No generation needed
}

/**
 * AJAX handler for generating blog posts from Overview tab
 */
function ajax_generate_blog_post() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    
    // Check subscription status
    $subscription_data = sica_check_subscription_status();
    if (!$subscription_data || !$subscription_data['ok']) {
        wp_send_json_error('Subscription verification failed');
        return;
    }
    
    // Check if posts remaining
    $posts_used = $subscription_data['posts_used'] ?? 0;
    $posts_limit = $subscription_data['posts_limit'] ?? 2;
    
    if ($posts_used >= $posts_limit) {
        wp_send_json_error('Monthly post limit reached. Upgrade to continue publishing.');
        return;
    }
    
    // Get settings
    $publish_immediately = isset($_POST['publish_immediately']) && $_POST['publish_immediately'] === '1';
    $author_id = isset($_POST['author_id']) ? intval($_POST['author_id']) : get_option('my_auto_blog_author', 1);
    $custom_title = isset($_POST['custom_title']) ? trim(wp_unslash($_POST['custom_title'])) : '';
    $remove_from_queue = isset($_POST['remove_from_queue']) && $_POST['remove_from_queue'] === '1';
    
    // Update publish setting
    update_option('publish_immediately', $publish_immediately ? 1 : 0);
    update_option('my_auto_blog_author', $author_id);
    
    // Determine title source: custom vs queue
    $titles = get_option('my_auto_blog_title', '');
    $titles_array = array_filter(array_map('trim', explode("\n", (string) $titles)));
    if (!empty($custom_title)) {
        $title = $custom_title;
        $should_remove_from_queue = false;
        if ($remove_from_queue) {
            // Remove the first matching occurrence from queue if present
            $key = array_search($custom_title, $titles_array, true);
            if ($key !== false) {
                unset($titles_array[$key]);
                $should_remove_from_queue = true;
            }
        }
    } else {
        if (empty($titles_array)) {
            wp_send_json_error('No titles available. Please add some titles in the Content Planner first.');
            return;
        }
        // Default: use the first queued title
        $title = reset($titles_array);
        $should_remove_from_queue = true; // maintain previous behavior for queued generation
    }
    
    // Generate the post
    try {
        $result = generate_and_create_blog_post($title, false);
        if ($result) {
            // Remove from queue if requested or default queued path
            if ($should_remove_from_queue) {
                // If we used reset earlier, remove that element; otherwise titles_array already had item unset
                if (empty($custom_title)) {
                    // remove first element
                    array_shift($titles_array);
                }
                update_option('my_auto_blog_title', implode("\n", $titles_array));
                
                // Auto-generate more titles if queue is running low
                auto_generate_titles_if_needed();
            }
            
            wp_send_json_success(array(
                'message' => 'Blog post generated successfully!',
                'post_title' => $title
            ));
        } else {
            wp_send_json_error('Failed to generate blog post. Please try again.');
        }
    } catch (Exception $e) {
        wp_send_json_error('Error generating post: ' . $e->getMessage());
    }
}
add_action('wp_ajax_generate_blog_post', 'ajax_generate_blog_post');

/**
 * Async generation: enqueue and return job id immediately
 */
function ajax_generate_blog_post_async() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');

    $publish_immediately = isset($_POST['publish_immediately']) && $_POST['publish_immediately'] === '1';
    $author_id = isset($_POST['author_id']) ? intval($_POST['author_id']) : get_option('my_auto_blog_author', 1);
    $custom_title = isset($_POST['custom_title']) ? trim(wp_unslash($_POST['custom_title'])) : '';
    $remove_from_queue = isset($_POST['remove_from_queue']) && $_POST['remove_from_queue'] === '1';

    if ($custom_title === '') {
        wp_send_json_error('Missing title');
        return;
    }

    // Persist settings
    update_option('publish_immediately', $publish_immediately ? 1 : 0);
    update_option('my_auto_blog_author', $author_id);

    // Create job id and store payload in transient
    $job_id = 'job_' . wp_generate_password(12, false, false);
    $payload = array(
        'title' => $custom_title,
        'publish' => $publish_immediately,
        'author' => $author_id,
        'remove_from_queue' => $remove_from_queue,
        'created_at' => time(),
    );
    set_transient($job_id, array('state' => 'pending', 'payload' => $payload), 60 * 20);

    // Mark as high priority for manual generation
    set_transient($job_id . '_priority', 'high', 60 * 20);
    
    // Schedule single event as backup (longer delay since we prioritize immediate processing)
    if (!wp_next_scheduled('sica_process_generation_job', array($job_id))) {
        wp_schedule_single_event(time() + 30, 'sica_process_generation_job', array($job_id));
    }

    // Multiple strategies for immediate processing to ensure reliability
    
    // Strategy 1: Direct background spawn (primary)
    $bg_response = wp_remote_post(admin_url('admin-ajax.php'), array(
        'timeout' => 0.5, // Slightly longer timeout for reliability
        'blocking' => false,
        'body' => array(
            'action' => 'sica_process_generation_now',
            'job_id' => $job_id,
            'priority' => 'high',
            '_ajax_nonce' => wp_create_nonce('my_auto_blog_nonce')
        ),
        'user-agent' => 'WordPress/' . get_bloginfo('version') . '; ' . home_url() . ' (immediate-processing)',
        'httpversion' => '1.1'
    ));
    
    // Strategy 2: Alternative spawn via different method (fallback)
    if (function_exists('wp_remote_request')) {
        wp_remote_request(admin_url('admin-ajax.php'), array(
            'method' => 'POST',
            'timeout' => 0.1,
            'blocking' => false,
            'body' => array(
                'action' => 'sica_process_generation_now',
                'job_id' => $job_id,
                'priority' => 'high',
                '_ajax_nonce' => wp_create_nonce('my_auto_blog_nonce')
            )
        ));
    }
    
    // Strategy 3: If fastcgi_finish_request is available, process inline after response
    if (function_exists('fastcgi_finish_request')) {
        // Send response immediately
        wp_send_json_success(array('job_id' => $job_id));
        
        // Finish the request to user (returns control to browser)
        fastcgi_finish_request();
        
        // Continue processing in background
    if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Starting fastcgi background processing'); }
        
        // Set limits for background processing
        @ini_set('max_execution_time', 300);
        @ini_set('memory_limit', '512M');
        
        // Small delay to let the background spawns start first
        sleep(1);
        
        // Check if job was already processed by background spawn
        $current_job = get_transient($job_id);
        if ($current_job && isset($current_job['state']) && $current_job['state'] !== 'pending') {
            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Job already processed by background spawn'); }
            return; // Exit - background spawn handled it
        }
        
        // Process the job inline as final fallback
        $processing_key = $job_id . '_processing';
        if (!get_transient($processing_key)) {
            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Processing via fastcgi fallback'); }
            ajax_sica_process_generation_now_inline($job_id);
        } else {
            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Job already being processed, skipping fastcgi fallback'); }
        }
        
        return; // Important: don't call wp_send_json_success again
    }

    wp_send_json_success(array('job_id' => $job_id));
}
add_action('wp_ajax_generate_blog_post_async', 'ajax_generate_blog_post_async');

/**
 * Process a queued generation job (cron handler)
 */
add_action('sica_process_generation_job', function($job_id) {
    $job = get_transient($job_id);
    if (!$job || empty($job['payload'])) {
        return;
    }
    $payload = $job['payload'];
    $title = $payload['title'];
    $remove_from_queue = !empty($payload['remove_from_queue']);

    // Execute generation
    $ok = generate_and_create_blog_post($title, false);
    if ($ok) {
        // Resolve latest post by title
        $post = get_page_by_title($title, OBJECT, 'post');
        $post_url = $post ? get_permalink($post->ID) : '';
        $thumb = $post ? get_the_post_thumbnail_url($post->ID, 'thumbnail') : '';

        // Remove from queue if requested
        if ($remove_from_queue) {
            $titles = get_option('my_auto_blog_title', '');
            $arr = array_values(array_filter(array_map('trim', explode("\n", (string)$titles))));
            $idx = array_search($title, $arr, true);
            if ($idx !== false) { unset($arr[$idx]); update_option('my_auto_blog_title', implode("\n", $arr)); }
            // Top-up if needed
            if (function_exists('auto_generate_titles_if_needed')) { auto_generate_titles_if_needed(); }
        }

        set_transient($job_id, array(
            'state' => 'done',
            'result' => array(
                'post_title' => $title,
                'post_url' => $post_url,
                'thumb_url' => $thumb,
            )
        ), 60 * 20);
    } else {
        // Get the last error from the log for more specific error message
        $error_message = 'Generation failed';
        if (function_exists('error_get_last')) {
            $last_error = error_get_last();
            if ($last_error) {
                $last_msg = $last_error['message'];
                if (strpos($last_msg, 'No valid API token') !== false || strpos($last_msg, 'unauthorized') !== false || strpos($last_msg, '401') !== false) {
                    $error_message = 'Authentication failed. Please check your license key in Account Settings and try again.';
                } elseif (strpos($last_msg, 'generate_and_create_blog_post') !== false) {
                    $error_message = 'Content generation failed. Please try again or contact support if the issue persists.';
                }
            }
        }
        set_transient($job_id, array('state' => 'error', 'error' => $error_message), 60 * 20);
    }
}, 10, 1);

/**
 * AJAX fallback to process generation immediately (non-blocking spawn)
 */
function ajax_sica_process_generation_now() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');

    $job_id = isset($_POST['job_id']) ? sanitize_text_field($_POST['job_id']) : '';
    $priority = isset($_POST['priority']) ? sanitize_text_field($_POST['priority']) : 'normal';
    
    if (!$job_id) { 
        wp_send_json_error('Missing job id'); 
        return; 
    }
    
    if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: process-now handler invoked'); }

    $job = get_transient($job_id);
    if (!$job || empty($job['payload'])) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Job not found or empty payload'); }
        wp_send_json_success(['status' => 'noop']);
        return;
    }

    // Check if job is already being processed (prevent double-processing)
    $processing_key = $job_id . '_processing';
    if (get_transient($processing_key)) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Job already being processed'); }
        wp_send_json_success(['status' => 'already_processing']);
        return;
    }

    // Mark as processing to prevent concurrent execution
    set_transient($processing_key, time(), 60 * 15); // 15-minute processing window
    
    // Set execution limits for immediate processing
    if ($priority === 'high') {
        @ini_set('max_execution_time', 300); // 5 minutes for high priority
        @ini_set('memory_limit', '512M');
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Set high priority execution limits'); }
    }

    $payload = $job['payload'];
    $title = $payload['title'];
    $remove_from_queue = !empty($payload['remove_from_queue']);

    // Update job state to processing
    set_transient($job_id, array(
        'state' => 'processing',
        'payload' => $payload,
        'started_at' => time()
    ), 60 * 20);

    try {
        $ok = generate_and_create_blog_post($title, false);
        
        if ($ok) {
            $post = sica_get_post_by_exact_title($title);
            $post_url = $post ? get_permalink($post->ID) : '';
            $thumb = $post ? get_the_post_thumbnail_url($post->ID, 'thumbnail') : '';

            if ($remove_from_queue) {
                $titles = get_option('my_auto_blog_title', '');
                $arr = array_values(array_filter(array_map('trim', explode("\n", (string)$titles))));
                $idx = array_search($title, $arr, true);
                if ($idx !== false) { 
                    unset($arr[$idx]); 
                    update_option('my_auto_blog_title', implode("\n", $arr)); 
                            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Removed a title from queue'); }
                }
                if (function_exists('auto_generate_titles_if_needed')) { 
                    auto_generate_titles_if_needed(); 
                }
            }

            set_transient($job_id, array(
                'state' => 'done',
                'result' => array(
                    'post_title' => $title,
                    'post_url' => $post_url,
                    'thumb_url' => $thumb,
                ),
                'completed_at' => time()
            ), 60 * 20);
            
            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Successfully processed job'); }
            
            // Clean up processing lock
            delete_transient($processing_key);
            delete_transient($job_id . '_priority');
            
            wp_send_json_success(['status' => 'processed', 'post_url' => $post_url]);
        } else {
            // Get more specific error message for failed generation
            $error_message = 'Generation failed';
            if (function_exists('error_get_last')) {
                $last_error = error_get_last();
                if ($last_error) {
                    $last_msg = $last_error['message'];
                    if (strpos($last_msg, 'No valid API token') !== false || strpos($last_msg, 'unauthorized') !== false || strpos($last_msg, '401') !== false) {
                        $error_message = 'Authentication failed. Please check your license key in Account Settings and try again.';
                    } elseif (strpos($last_msg, 'generate_and_create_blog_post') !== false) {
                        $error_message = 'Content generation failed. Please try again or contact support if the issue persists.';
                    }
                }
            }
            set_transient($job_id, array(
                'state' => 'error', 
                'error' => $error_message,
                'failed_at' => time()
            ), 60 * 20);
            
            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Job failed during generation - ' . $error_message); }
            
            // Clean up processing lock
            delete_transient($processing_key);
            
            wp_send_json_success(['status' => 'failed']);
        }
    } catch (Exception $e) {
        $error_msg = $e->getMessage();
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Exception in job - ' . $error_msg); }
        
        // Provide user-friendly error messages for common issues
        $user_friendly_error = $error_msg;
        if (strpos($error_msg, 'No valid API token') !== false || strpos($error_msg, 'unauthorized') !== false) {
            $user_friendly_error = 'Authentication failed. Please check your license key in Account Settings and try again.';
        } elseif (strpos($error_msg, 'timeout') !== false || strpos($error_msg, 'cURL error 28') !== false) {
            $user_friendly_error = 'The blog generation process timed out. Please try again in a few minutes.';
        } elseif (strpos($error_msg, 'memory') !== false) {
            $user_friendly_error = 'The server ran out of memory during blog generation. Please contact your hosting provider.';
        }
        
        set_transient($job_id, array(
            'state' => 'error', 
            'error' => $user_friendly_error,
            'failed_at' => time()
        ), 60 * 20);
        
        // Clean up processing lock
        delete_transient($processing_key);
        
        wp_send_json_success(['status' => 'failed', 'error' => $user_friendly_error]);
    }
}
add_action('wp_ajax_sica_process_generation_now', 'ajax_sica_process_generation_now');

/**
 * Inline processing for fastcgi_finish_request scenario (no AJAX response)
 */
function ajax_sica_process_generation_now_inline($job_id) {
    if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Inline processing for job'); }
    
    $job = get_transient($job_id);
    if (!$job || empty($job['payload'])) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Inline job not found or empty payload'); }
        return false;
    }

    // Check if job is already being processed (prevent double-processing)
    $processing_key = $job_id . '_processing';
    if (get_transient($processing_key)) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Inline job already being processed'); }
        return false;
    }

    // Mark as processing to prevent concurrent execution
    set_transient($processing_key, time(), 60 * 15); // 15-minute processing window

    $payload = $job['payload'];
    $title = $payload['title'];
    $remove_from_queue = !empty($payload['remove_from_queue']);

    // Update job state to processing
    set_transient($job_id, array(
        'state' => 'processing',
        'payload' => $payload,
        'started_at' => time()
    ), 60 * 20);

    try {
        $ok = generate_and_create_blog_post($title, false);
        
        if ($ok) {
            $post = sica_get_post_by_exact_title($title);
            $post_url = $post ? get_permalink($post->ID) : '';
            $thumb = $post ? get_the_post_thumbnail_url($post->ID, 'thumbnail') : '';

            if ($remove_from_queue) {
                $titles = get_option('my_auto_blog_title', '');
                $arr = array_values(array_filter(array_map('trim', explode("\n", (string)$titles))));
                $idx = array_search($title, $arr, true);
                if ($idx !== false) { 
                    unset($arr[$idx]); 
                    update_option('my_auto_blog_title', implode("\n", $arr)); 
                    if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Inline removed a title from queue'); }
                }
                if (function_exists('auto_generate_titles_if_needed')) { 
                    auto_generate_titles_if_needed(); 
                }
            }

            set_transient($job_id, array(
                'state' => 'done',
                'result' => array(
                    'post_title' => $title,
                    'post_url' => $post_url,
                    'thumb_url' => $thumb,
                ),
                'completed_at' => time()
            ), 60 * 20);
            
            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Inline processing successfully completed job'); }
            
            // Clean up processing lock
            delete_transient($processing_key);
            delete_transient($job_id . '_priority');
            
            return true;
        } else {
            set_transient($job_id, array(
                'state' => 'error', 
                'error' => 'Generation failed',
                'failed_at' => time()
            ), 60 * 20);
            
            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Inline job failed during generation'); }
            
            // Clean up processing lock
            delete_transient($processing_key);
            
            return false;
        }
    } catch (Exception $e) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('SICA: Inline exception in job'); }
        
        set_transient($job_id, array(
            'state' => 'error', 
            'error' => 'Exception: ' . $e->getMessage(),
            'failed_at' => time()
        ), 60 * 20);
        
        // Clean up processing lock
        delete_transient($processing_key);
        
        return false;
    }
}

/**
 * Poll generation status
 */
function ajax_get_generation_status() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');

    $job_id = isset($_POST['job_id']) ? sanitize_text_field($_POST['job_id']) : '';
    if (!$job_id) { wp_send_json_error('Missing job id'); return; }
    $job = get_transient($job_id);
    if (!$job) { wp_send_json_success(array('state' => 'pending')); return; }
    wp_send_json_success($job);
}
add_action('wp_ajax_get_generation_status', 'ajax_get_generation_status');

/**
 * AJAX handler to cancel a timed-out generation and refund credit
 */
// Helper: fetch a post by exact title without using deprecated get_page_by_title
function sica_get_post_by_exact_title($title) {
    global $wpdb;
    if ($title === '') { return null; }
    // Exact match by title; prefer most recent if duplicates
    $post_id = $wpdb->get_var($wpdb->prepare(
        "SELECT ID FROM {$wpdb->posts}
         WHERE post_title = %s
           AND post_type = 'post'
           AND post_status IN ('publish','draft','pending','future','private')
         ORDER BY post_date_gmt DESC
         LIMIT 1",
        $title
    ));
    return $post_id ? get_post((int)$post_id) : null;
}

function ajax_find_post_by_title() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    $title = isset($_POST['title']) ? sanitize_text_field(wp_unslash($_POST['title'])) : '';
    if ($title === '') { wp_send_json_error('Missing title'); return; }
    $post = sica_get_post_by_exact_title($title);
    if (!$post) { wp_send_json_success(['found' => false]); return; }
    $post_url = get_permalink($post->ID);
    $thumb = get_the_post_thumbnail_url($post->ID, 'thumbnail');
    wp_send_json_success([
        'found' => true,
        'result' => [
            'post_title' => $post->post_title,
            'post_url' => $post_url,
            'thumb_url' => $thumb ?: ''
        ]
    ]);
}
add_action('wp_ajax_find_post_by_title', 'ajax_find_post_by_title');

/**
 * AJAX handler to cancel a timed-out generation and refund credit
 */
function ajax_cancel_generation() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');

    $job_id = isset($_POST['job_id']) ? sanitize_text_field($_POST['job_id']) : '';
    $reason = isset($_POST['reason']) ? sanitize_text_field($_POST['reason']) : 'user_cancel';
    $elapsed_seconds = isset($_POST['elapsed_seconds']) ? intval($_POST['elapsed_seconds']) : 0;
    
    if (!$job_id) {
        wp_send_json_error('Missing job id');
        return;
    }
    
    if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Cancelling generation'); }
    
    // Clear the job from transients
    delete_transient($job_id);
    
    // Notify server to cancel/refund the generation
    $customer_email = get_option('sica_customer_email', '');
    $site_url = get_site_url();
    
    if ($customer_email) {
        $response = wp_remote_post('https://api.sica.ai/subscription.php', array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-Sica-Token' => get_option('sica_api_token', '')
            ),
            'body' => json_encode(array(
                'action' => 'cancel_generation',
                'customer_email' => $customer_email,
                'site_url' => $site_url,
                'job_id' => $job_id,
                'reason' => $reason,
                'elapsed_seconds' => $elapsed_seconds
            )),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Failed to notify server of generation cancellation'); }
            wp_send_json_success(array('refunded' => false, 'local_only' => true));
            return;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if ($data && isset($data['ok']) && $data['ok']) {
            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Server confirmed generation cancellation and refund'); }
            // Clear subscription cache to refresh usage counts
            delete_transient('sica_subscription_status');
            wp_send_json_success(array('refunded' => true, 'server_confirmed' => true));
        } else {
            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Server did not confirm cancellation'); }
            wp_send_json_success(array('refunded' => false, 'server_error' => true));
        }
    } else {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('No customer email found for cancellation notification'); }
        wp_send_json_success(array('refunded' => false, 'no_email' => true));
    }
}
add_action('wp_ajax_cancel_generation', 'ajax_cancel_generation');

/**
 * AJAX handler to delete a planned title from the queue
 */
function ajax_delete_planned_title() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');

    $title = isset($_POST['title']) ? trim(wp_unslash($_POST['title'])) : '';
    if ($title === '') {
        wp_send_json_error('Missing title');
        return;
    }

    $titles = get_option('my_auto_blog_title', '');
    $titles_array = array_values(array_filter(array_map('trim', explode("\n", (string) $titles))));
    $key = array_search($title, $titles_array, true);
    if ($key === false) {
        wp_send_json_error('Title not found');
        return;
    }
    unset($titles_array[$key]);
    update_option('my_auto_blog_title', implode("\n", $titles_array));
    wp_send_json_success(true);
}
add_action('wp_ajax_delete_planned_title', 'ajax_delete_planned_title');

/**
 * AJAX handler for toggling auto-schedule settings
 */
function ajax_toggle_auto_schedule() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    
    $enable = isset($_POST['enable']) && $_POST['enable'] === '1';
    $frequency = isset($_POST['frequency']) ? sanitize_text_field($_POST['frequency']) : 'weekly';
    
    // Update settings
    update_option('my_auto_blog_cron_active', $enable ? 1 : 0);
    update_option('my_auto_blog_schedule', $frequency);
    
    // Set smart default start date and time if not set
    if (!get_option('my_auto_blog_start_date')) {
        update_option('my_auto_blog_start_date', date('Y-m-d'));
    }
    if (!get_option('my_auto_blog_start_time')) {
        update_option('my_auto_blog_start_time', '09:00');
    }
    
    // Ensure start time is in the future when enabling
    if ($enable) {
        $start_date = get_option('my_auto_blog_start_date');
        $start_time = get_option('my_auto_blog_start_time');
        $current_datetime = new DateTime('now', new DateTimeZone(get_option('timezone_string', 'UTC')));
        $scheduled_datetime = DateTime::createFromFormat('Y-m-d H:i', $start_date . ' ' . $start_time, new DateTimeZone(get_option('timezone_string', 'UTC')));
        
        // If scheduled time is in the past, move to next occurrence
        if ($scheduled_datetime && $scheduled_datetime <= $current_datetime) {
            $interval_map = [
                'daily' => '+1 day',
                'every_three_days' => '+3 days', 
                'weekly' => '+1 week',
                'monthly' => '+1 month'
            ];
            $interval = $interval_map[$frequency] ?? '+1 week';
            $scheduled_datetime->modify($interval);
            update_option('my_auto_blog_start_date', $scheduled_datetime->format('Y-m-d'));
        }
    }
    
    // Activate or deactivate cron job (use scheduler functions)
    if ($enable) {
        if (function_exists('my_auto_blog_activate_cron')) {
            my_auto_blog_activate_cron();
        }
    } else {
        if (function_exists('my_auto_blog_deactivate_cron')) {
            my_auto_blog_deactivate_cron();
        }
    }
    
    wp_send_json_success([
        'message' => $enable ? 'Auto-schedule enabled' : 'Auto-schedule disabled',
        'enabled' => $enable,
        'frequency' => $frequency
    ]);
}
add_action('wp_ajax_toggle_auto_schedule', 'ajax_toggle_auto_schedule');

/**
 * AJAX handler for auto-generating titles with smart defaults
 */
function ajax_auto_generate_titles() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    
    try {
        // Get site title for business detection
        $site_title = get_bloginfo('name');
        
        // Auto-fill business fields if empty
        $business_category = get_option('my_auto_blog_business_category', '');
        $service_area = get_option('my_auto_blog_service_area', '');
        $products_services = get_option('my_auto_blog_products_services', '');
        $keywords = get_option('my_auto_blog_keywords', '');
        
        // Smart defaults based on site title
        if (empty($business_category)) {
            // Extract potential business type from site title
            $business_category = detect_business_category_from_title($site_title);
            update_option('my_auto_blog_business_category', $business_category);
        }
        
        if (empty($service_area)) {
            // Default to a generic area that can be refined later
            $service_area = 'Local area';
            update_option('my_auto_blog_service_area', $service_area);
        }
        
        if (empty($products_services)) {
            // Generate based on business category
            $products_services = generate_default_services($business_category);
            update_option('my_auto_blog_products_services', $products_services);
        }
        
        if (empty($keywords)) {
            // Generate basic keywords from business info
            $keywords = generate_default_keywords($business_category, $site_title);
            update_option('my_auto_blog_keywords', $keywords);
        }
        
        // Generate initial titles using existing function
        if (function_exists('generate_initial_title_set')) {
            $titles = generate_initial_title_set();
            if (!empty($titles)) {
                update_option('my_auto_blog_title', implode("\n", $titles));
                
                // If scheduler is enabled but not properly scheduled, reactivate it
                $cron_active = get_option('my_auto_blog_cron_active', 0);
                if ($cron_active && !wp_next_scheduled('my_auto_blog_cron_event')) {
                    if (function_exists('my_auto_blog_activate_cron')) {
                        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Auto-generate titles: Reactivating cron'); }
                        my_auto_blog_activate_cron();
                    }
                }
                
                wp_send_json_success([
                    'message' => 'Generated ' . count($titles) . ' blog titles successfully!',
                    'titles_count' => count($titles),
                    'auto_filled' => [
                        'business_category' => $business_category,
                        'service_area' => $service_area,
                        'products_services' => $products_services,
                        'keywords' => $keywords
                    ]
                ]);
            } else {
                wp_send_json_error('Failed to generate titles. Please check your business settings and try again.');
            }
        } else {
            wp_send_json_error('Title generation function not available. Please use the Content Planner tab.');
        }
        
    } catch (Exception $e) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Auto-generate titles error'); }
        wp_send_json_error('An error occurred while generating titles: ' . $e->getMessage());
    }
}
add_action('wp_ajax_auto_generate_titles', 'ajax_auto_generate_titles');

/**
 * AJAX handler for fixing broken schedule
 */
function ajax_fix_schedule() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    
    try {
        // Get current settings
        $cron_active = get_option('my_auto_blog_cron_active', 0);
        $current_schedule = get_option('my_auto_blog_schedule', 'weekly');
        
        if (!$cron_active) {
            wp_send_json_error('Scheduler is not enabled');
            return;
        }
        
        // Debug current state
        $next_timestamp = wp_next_scheduled('my_auto_blog_cron_event');
        $start_date = get_option('my_auto_blog_start_date');
        $start_time = get_option('my_auto_blog_start_time');
        
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Fix Schedule - Current state'); }
        
        // Ensure start time is in the future before reactivating
        // Resolve a reliable timezone (prefer named tz; fallback to gmt_offset; then UTC)
        $wp_tz = get_option('timezone_string');
        if (!$wp_tz) {
            $offset = get_option('gmt_offset');
            if ($offset !== false && $offset !== '' && $offset !== null) {
                $seconds = (float)$offset * 3600;
                $name = @timezone_name_from_abbr('', (int)$seconds, 0);
                if ($name) { $wp_tz = $name; }
            }
            if (!$wp_tz) { $wp_tz = 'UTC'; }
        }
        $tz = new DateTimeZone($wp_tz);

        // Default missing/invalid date/time to a few minutes in the future
        $now = new DateTime('now', $tz);
        $now->modify('+5 minutes');
        if (empty($start_date) || !preg_match('/^\d{4}-\d{2}-\d{2}$/', (string)$start_date)) {
            $start_date = $now->format('Y-m-d');
            update_option('my_auto_blog_start_date', $start_date);
        }
        if (empty($start_time) || !preg_match('/^\d{2}:\d{2}$/', (string)$start_time)) {
            $start_time = $now->format('H:i');
            update_option('my_auto_blog_start_time', $start_time);
        }

        $scheduled_datetime = DateTime::createFromFormat('Y-m-d H:i', $start_date . ' ' . $start_time, $tz);
        
        // If scheduled time is in the past, move to next occurrence
        if ($scheduled_datetime && $scheduled_datetime <= $current_datetime) {
            $interval_map = [
                'daily' => '+1 day',
                'every_three_days' => '+3 days', 
                'weekly' => '+1 week',
                'monthly' => '+1 month'
            ];
            $interval = $interval_map[$current_schedule] ?? '+1 week';
            $scheduled_datetime->modify($interval);
            $new_start_date = $scheduled_datetime->format('Y-m-d');
            $new_start_time = $scheduled_datetime->format('H:i');
            update_option('my_auto_blog_start_date', $new_start_date);
            update_option('my_auto_blog_start_time', $new_start_time);
            
            if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Updated start date (was in the past)'); }
        }
        
        // Force reactivation of cron
        if (function_exists('my_auto_blog_activate_cron')) {
            my_auto_blog_activate_cron();
            
            // Check if it worked
            $new_timestamp = wp_next_scheduled('my_auto_blog_cron_event');
            if ($new_timestamp) {
                wp_send_json_success([
                    'message' => 'Schedule fixed successfully!',
                    'next_run' => wp_date('M j, g:i A', $new_timestamp)
                ]);
            } else {
                wp_send_json_error('Failed to reactivate schedule. The start time may still be invalid.');
            }
        } else {
            wp_send_json_error('Scheduler function not available');
        }
        
    } catch (Exception $e) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Fix schedule error'); }
        wp_send_json_error('An error occurred while fixing the schedule: ' . $e->getMessage());
    }
}
add_action('wp_ajax_fix_schedule', 'ajax_fix_schedule');

/**
 * AJAX handler for saving notification emails
 */
function ajax_save_notification_emails() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    
    $emails = isset($_POST['emails']) ? sanitize_text_field($_POST['emails']) : '';
    
    // Validate emails
    if (!empty($emails)) {
        $email_list = array_filter(array_map('trim', explode(',', $emails)));
        $invalid_emails = [];
        
        foreach ($email_list as $email) {
            if (!is_email($email)) {
                $invalid_emails[] = $email;
            }
        }
        
        if (!empty($invalid_emails)) {
            wp_send_json_error('Invalid email addresses: ' . implode(', ', $invalid_emails));
            return;
        }
    }
    
    // Save the emails
    update_option('my_auto_blog_notify_emails', $emails);
    
    if (empty($emails)) {
        wp_send_json_success('Notification emails cleared successfully');
    } else {
        $count = count(explode(',', $emails));
        wp_send_json_success('Saved ' . $count . ' notification email' . ($count > 1 ? 's' : '') . ' successfully');
    }
}
add_action('wp_ajax_save_notification_emails', 'ajax_save_notification_emails');

/**
 * Detect business category from site title
 */
function detect_business_category_from_title($site_title) {
    $title_lower = strtolower($site_title);
    
    // Common business type patterns
    $patterns = [
        'construction|builder|contractor|remodel|renovation|home improvement' => 'Construction & Home Improvement',
        'plumber|plumbing|pipe|drain|water|hvac|heating|cooling' => 'Plumbing & HVAC',
        'electrician|electrical|wiring|electric|power' => 'Electrical Services',
        'lawyer|attorney|legal|law firm|counsel' => 'Legal Services',
        'doctor|medical|health|clinic|dentist|dental' => 'Healthcare',
        'restaurant|cafe|food|catering|dining|pizza|burger' => 'Food & Dining',
        'auto|car|automotive|repair|mechanic|garage' => 'Automotive',
        'real estate|realtor|property|homes|house' => 'Real Estate',
        'salon|beauty|hair|spa|massage|nails' => 'Beauty & Wellness',
        'fitness|gym|trainer|yoga|pilates|health' => 'Fitness & Health',
        'consulting|business|marketing|agency|services' => 'Professional Services'
    ];
    
    foreach ($patterns as $pattern => $category) {
        if (preg_match('/(' . $pattern . ')/i', $title_lower)) {
            return $category;
        }
    }
    
    // Default to generic business
    return 'Professional Services';
}

/**
 * Generate default services based on business category
 */
function generate_default_services($business_category) {
    $services_map = [
        'Construction & Home Improvement' => 'Home renovations, kitchen remodeling, bathroom upgrades, flooring installation, roofing, general contracting',
        'Plumbing & HVAC' => 'Plumbing repairs, drain cleaning, water heater installation, HVAC maintenance, heating and cooling systems',
        'Electrical Services' => 'Electrical repairs, wiring installation, lighting upgrades, electrical inspections, panel upgrades',
        'Legal Services' => 'Legal consultation, contract review, litigation support, estate planning, business law',
        'Healthcare' => 'Medical consultations, preventive care, treatment services, patient education, health screenings',
        'Food & Dining' => 'Fresh meals, catering services, takeout, delivery, special events, custom menus',
        'Automotive' => 'Auto repairs, maintenance services, diagnostics, parts replacement, vehicle inspections',
        'Real Estate' => 'Property sales, home buying assistance, market analysis, property management, real estate consultation',
        'Beauty & Wellness' => 'Hair styling, beauty treatments, spa services, wellness consultations, skincare',
        'Fitness & Health' => 'Personal training, fitness classes, wellness coaching, nutrition guidance, health assessments',
        'Professional Services' => 'Business consulting, strategic planning, professional advice, custom solutions, expert services'
    ];
    
    return $services_map[$business_category] ?? 'Professional services, consultation, expert solutions, customer support';
}

/**
 * Generate default keywords from business category and site title
 */
function generate_default_keywords($business_category, $site_title) {
    $business_words = array_filter(explode(' ', strtolower($site_title)));
    $category_keywords = [];
    
    // Category-specific keywords
    switch ($business_category) {
        case 'Construction & Home Improvement':
            $category_keywords = ['home renovation', 'construction services', 'remodeling', 'contractor', 'home improvement'];
            break;
        case 'Plumbing & HVAC':
            $category_keywords = ['plumbing services', 'drain cleaning', 'HVAC repair', 'heating and cooling', 'plumber'];
            break;
        case 'Electrical Services':
            $category_keywords = ['electrical services', 'electrician', 'electrical repair', 'wiring', 'electrical installation'];
            break;
        case 'Legal Services':
            $category_keywords = ['legal services', 'attorney', 'lawyer', 'legal consultation', 'law firm'];
            break;
        default:
            $category_keywords = ['professional services', 'business solutions', 'expert consultation', 'quality service', 'local business'];
    }
    
    // Combine business name words with category keywords
    $all_keywords = array_merge(array_slice($business_words, 0, 2), $category_keywords);
    
    return implode("\n", array_slice(array_unique($all_keywords), 0, 8));
}

/**
 * AJAX handler for license key activation
 */
function ajax_activate_license_key() {
    // Security checks
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    
    $license_key = isset($_POST['license_key']) ? sanitize_text_field($_POST['license_key']) : '';
    
    if (empty($license_key)) {
        wp_send_json_error('Please enter a license key');
        return;
    }
    
    // Call the server to activate the license
    $site_url = get_site_url();
    
    $response = wp_remote_post('https://api.sica.ai/activate.php', array(
        'headers' => array(
            'Content-Type' => 'application/json'
        ),
        'body' => json_encode(array(
            'license_key' => $license_key,
            'site_url' => $site_url
        )),
        'timeout' => 30
    ));
    
    if (is_wp_error($response)) {
        wp_send_json_error('Network error: ' . $response->get_error_message());
        return;
    }
    
    $data = json_decode(wp_remote_retrieve_body($response), true);
    
    if (defined('WP_DEBUG') && WP_DEBUG) { error_log('License activation response received'); }
    
    if (!$data || !$data['ok']) {
        $error_msg = $data['error'] ?? 'Invalid license key';
        wp_send_json_error($error_msg);
        return;
    }
    
    // License activated successfully - save license data locally
    // Old license system (for Setup tab)
    update_option('sica_license_key', $license_key, false); // autoload=false for security
    update_option('sica_license_status', 'active');
    update_option('sica_license_type', 'paid');
    update_option('sica_posts_remaining', $data['credits_limit']);
    update_option('sica_monthly_limit', $data['credits_limit']);
    
    // New subscription system (for PLG planner)
    // Use real customer email for server API authentication
    if (isset($data['customer_email'])) {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('Saving customer email from license activation'); }
        update_option('sica_customer_email', $data['customer_email']);
    } else {
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('No customer_email found in license activation response'); }
    }
    update_option('sica_email_verified', true);
    
    // Store API token from server if provided (CRITICAL for 401 fix)
    if (!empty($data['api_token'])) {
        update_option('sica_api_token', sanitize_text_field($data['api_token']), false); // autoload=false for security
        if (defined('WP_DEBUG') && WP_DEBUG) { error_log('API token saved from license activation'); }
    } else {
        // Try to reissue token using the customer email
        if (function_exists('sica_ensure_api_token')) {
            $token = sica_ensure_api_token();
            if (empty($token)) {
                if (defined('WP_DEBUG') && WP_DEBUG) { error_log('WARNING: No API token received during license activation'); }
            }
        }
    }
    
    // Clear any cached subscription status to force fresh server check
    delete_transient('sica_subscription_status');
    
    // Also clear the cached status in the subscription check function
    if (function_exists('sica_check_subscription_status_server')) {
        // Force a fresh server check by clearing all related caches
        $current_month = date('Y-m');
        delete_option("sica_monthly_usage_$current_month");
        delete_transient('sica_subscription_status');
    }
    
    // Also clear local usage cache to force server sync
    $current_month = date('Y-m');
    delete_option("sica_monthly_usage_$current_month");
    
    wp_send_json_success('License activated successfully! Your plan has been upgraded.');
}
add_action('wp_ajax_activate_license_key', 'ajax_activate_license_key');

/**
 * AJAX handler to force refresh subscription status (for debugging conflicting status)
 */
function ajax_force_refresh_subscription() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('my_auto_blog_nonce', '_ajax_nonce');
    
    // Clear all caches
    delete_transient('sica_subscription_status');
    $current_month = date('Y-m');
    delete_option("sica_monthly_usage_$current_month");
    
    // Force fresh server check
    if (function_exists('sica_check_subscription_status_server')) {
        $server_status = sica_check_subscription_status_server();
        if ($server_status && $server_status['ok']) {
            wp_send_json_success(['message' => 'Subscription status refreshed from server', 'status' => $server_status]);
        } else {
            wp_send_json_error(['message' => 'Failed to refresh from server', 'server_response' => $server_status]);
        }
    } else {
        wp_send_json_error('Subscription functions not available');
    }
}
add_action('wp_ajax_force_refresh_subscription', 'ajax_force_refresh_subscription');

/**
 * Get recent AI-generated posts
 */
function get_recent_ai_posts($limit = 3) {
    $posts = get_posts(array(
        'meta_key' => '_ai_blogger_generated',
        'meta_value' => '1',
        'posts_per_page' => $limit,
        'post_status' => array('publish', 'draft'),
        'orderby' => 'date',
        'order' => 'DESC'
    ));
    
    $recent_posts = array();
    foreach ($posts as $post) {
        $status_label = ($post->post_status === 'publish') ? 'Published' : ucfirst($post->post_status);
        // Robust thumbnail retrieval
        $thumb = '';
        $thumb_id = get_post_thumbnail_id($post->ID);
        if ($thumb_id) {
            $src = wp_get_attachment_image_src($thumb_id, 'thumbnail');
            if (is_array($src) && !empty($src[0])) {
                $thumb = $src[0];
            } else {
                $thumb = wp_get_attachment_url($thumb_id);
            }
        }
        $recent_posts[] = array(
            'id' => $post->ID,
            'title' => $post->post_title,
            'date' => human_time_diff(strtotime($post->post_date), current_time('timestamp')) . ' ago',
            'status' => $status_label,
            'edit_url' => get_edit_post_link($post->ID),
            'thumb' => $thumb
        );
    }
    
    return $recent_posts;
}
