<?php
/**
 * Welcome Screen for New Users
 * 
 * Creates a world-class PLG onboarding experience with simulated demo,
 * business context detection, and progressive value delivery
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Check if user should see welcome screen
 */
function should_show_welcome_screen() {
    // Show if never seen before OR if manually accessed
    $welcome_completed = get_option('sica_welcome_completed', false);
    $manual_access = isset($_GET['page']) && $_GET['page'] === 'sica-welcome';
    
    return !$welcome_completed || $manual_access;
}

/**
 * Auto-detect industry from existing WordPress content
 */
function detect_business_industry() {
    // Priority 1: Check if business category already set
    $business_category = get_option('my_auto_blog_business_category', '');
    if (!empty($business_category)) {
        return $business_category;
    }
    
    // Priority 2: Check Yoast SEO focus keywords
    if (function_exists('YoastSEO')) {
        $focus_keywords = get_option('wpseo_focus_keyword', '');
        if (!empty($focus_keywords)) {
            return $focus_keywords;
        }
    }
    
    // Priority 3: Check RankMath focus keywords
    if (function_exists('rank_math')) {
        global $wpdb;
        $focus_keyword = $wpdb->get_var("
            SELECT meta_value 
            FROM {$wpdb->postmeta} 
            WHERE meta_key = 'rank_math_focus_keyword' 
            AND meta_value != ''
            LIMIT 1
        ");
        if (!empty($focus_keyword)) {
            return $focus_keyword;
        }
    }
    
    // Priority 4: Most used category
    $categories = get_categories(array('orderby' => 'count', 'order' => 'DESC', 'number' => 1));
    if (!empty($categories)) {
        return $categories[0]->name;
    }
    
    // Priority 5: Site tagline
    $tagline = get_bloginfo('description');
    if (!empty($tagline) && $tagline !== 'Just another WordPress site') {
        return $tagline;
    }
    
    // Fallback
    return 'Professional Services';
}

/**
 * Render the main welcome screen
 */
function render_welcome_screen() {
    $business_name = get_bloginfo('name');
    $detected_industry = detect_business_industry();
    $site_url = home_url();
    
    // Check if email verification already completed
    $email_verified = get_option('sica_email_verified', false);
    $customer_email = get_option('sica_customer_email', '');
    
    ?>
    <!DOCTYPE html>
    <html <?php language_attributes(); ?>>
    <head>
        <meta charset="<?php bloginfo('charset'); ?>">
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <title>Welcome to Sica Content AI</title>
        <?php wp_head(); ?>
        <link rel="stylesheet" href="<?php echo plugin_dir_url(__FILE__) . '../assets/welcome.css'; ?>">
    </head>
    <body class="sica-welcome-body">
        
        <!-- Progress Bar at Top -->
        <div id="welcome-progress-bar" class="welcome-progress-container">
            <div class="welcome-progress-steps">
                <div class="step active" data-step="1">
                    <span class="step-number">1</span>
                    <span class="step-label">Create Post</span>
                </div>
                <div class="step" data-step="2">
                    <span class="step-number">2</span>
                    <span class="step-label">Schedule Flow</span>
                </div>
                <div class="step" data-step="3">
                    <span class="step-number">3</span>
                    <span class="step-label">Start Ranking</span>
                </div>
            </div>
        </div>

        <div class="welcome-container">
            
            <!-- Zone 1: Zero-Friction Hero -->
            <div class="welcome-hero">
                <h1 class="hero-headline">
                    Get your first SEO-optimized blog post in 60 seconds
                </h1>
                <p class="hero-subtext">
                    We'll create content about <strong><?php echo esc_html($detected_industry); ?></strong> that helps your customers find you on Google.
                </p>
            </div>

            <!-- Zone 2: Micro-Setup Banner -->
            <div class="welcome-setup-form" id="setup-form">
                <div class="setup-form-container">
                    <label for="key-service" class="setup-label">
                        What's one service you want to rank higher on Google for?
                    </label>
                    <div class="setup-input-row">
                        <input 
                            type="text" 
                            id="key-service" 
                            class="setup-input"
                            placeholder="e.g. Emergency plumbing, Legal consultation, Home renovation"
                            autocomplete="off"
                        >
                        <div class="setup-checkbox-container">
                            <input type="checkbox" id="no-idea-checkbox" class="setup-checkbox">
                            <label for="no-idea-checkbox" class="checkbox-label">I don't know—suggest one</label>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Zone 3: One-Click Demo -->
            <div class="welcome-demo-section" id="demo-section">
                <button id="generate-demo-btn" class="demo-button" disabled>
                    <span class="button-icon">✨</span>
                    <span class="button-text">Generate Demo Post</span>
                </button>
                <p class="demo-note">Free preview - see what AI creates for your business</p>
            </div>

            <!-- Zone 4: Live Preview Card (Hidden initially) -->
            <div class="welcome-preview-card" id="preview-card" style="display: none;">
                <div class="preview-header">
                    <h3>🎯 Your AI-Generated Post Preview</h3>
                    <span class="preview-status" id="preview-status">Generating...</span>
                </div>
                
                <!-- Progress Component Integration -->
                <div id="progress-container" style="display: none;">
                    <?php render_article_progress_component(); ?>
                </div>
                
                <div class="preview-content" id="preview-content" style="display: none;">
                    <div class="preview-title">
                        <h4 id="generated-title">Loading...</h4>
                    </div>
                    <div class="preview-image">
                        <img id="generated-image" src="" alt="Generated featured image" style="display: none;">
                        <div class="image-placeholder" id="image-placeholder">
                            <span>🖼️ Featured image generating...</span>
                        </div>
                    </div>
                    <div class="preview-text">
                        <p id="generated-intro">Loading content preview...</p>
                    </div>
                </div>
                
                <div class="preview-actions" id="preview-actions" style="display: none;">
                    <button id="publish-demo-btn" class="action-button primary">
                        📝 Create Real Post
                    </button>
                    <button id="retry-demo-btn" class="action-button secondary">
                        🔄 Try Different Topic
                    </button>
                </div>
            </div>

            <!-- Email Verification Modal (Hidden initially) -->
            <div class="email-modal" id="email-modal" style="display: none;">
                <div class="email-modal-content">
                    <div class="modal-header">
                        <h3>🎉 Ready to publish your SEO content?</h3>
                        <p>Verify your email to create this post + get 2 free AI articles monthly</p>
                    </div>
                    
                    <?php if (!$email_verified): ?>
                        <div class="email-form" id="email-form">
                            <div class="email-step" id="email-step-1" style="display: <?php echo empty($customer_email) ? 'block' : 'none'; ?>;">
                                <label for="verification-email">Enter your email address:</label>
                                <input type="email" id="verification-email" placeholder="your-email@example.com" 
                                       autocomplete="off" autocapitalize="none" spellcheck="false" 
                                       data-lpignore="true" data-form-type="other"
                                       value="<?php echo esc_attr($customer_email); ?>">
                                
                                <button id="send-verification-btn" class="action-button primary">
                                    Send Verification Code
                                </button>
                            </div>
                            
                            <div class="email-step" id="email-step-2" style="display: <?php echo !empty($customer_email) ? 'block' : 'none'; ?>;">
                                <p class="verification-sent">
                                    <strong>Click "Resend Code" to continue</strong><br>
                                    We'll send a 6-digit verification code to: <strong id="verification-email-display"><?php echo esc_html($customer_email); ?></strong>
                                </p>
                                <label for="verification-code">Enter verification code:</label>
                                <input type="text" id="verification-code" placeholder="123456" maxlength="6" class="verification-code-input">
                                <div class="verification-actions">
                                    <button id="verify-email-btn" class="action-button primary">
                                        Verify & Create Post
                                    </button>
                                    <button id="resend-code-btn" class="action-button secondary">
                                        Resend Code
                                    </button>
                                </div>
                                <p class="change-email">
                                    Wrong email? <a href="#" id="change-email-link">Change email address</a>
                                </p>
                            </div>
                        </div>
                    <?php else: ?>
                        <div class="email-verified">
                            <p>Email verified! Creating your real post now...</p>
                        </div>
                    <?php endif; ?>
                    
                    <div class="modal-close">
                        <button id="close-email-modal" class="close-button">×</button>
                    </div>
                </div>
                <div class="modal-overlay"></div>
            </div>

            <!-- Zone 7: Quota & Upgrade Chip -->
            <div class="welcome-quota-chip">
                <div class="quota-ring">
                    <svg width="24" height="24" viewBox="0 0 24 24">
                        <circle cx="12" cy="12" r="10" fill="none" stroke="#e0e0e0" stroke-width="2"/>
                        <circle cx="12" cy="12" r="10" fill="none" stroke="#4CAF50" stroke-width="2" 
                                stroke-dasharray="62.8" stroke-dashoffset="41.9" transform="rotate(-90 12 12)"/>
                    </svg>
                </div>
                <span class="quota-text">2 free posts remaining this month</span>
                <a href="#" class="upgrade-link">Upgrade for unlimited</a>
            </div>

            <!-- Zone 8: Trust Signals Row -->
            <div class="welcome-trust-signals">
                <div class="trust-logos">
                    <span class="trust-logo">🔍 SEO Optimized</span>
                    <span class="trust-logo">🤖 GPT-4o Powered</span>
                    <span class="trust-logo">📝 WordPress Ready</span>
                </div>
                <p class="trust-note">Content stays in drafts until you approve and publish</p>
            </div>

            <!-- Zone 9: Help & Chat Pop-over -->
            <div class="welcome-help">
                <button id="help-button" class="help-button">?</button>
                <div id="help-popover" class="help-popover" style="display: none;">
                    <h4>How It Works</h4>
                    <ul>
                        <li>Tell us what service you want to rank for</li>
                        <li>AI creates SEO-optimized blog posts</li>
                        <li>Content appears as drafts in WordPress</li>
                        <li>Review, edit if needed, then publish!</li>
                    </ul>
                    <p><strong>Questions?</strong> Email support@sica.ai</p>
                </div>
            </div>

        </div>

        <!-- Celebration Confetti Container -->
        <div id="celebration-confetti" style="display: none;"></div>

        <script src="<?php echo plugin_dir_url(__FILE__) . '../assets/welcome.js'; ?>"></script>
        <script>
            // Pass PHP data to JavaScript
            window.sicaWelcome = {
                ajaxUrl: '<?php echo admin_url('admin-ajax.php'); ?>',
                nonce: '<?php echo wp_create_nonce('sica_welcome_nonce'); ?>',
                businessName: '<?php echo esc_js($business_name); ?>',
                detectedIndustry: '<?php echo esc_js($detected_industry); ?>',
                isEmailVerified: <?php echo $email_verified ? 'true' : 'false'; ?>,
                customerEmail: '<?php echo esc_js($customer_email); ?>'
            };
        </script>
        
        <?php wp_footer(); ?>
    </body>
    </html>
    <?php
}

/**
 * Handle AJAX requests for welcome screen
 */
function handle_welcome_ajax() {
    check_ajax_referer('sica_welcome_nonce', 'nonce');
    
    $action_type = sanitize_text_field($_POST['action_type']);
    
    switch ($action_type) {
        case 'generate_demo':
            generate_demo_content();
            break;
            
        case 'send_verification':
            send_email_verification();
            break;
            
        case 'verify_email':
            verify_email_code();
            break;
            
        case 'create_real_post':
            create_real_post();
            break;
            
        case 'complete_welcome':
            complete_welcome_flow();
            break;
            
        default:
            wp_send_json_error('Invalid action');
    }
}
add_action('wp_ajax_sica_welcome', 'handle_welcome_ajax');

/**
 * Generate simulated demo content (no API calls)
 */
function generate_demo_content() {
    $key_service = sanitize_text_field($_POST['key_service']);
    $business_name = get_bloginfo('name');
    $detected_industry = detect_business_industry();
    
    // Simulate realistic processing time with staged responses
    $stage = intval($_POST['stage'] ?? 1);
    
    switch ($stage) {
        case 1:
            // Stage 1: Start generation
            wp_send_json_success(array(
                'stage' => 1,
                'message' => 'Analyzing your business and target keywords...',
                'progress' => 20
            ));
            break;
            
        case 2:
            // Stage 2: Generate title
            $demo_title = generate_demo_title($key_service, $detected_industry);
            wp_send_json_success(array(
                'stage' => 2,
                'message' => 'Creating engaging blog title...',
                'progress' => 40,
                'title' => $demo_title
            ));
            break;
            
        case 3:
            // Stage 3: Generate content preview
            $demo_content = generate_demo_content_preview($key_service, $detected_industry);
            wp_send_json_success(array(
                'stage' => 3,
                'message' => 'Writing SEO-optimized content...',
                'progress' => 70,
                'content' => $demo_content
            ));
            break;
            
        case 4:
            // Stage 4: Complete with image
            $demo_image = generate_demo_image_url($key_service);
            wp_send_json_success(array(
                'stage' => 4,
                'message' => 'Complete! Ready to publish.',
                'progress' => 100,
                'image' => $demo_image,
                'complete' => true
            ));
            break;
            
        default:
            wp_send_json_error('Invalid stage');
    }
}

/**
 * Generate realistic demo title
 */
function generate_demo_title($key_service, $industry) {
    $title_templates = array(
        "The Complete Guide to {service} in 2024",
        "Why {service} is Essential for Every {industry} Business", 
        "5 Signs You Need Professional {service} Right Now",
        "How to Choose the Best {service} Provider Near You",
        "The Hidden Costs of Delaying {service}: What You Need to Know",
        "{service}: Everything You Need to Know Before You Buy",
        "Common {service} Mistakes That Cost Businesses Thousands"
    );
    
    $template = $title_templates[array_rand($title_templates)];
    $title = str_replace(array('{service}', '{industry}'), array($key_service, $industry), $template);
    
    return $title;
}

/**
 * Generate realistic demo content preview
 */
function generate_demo_content_preview($key_service, $industry) {
    return "When it comes to {$key_service}, most {$industry} businesses make critical mistakes that cost them customers and revenue. In this comprehensive guide, we'll walk you through everything you need to know about {$key_service}, including insider tips that can save you thousands of dollars and help you make the right decision for your business...";
}

/**
 * Generate demo image URL (placeholder service)
 */
function generate_demo_image_url($key_service) {
    // Return a realistic placeholder that looks professional
    $width = 800;
    $height = 400;
    $text = urlencode(substr($key_service, 0, 30));
    return "https://via.placeholder.com/{$width}x{$height}/4CAF50/white?text=" . $text;
}

/**
 * Send email verification using real SaaS API
 */
function send_email_verification() {
    $email = sanitize_email($_POST['email']);
    $consent_source = sanitize_text_field($_POST['consent_source'] ?? 'plugin_setup');
    
    if (!is_email($email)) {
        wp_send_json_error('Invalid email address');
    }
    
    // Save email for plugin setup only - no marketing consent here
    update_option('sica_customer_email', $email);
    update_option('sica_consent_source', $consent_source);
    
    // Get client IP and user agent for consent logging
    $client_ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
    
    // Call real email verification API with consent data
    $site_url = get_site_url();
    $response = wp_remote_post('https://api.sica.ai/verification.php', array(
        'headers' => array(
            'Content-Type' => 'application/json'
        ),
        'body' => json_encode(array(
            'action' => 'send_verification',
            'email' => $email,
            'site_url' => $site_url,
            'marketing_consent' => false, // Plugin setup emails never include marketing consent
            'consent_source' => $consent_source,
            'client_ip' => $client_ip,
            'user_agent' => $user_agent,
            'consent_timestamp' => current_time('c')
        )),
        'timeout' => 15
    ));
    
    if (is_wp_error($response)) {
        wp_send_json_error('Failed to send verification email. Please try again.');
        return;
    }
    
    $response_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    $data = json_decode($response_body, true);
    
    if ($response_code === 200 && $data && $data['ok']) {
        wp_send_json_success(array(
            'message' => 'Verification code sent to your email!',
            'email' => $email
        ));
    } else {
        $error_message = $data['error'] ?? 'Failed to send verification email';
        wp_send_json_error($error_message);
    }
}

/**
 * Verify email code
 */
function verify_email_code() {
    $code = sanitize_text_field($_POST['code']);
    $stored_code = get_option('sica_verification_code');
    $verification_time = get_option('sica_verification_time', 0);
    
    // Check if code is expired (10 minutes)
    if (time() - $verification_time > 600) {
        wp_send_json_error('Verification code expired. Please request a new one.');
    }
    
    if ($code === $stored_code) {
        update_option('sica_email_verified', true);
        update_option('sica_license_status', 'active');
        delete_option('sica_verification_code');
        
        wp_send_json_success(array(
            'message' => 'Email verified successfully!',
            'verified' => true
        ));
    } else {
        wp_send_json_error('Invalid verification code. Please try again.');
    }
}

/**
 * Create real post after email verification
 */
function create_real_post() {
    if (!get_option('sica_email_verified', false)) {
        wp_send_json_error('Email not verified');
    }
    
    $key_service = sanitize_text_field($_POST['key_service']);
    $title = sanitize_text_field($_POST['title']);
    
    // This will use the existing real API call system
    $result = generate_and_create_blog_post($title, false);
    
    if ($result) {
        // Mark welcome as completed and show celebration
        update_option('sica_welcome_completed', true);
        
        wp_send_json_success(array(
            'message' => 'Real post created successfully!',
            'post_id' => $result,
            'celebrate' => true
        ));
    } else {
        wp_send_json_error('Failed to create post. Please try again.');
    }
}

/**
 * Complete welcome flow
 */
function complete_welcome_flow() {
    update_option('sica_welcome_completed', true);
    wp_send_json_success(array('message' => 'Welcome flow completed'));
}

/**
 * Add welcome screen to admin menu
 * DISABLED FOR V1 LAUNCH - RE-ENABLE POST-LAUNCH
 */
function add_welcome_menu() {
    if (should_show_welcome_screen()) {
        add_dashboard_page(
            'Welcome to Sica Content AI',
            'Welcome Setup',
            'manage_options',
            'sica-welcome',
            'render_welcome_screen'
        );
    }
}
// Welcome screen enabled for enhanced onboarding
add_action('admin_menu', 'add_welcome_menu');

/**
 * Redirect to welcome screen on first activation
 * DISABLED FOR V1 LAUNCH - RE-ENABLE POST-LAUNCH
 */
function redirect_to_welcome() {
    if (get_transient('sica_activation_redirect')) {
        delete_transient('sica_activation_redirect');
        if (!should_show_welcome_screen()) {
            return; // Already completed
        }
        wp_redirect(admin_url('admin.php?page=sica-welcome'));
        exit;
    }
}
// Auto-redirect to welcome screen on first activation
add_action('admin_init', 'redirect_to_welcome');