<?php
/*
Plugin Name: Sica Content AI for WordPress
Description: Automatic AI-written content that helps small businesses rank in Google and AI search—directly inside WordPress. Professional SEO-focused blog posts with featured images on flexible subscription plans.
Version: 1.1.5
Author: sica.ai - Sica Marketing Inc
Author URI: https://sica.ai
Text Domain: sica-content-ai
Domain Path: /languages
Plugin URI: https://sica.ai/content-ai-wordpress-plugin-for-automatically-generating-seo-and-geo-ranking-content/
Requires at least: WordPress 5.8
Tested up to: WordPress 6.4
Requires PHP: 7.4
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Network: false
*/

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Load text domain for translations
add_action('init', function () {
    load_plugin_textdomain('sica-content-ai', false, dirname(plugin_basename(__FILE__)) . '/languages');
});

// Register hooks for activation and deactivation
register_activation_hook(__FILE__, 'my_auto_blog_activate');
function my_auto_blog_activate() {
    if (function_exists('my_auto_blog_run_activation_tasks')) {
        my_auto_blog_run_activation_tasks();
    }
    
    // Ensure WordPress cron is enabled
    if (!defined('DISABLE_WP_CRON') || DISABLE_WP_CRON !== true) {
        // Check if external cron is configured
        $external_cron = false;
        if (defined('ALTERNATE_WP_CRON') && ALTERNATE_WP_CRON === true) {
            $external_cron = true;
        }
        
        if (!$external_cron) {
            // Suggest setting up real cron
            add_option('sica_suggest_real_cron', true);
            error_log('[SICA Activation] WordPress default cron detected. Consider using system cron for reliability.');
        }
    } else {
        error_log('[SICA Activation] WordPress cron is disabled. Posts will not be auto-generated!');
    }
    
    // After activation, redirect to the Overview tab
    set_transient('sica_activation_redirect_overview', true, 60);
}

// Add admin notice for cron configuration
function sica_admin_cron_notice() {
    if (get_option('sica_suggest_real_cron') && current_user_can('manage_options')) {
        ?>
        <div class="notice notice-warning is-dismissible">
            <p><strong>Sica Content AI:</strong> For more reliable post scheduling, consider configuring a system cron job. Add this to your server's crontab:</p>
            <code>*/15 * * * * wget -q -O /dev/null '<?php echo site_url('wp-cron.php?doing_wp_cron'); ?>' >/dev/null 2>&1</code>
            <p>Then add <code>define('DISABLE_WP_CRON', true);</code> to your wp-config.php</p>
        </div>
        <?php
    }
}
add_action('admin_notices', 'sica_admin_cron_notice');

// Allow users to dismiss the cron notice
function sica_dismiss_cron_notice() {
    if (current_user_can('manage_options')) {
        delete_option('sica_suggest_real_cron');
    }
}
add_action('wp_ajax_sica_dismiss_cron_notice', 'sica_dismiss_cron_notice');

register_deactivation_hook(__FILE__, 'my_auto_blog_deactivate_cron');

// Add settings link on plugin page
$plugin = plugin_basename(__FILE__); 
add_filter("plugin_action_links_$plugin", 'ai_blog_writer_settings_link');

// Include the AI Post Enhancer
require_once plugin_dir_path(__FILE__) . 'ai-post-enhancer.php';

// Include subscription functions (new SaaS system)
require_once plugin_dir_path(__FILE__) . 'includes/subscription-functions.php';
require_once plugin_dir_path(__FILE__) . 'includes/subscription-interface.php';

// Include all refactored component files
require_once plugin_dir_path(__FILE__) . 'includes/admin-interface.php';
require_once plugin_dir_path(__FILE__) . 'includes/content-planner.php';
require_once plugin_dir_path(__FILE__) . 'includes/scheduler.php';
require_once plugin_dir_path(__FILE__) . 'includes/ai-content-generator.php';
require_once plugin_dir_path(__FILE__) . 'includes/image-generator.php';
require_once plugin_dir_path(__FILE__) . 'includes/titles-keywords.php';
require_once plugin_dir_path(__FILE__) . 'includes/prompts-manager.php';
require_once plugin_dir_path(__FILE__) . 'includes/seo-schema.php';
require_once plugin_dir_path(__FILE__) . 'includes/analytics-utilities.php';
require_once plugin_dir_path(__FILE__) . 'includes/welcome-screen.php';
require_once plugin_dir_path(__FILE__) . 'includes/progress-component.php';

// Initialize the AI Post Enhancer
if (class_exists('AI_Post_Enhancer')) {
    new AI_Post_Enhancer();
}

// Add setup tab function (this contains license/subscription management)
function my_auto_blog_setup_tab() {
    if (get_transient('my_auto_blog_settings_updated')) {
        ?>
        <div class="notice notice-success is-dismissible">
            <p><?php echo esc_html__('Setup settings updated successfully.', 'sica-content-ai'); ?></p>
        </div>
        <?php
        delete_transient('my_auto_blog_settings_updated');
    }

    $errors = get_transient('my_auto_blog_errors');
    if ($errors) {
        foreach ($errors as $error) {
            echo '<div class="notice notice-' . esc_attr($error['type']) . ' is-dismissible"><p>' . esc_html($error['message']) . '</p></div>';
        }
        delete_transient('my_auto_blog_errors');
    }

    $api_key = get_api_key();
    $publish_immediately = get_option('publish_immediately');
    $author = get_option('my_auto_blog_author');
    $notify_emails = get_option('my_auto_blog_notify_emails');

    ?>
    <style>
        .sica-account-grid {
            display: grid;
            grid-template-columns: 1fr 300px;
            gap: 20px;
            margin-top: 20px;
        }
        @media (max-width: 1024px) {
            .sica-account-grid {
                grid-template-columns: 1fr;
            }
        }
        .sica-account-card {
            background: #ffffff;
            border: 1px solid #e5e7eb;
            border-radius: 12px;
            padding: 20px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.04);
        }
        .sica-account-card h3 {
            margin: 0 0 16px 0;
            font-size: 18px;
            font-weight: 600;
            color: #111827;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        .sica-plan-icon {
            width: 20px;
            height: 20px;
            border-radius: 50%;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            font-size: 12px;
            font-weight: bold;
            color: white;
        }
        .sica-plan-stats {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin: 12px 0;
            font-size: 14px;
        }
        .sica-progress-bar {
            height: 8px;
            background: #f3f4f6;
            border-radius: 4px;
            overflow: hidden;
            margin: 12px 0;
        }
        .sica-progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #667eea, #764ba2);
            transition: width 0.3s ease;
        }
        .sica-license-key {
            background: #f9fafb;
            border: 1px solid #e5e7eb;
            border-radius: 8px;
            padding: 12px;
            font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, monospace;
            font-size: 13px;
            color: #374151;
            margin: 12px 0;
        }
        .sica-actions {
            display: flex;
            flex-wrap: wrap;
            gap: 8px;
            margin-top: 16px;
        }
        .sica-actions .button {
            border: 1px solid #e5e7eb;
            background: #ffffff;
            color: #374151;
            border-radius: 6px;
            padding: 8px 12px;
            font-size: 13px;
            transition: all 0.2s ease;
        }
        .sica-actions .button:hover {
            border-color: #d1d5db;
            background: #f9fafb;
        }
        .sica-actions .button.button-primary {
            background: #4f46e5;
            border-color: #4f46e5;
            color: #ffffff;
        }
        .sica-actions .button.button-primary:hover {
            background: #4338ca;
            border-color: #4338ca;
        }
        .sica-text-subtle {
            color: #6b7280;
            font-size: 14px;
        }
        .sica-text-small {
            font-size: 12px;
            color: #9ca3af;
        }
        .sica-input {
            width: 100%;
            padding: 12px;
            border: 1px solid #e5e7eb;
            border-radius: 8px;
            font-size: 14px;
            transition: border-color 0.2s ease;
        }
        .sica-input:focus {
            outline: none;
            border-color: #4f46e5;
            box-shadow: 0 0 0 3px rgba(79, 70, 229, 0.1);
        }
    </style>

    <?php
    // Get unified subscription data
    $license_key = get_option('sica_license_key', '');
    $subscription_data = function_exists('sica_check_subscription_status') ? sica_check_subscription_status() : null;
    $customer_email = get_option('sica_customer_email', '');
    $is_email_verified = !empty($customer_email) && get_option('sica_email_verified', false);
    
    // Check for verification mismatch: locally verified but no API token (server lost verification)
    $api_token = get_option('sica_api_token', '');
    $verification_mismatch = $is_email_verified && empty($api_token);
    
    // Auto-fix verification mismatch by clearing stale local verification
    if ($verification_mismatch) {
        update_option('sica_email_verified', false);
        $is_email_verified = false; // Update the variable for current page render
    }
    
    // Determine plan details
    if ($subscription_data && $subscription_data['ok'] && $is_email_verified) {
        $tier = $subscription_data['tier'] ?? 'lite';
        $posts_limit = (int)($subscription_data['posts_limit'] ?? 2);
        $posts_used = (int)($subscription_data['posts_used'] ?? 0);
        $posts_remaining = max(0, $posts_limit - $posts_used);
        $billing_end = !empty($subscription_data['billing_period_end']) ? date('M j, Y', strtotime($subscription_data['billing_period_end'])) : date('M t, Y');
    } else {
        // Fallback to local data
        $license_status = get_option('sica_license_status', 'inactive');
        $license_type = get_option('sica_license_type', 'free');
        $tier = $license_status === 'active' && $license_type === 'paid' ? 'starter' : 'lite';
        
        // Set appropriate limits based on tier
        if ($tier === 'starter') {
            $posts_limit = 12;  // Starter plan gets 12 posts/month
            $posts_used = 0;
            $posts_remaining = 12;
        } else {
            $posts_limit = 2;   // Lite plan gets 2 posts/month
            $posts_used = 0;
            $posts_remaining = 2;
        }
        
        $billing_end = date('M t, Y');
    }
    
    $usage_percentage = $posts_limit > 0 ? min(100, ($posts_used / $posts_limit) * 100) : 0;
    $is_lite = ($tier === 'lite');
    ?>

    <div class="sica-account-grid">
        <div style="display: flex; flex-direction: column; gap: 20px;">
            <!-- Plan Status Card -->
            <div class="sica-account-card">
                <h3>
                    <span class="sica-plan-icon" style="background: <?php echo $is_lite ? '#6b7280' : '#059669'; ?>">
                        <?php echo $is_lite ? '⚡' : '🚀'; ?>
                    </span>
                    <?php echo $is_lite ? esc_html__('Content AI Lite', 'sica-content-ai') : esc_html__('Content AI Starter', 'sica-content-ai'); ?>
                </h3>
                
                <div class="sica-plan-stats">
                    <span class="sica-text-subtle">
                        <?php echo esc_html(sprintf(_n('%d post used', '%d posts used', $posts_used, 'sica-content-ai'), $posts_used)); ?>
                    </span>
                    <span class="sica-text-subtle">
                        <?php echo esc_html(sprintf(_n('%d remaining', '%d remaining', $posts_remaining, 'sica-content-ai'), $posts_remaining)); ?>
                    </span>
                </div>
                
                <div class="sica-progress-bar">
                    <div class="sica-progress-fill" style="width: <?php echo number_format($usage_percentage, 1); ?>%"></div>
                </div>
                
                <div class="sica-text-small">
                    📅 <?php echo esc_html__('Billing period ends:', 'sica-content-ai'); ?> <?php echo esc_html($billing_end); ?>
                </div>
                
                <?php if (!empty($license_key)): ?>
                <div class="sica-license-key" style="user-select: none; -webkit-user-select: none; -moz-user-select: none; -ms-user-select: none; cursor: default;">
                    <?php echo esc_html(substr($license_key, 0, 8) . '••••••••••••••••••••••••' . substr($license_key, -4)); ?>
                </div>
                <?php endif; ?>
                
                <div class="sica-actions">
                    <?php if ($is_lite): ?>
                        <a href="https://sica.ai/pricing" target="_blank" rel="noopener" class="button button-primary">
                            🚀 <?php echo esc_html__('Upgrade Plan', 'sica-content-ai'); ?>
                        </a>
                    <?php else: ?>
                        <a href="https://billing.stripe.com/p/login/7sYcN7dwgdEC71Ea6YcV200" target="_blank" rel="noopener" class="button button-primary">
                            💳 <?php echo esc_html__('Manage Billing', 'sica-content-ai'); ?>
                        </a>
                        <a href="https://sica.ai/pricing" target="_blank" rel="noopener" class="button">
                            📊 <?php echo esc_html__('View Plans', 'sica-content-ai'); ?>
                        </a>
                    <?php endif; ?>
                    
                    <?php if (!empty($license_key)): ?>
                        <button type="button" class="button" onclick="setupReactivateLicense()">
                            🔄 <?php echo esc_html__('Refresh License', 'sica-content-ai'); ?>
                        </button>
                        <button type="button" class="button" onclick="setupUpdateLicense()">
                            ✏️ <?php echo esc_html__('Update Key', 'sica-content-ai'); ?>
                        </button>
                        <button type="button" class="button" onclick="setupDowngradeToFree()" style="color: #dc2626; border-color: #fecaca;">
                            ⬇️ <?php echo esc_html__('Downgrade', 'sica-content-ai'); ?>
                        </button>
                        <button type="button" class="button" onclick="setupDeactivateLicense()" style="color: #dc2626; border-color: #fecaca;">
                            🔄 <?php echo esc_html__('Transfer License', 'sica-content-ai'); ?>
                        </button>
                    <?php else: ?>
                        <!-- Always show "Enter License Key" for users without a license key -->
                        <button type="button" class="button" onclick="setupUpdateLicense()">
                            📄 <?php echo esc_html__('Enter License Key', 'sica-content-ai'); ?>
                        </button>
                    <?php endif; ?>
                    
                    <?php if (defined('WP_DEBUG') && WP_DEBUG): ?>
                        <div style="margin-top:8px; font-size:11px; color:#6b7280; font-family:monospace;">
                            DEBUG: Email=<?php echo esc_html($customer_email ?: 'none'); ?>, Verified=<?php echo $is_email_verified ? 'yes' : 'no'; ?>, License=<?php echo esc_html($license_key ?: 'none'); ?>, Tier=<?php echo esc_html($tier); ?>
                            <br/>API Token=<?php echo !empty($api_token) ? 'present' : 'missing'; ?>
                            <?php if ($verification_mismatch): ?>
                                <br/>⚠️ MISMATCH DETECTED & FIXED: Local verification cleared due to missing API token
                            <?php endif; ?>
                        </div>
                        <button type="button" class="button" onclick="clearCacheDebug()" style="margin-top:8px; font-size:11px; padding:4px 8px; background:#dc2626; color:white; border:none;">
                            🧪 Clear Cache & Test Fixes
                        </button>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Email Verification Card (if needed) -->
            <?php if (!$is_email_verified): ?>
            <div class="sica-account-card">
                <?php if ($verification_mismatch): ?>
                    <div style="background: #fff3cd; border: 1px solid #ffc107; border-radius: 8px; padding: 12px; margin-bottom: 16px;">
                        <strong>⚠️ <?php echo esc_html__('Verification Sync Issue Fixed', 'sica-content-ai'); ?></strong><br>
                        <?php echo esc_html__('Your local verification was out of sync with the server. Please re-verify your email to restore access.', 'sica-content-ai'); ?>
                    </div>
                <?php endif; ?>
                <?php sica_render_subscription_interface(); ?>
            </div>
            <?php endif; ?>

            <!-- Email Management Card -->
            <?php if ($is_email_verified): ?>
            <div class="sica-account-card">
                <h3>📬 <?php echo esc_html__('Email Management', 'sica-content-ai'); ?></h3>
                <p class="sica-text-subtle" style="margin: 0 0 12px 0;">
                    <?php echo esc_html__('Manage your account email and verification status', 'sica-content-ai'); ?>
                </p>
                
                <div style="background: #f9fafb; border: 1px solid #e5e7eb; border-radius: 8px; padding: 12px; margin: 12px 0;">
                    <div style="display: flex; align-items: center; gap: 8px;">
                        <span style="color: #059669;">✅</span>
                        <span style="font-weight: 500;"><?php echo esc_html($customer_email); ?></span>
                        <span class="sica-text-small" style="color: #059669;">(verified)</span>
                    </div>
                </div>
                
                <div class="sica-actions">
                    <button type="button" class="button" onclick="changeEmail()">
                        ✏️ <?php echo esc_html__('Change Email', 'sica-content-ai'); ?>
                    </button>
                    <button type="button" class="button" onclick="removeEmailRegistration()" style="color: #dc2626; border-color: #fecaca;">
                        🗑️ <?php echo esc_html__('Remove Registration', 'sica-content-ai'); ?>
                    </button>
                </div>
            </div>
            <?php endif; ?>

            <!-- Email Notifications Card -->
            <?php if ($is_email_verified): ?>
            <div class="sica-account-card">
                <h3>📧 <?php echo esc_html__('Email Notifications', 'sica-content-ai'); ?></h3>
                <p class="sica-text-subtle" style="margin: 0 0 12px 0;">
                    <?php echo esc_html__('Get notified when new posts are published', 'sica-content-ai'); ?>
                </p>
                
                <div style="margin-top: 12px;">
                    <form autocomplete="off" onsubmit="return false;" style="margin: 0; padding: 0;">
                        <div id="account-notification-chips" style="display:flex; flex-wrap:wrap; gap:6px; padding:8px; border:1px solid #d1d5db; border-radius:6px; background:#fff;">
                            <!-- Chips injected by JS -->
                            <input id="account-notification-input" type="email" placeholder="Add email…" 
                                   autocomplete="new-email" autocapitalize="none" spellcheck="false" 
                                   data-lpignore="true" data-form-type="other" data-1p-ignore data-bwignore
                                   name="notification-email-input" readonly onfocus="this.removeAttribute('readonly');"
                                   style="flex:1; min-width:180px; border:none; outline:none; padding:6px; font-size:13px;" />
                        </div>
                    </form>
                    <div id="account-notification-error" style="color:#dc2626; font-size:12px; margin-top:6px; display:none;"></div>
                    <div style="display:flex; align-items:center; justify-content:space-between; margin-top:10px;">
                        <div style="display:flex; align-items:center; gap:8px;">
                            <span id="account-notification-hint" style="font-size:12px; color:#6b7280;">Press Enter to add; paste multiple to auto-add</span>
                            <span id="account-notification-save-status" style="font-size:12px; color:#6b7280;"></span>
                        </div>
                        <button type="button" class="button button-small" onclick="sicaTestEmailNotifications()" style="font-size:12px;">
                            📧 Test Email
                        </button>
                    </div>
                    
                    <!-- Email Delivery Method Selection -->
                    <div style="margin-top: 16px; padding-top: 16px; border-top: 1px solid #e5e7eb;">
                        <label style="display: block; font-weight: 500; margin-bottom: 8px; font-size: 13px;">
                            📡 Email Delivery Method
                        </label>
                        <select id="email-delivery-method" style="width: 100%; padding: 6px 8px; border: 1px solid #d1d5db; border-radius: 4px; font-size: 13px;">
                            <option value="auto">🚀 Auto (Server → SMTP → WordPress)</option>
                            <option value="server">🌐 Server Only (Most Reliable)</option>
                            <option value="local">🔧 Local SMTP Only</option>
                            <option value="wordpress">⚠️ WordPress Only (Least Reliable)</option>
                        </select>
                        <div style="font-size: 11px; color: #6b7280; margin-top: 4px;">
                            Auto mode tries server first, then falls back to local SMTP, then WordPress mail.
                        </div>
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </div>
    
    <script>
    // Reusable function for showing status messages with nice UI
    function showStatusMessage(message, type = 'loading', targetCard = null) {
        // Remove any existing status div
        const existingDiv = document.getElementById('action-status');
        if (existingDiv) {
            existingDiv.remove();
        }
        
        // Create new status div
        const statusDiv = document.createElement('div');
        statusDiv.id = 'action-status';
        
        // Set style based on type
        let bgColor = '#e7f3ff', borderColor = '#0073aa';
        if (type === 'success') {
            bgColor = '#d4edda'; borderColor = '#28a745';
        } else if (type === 'error') {
            bgColor = '#f8d7da'; borderColor = '#dc3545';
        } else if (type === 'warning') {
            bgColor = '#fff3cd'; borderColor = '#ffc107';
        }
        
        statusDiv.style.cssText = `margin-top: 10px; padding: 12px; background: ${bgColor}; border: 1px solid ${borderColor}; border-radius: 8px; font-size: 14px; transition: all 0.3s ease;`;
        statusDiv.innerHTML = message;
        
        // Find the appropriate card to append the status message
        let targetElement = targetCard;
        if (!targetElement) {
            // Default: try to find the plan card (first card)
            targetElement = document.querySelector('.sica-account-card');
        }
        
        if (targetElement) {
            targetElement.appendChild(statusDiv);
        } else {
            // Fallback: show alert if card not found
            alert(message.replace(/<[^>]*>/g, '')); // Strip HTML tags for alert
        }
        
        return statusDiv;
    }

    function setupReactivateLicense() {
        const hasKey = <?php echo !empty($license_key) ? 'true' : 'false'; ?>;
        if (!hasKey) {
            showStatusMessage('❌ <?php echo esc_js(__('No license key found. Please enter a license key first.', 'sica-content-ai')); ?>', 'error');
            return;
        }
        
        // Show confirmation with nice UI
        showStatusMessage('🔄 <?php echo esc_js(__('Refresh your license status from the server? This will revalidate your license key and update your subscription data.', 'sica-content-ai')); ?><br><div style="margin-top:8px;"><button onclick="confirmReactivate()" style="margin-right:8px; padding:6px 12px; background:#28a745; color:white; border:none; border-radius:4px; cursor:pointer;">🔄 Refresh License</button><button onclick="cancelAction()" style="padding:6px 12px; background:#6c757d; color:white; border:none; border-radius:4px; cursor:pointer;">❌ Cancel</button></div>', 'warning');
    }
    
    function confirmReactivate() {
        // Use server-side reactivation without exposing the key to JavaScript
        reactivateStoredLicense();
    }
    
    function reactivateStoredLicense() {
        // Show loading in button
        const buttons = document.querySelectorAll('button[onclick*="License"]');
        buttons.forEach(btn => {
            btn.disabled = true;
            btn.style.opacity = '0.6';
        });
        
        // Show loading message using reusable function
        showStatusMessage('⏳ <?php echo esc_js(__('Refreshing license...', 'sica-content-ai')); ?>', 'loading');
        
        // Make AJAX call to reactivate using stored license key
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'reactivate_stored_license',
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(response => {
            console.log('License reactivation response status:', response.status);
            return response.json();
        })
        .then(data => {
            console.log('License reactivation response data:', data);
            
            if (data.success) {
                showStatusMessage('✅ <?php echo esc_js(__('License refreshed successfully! Refreshing page...', 'sica-content-ai')); ?>', 'success');
                
                setTimeout(() => {
                    window.location.reload();
                }, 2000);
            } else {
                showStatusMessage('❌ <?php echo esc_js(__('License refresh failed: ', 'sica-content-ai')); ?>' + (data.data || '<?php echo esc_js(__('Unable to refresh license', 'sica-content-ai')); ?>'), 'error');
                
                // Re-enable buttons
                buttons.forEach(btn => {
                    btn.disabled = false;
                    btn.style.opacity = '1';
                });
            }
        })
        .catch(error => {
            console.error('License reactivation error:', error);
            showStatusMessage('❌ <?php echo esc_js(__('Network error occurred while refreshing license', 'sica-content-ai')); ?>', 'error');
            
            // Re-enable buttons
            buttons.forEach(btn => {
                btn.disabled = false;
                btn.style.opacity = '1';
            });
        });
    }
    
    function cancelAction() {
        const statusDiv = document.getElementById('action-status');
        if (statusDiv) {
            statusDiv.remove();
        }
    }
    
    function setupUpdateLicense() {
        const hasCurrentKey = <?php echo !empty($license_key) ? 'true' : 'false'; ?>;
        const displayKey = hasCurrentKey ? '<?php echo esc_js(substr($license_key, 0, 8) . '••••••••••••••••••••••••' . substr($license_key, -4)); ?>' : '';
        
        let message = <?php echo json_encode(__('Enter your license key:', 'sica-content-ai')); ?>;
        if (hasCurrentKey) {
            message = `<?php echo esc_js(__('Current license:', 'sica-content-ai')); ?> ${displayKey}\n\n<?php echo esc_js(__('Enter new license key (or press Cancel to keep current):', 'sica-content-ai')); ?>`;
        }
        
        const newKey = prompt(message);
        if (!newKey || !newKey.trim()) {
            return;
        }
        
        setupActivateLicense(newKey.trim());
    }

    function setupDowngradeToFree() {
        // Show confirmation with nice UI
        showStatusMessage('⬇️ <?php echo esc_js(__('Remove your license key and downgrade to the Free plan? You can re-activate with a key later.', 'sica-content-ai')); ?><br><div style="margin-top:8px;"><button onclick="confirmDowngrade()" style="margin-right:8px; padding:6px 12px; background:#dc3545; color:white; border:none; border-radius:4px; cursor:pointer;">⬇️ Downgrade</button><button onclick="cancelAction()" style="padding:6px 12px; background:#6c757d; color:white; border:none; border-radius:4px; cursor:pointer;">❌ Cancel</button></div>', 'warning');
    }
    
    function setupDeactivateLicense() {
        // Show explanation with nice UI
        showStatusMessage('🔄 <?php echo esc_js(__('Transfer License to Different Site', 'sica-content-ai')); ?><br><?php echo esc_js(__('This will deactivate your license on this site so you can activate it on a different site. This action cannot be undone automatically.', 'sica-content-ai')); ?><br><div style="margin-top:8px;"><button onclick="confirmDeactivate()" style="margin-right:8px; padding:6px 12px; background:#dc3545; color:white; border:none; border-radius:4px; cursor:pointer;">🔄 Transfer License</button><button onclick="cancelAction()" style="padding:6px 12px; background:#6c757d; color:white; border:none; border-radius:4px; cursor:pointer;">❌ Cancel</button></div>', 'warning');
    }
    
    function confirmDowngrade() {
        showStatusMessage('⏳ <?php echo esc_js(__('Downgrading to free plan...', 'sica-content-ai')); ?>', 'loading');
        
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: new URLSearchParams({
                action: 'sica_downgrade_to_free',
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(r => r.json())
        .then(data => {
            if (data && data.success) {
                showStatusMessage('✅ <?php echo esc_js(__('Downgraded to free plan successfully! Reloading...', 'sica-content-ai')); ?>', 'success');
                setTimeout(() => {
                    window.location.reload();
                }, 2000);
            } else {
                showStatusMessage('❌ <?php echo esc_js(__('Failed to downgrade: ', 'sica-content-ai')); ?>' + (data && data.data ? data.data : '<?php echo esc_js(__('Unknown error', 'sica-content-ai')); ?>'), 'error');
            }
        })
        .catch(e => {
            showStatusMessage('❌ <?php echo esc_js(__('Network error: ', 'sica-content-ai')); ?>' + e.message, 'error');
        });
    }
    
    function confirmDeactivate() {
        showStatusMessage('⏳ <?php echo esc_js(__('Requesting license deactivation...', 'sica-content-ai')); ?>', 'loading');
        
        // First request without confirmation to get details
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: new URLSearchParams({
                action: 'deactivate_license',
                confirm: 'false',
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(r => r.json())
        .then(data => {
            if (!data.success && data.data && data.data.needs_confirmation) {
                // Show final confirmation with details
                const confirmMsg = `⚠️ ${data.data.message}<br><br><strong>Current site:</strong> ${data.data.current_site}<br><strong>Warning:</strong> ${data.data.warning}<br><br><div style="margin-top:8px;"><button onclick="finalDeactivate()" style="margin-right:8px; padding:6px 12px; background:#dc3545; color:white; border:none; border-radius:4px; cursor:pointer;">✅ Yes, Deactivate</button><button onclick="cancelAction()" style="padding:6px 12px; background:#6c757d; color:white; border:none; border-radius:4px; cursor:pointer;">❌ Cancel</button></div>`;
                showStatusMessage(confirmMsg, 'warning');
            } else {
                showStatusMessage('❌ <?php echo esc_js(__('Deactivation failed: ', 'sica-content-ai')); ?>' + (data.data || '<?php echo esc_js(__('Unable to process request', 'sica-content-ai')); ?>'), 'error');
            }
        })
        .catch(error => {
            console.error('Deactivation error:', error);
            showStatusMessage('❌ <?php echo esc_js(__('Network error occurred', 'sica-content-ai')); ?>', 'error');
        });
    }
    
    function finalDeactivate() {
        showStatusMessage('⏳ <?php echo esc_js(__('Deactivating license...', 'sica-content-ai')); ?>', 'loading');
        
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: new URLSearchParams({
                action: 'deactivate_license',
                confirm: 'true',
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                showStatusMessage('✅ <?php echo esc_js(__('License successfully deactivated! You can now activate it on a different site. Refreshing page...', 'sica-content-ai')); ?>', 'success');
                setTimeout(() => {
                    window.location.reload();
                }, 3000);
            } else {
                showStatusMessage('❌ <?php echo esc_js(__('Deactivation failed: ', 'sica-content-ai')); ?>' + (data.data || '<?php echo esc_js(__('Unknown error', 'sica-content-ai')); ?>'), 'error');
            }
        })
        .catch(error => {
            console.error('Final deactivation error:', error);
            showStatusMessage('❌ <?php echo esc_js(__('Network error occurred during deactivation', 'sica-content-ai')); ?>', 'error');
        });
    }
    
    function setupActivateLicense(licenseKey) {
        // Show loading in button
        const buttons = document.querySelectorAll('button[onclick*="License"]');
        buttons.forEach(btn => {
            btn.disabled = true;
            btn.style.opacity = '0.6';
        });
        
        // Show loading message using reusable function
        showStatusMessage('⏳ <?php echo esc_js(__('Activating license...', 'sica-content-ai')); ?>', 'loading');
        
        // Make AJAX call
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'activate_license_key',
                license_key: licenseKey,
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(response => {
            console.log('License activation response status:', response.status);
            return response.json();
        })
        .then(data => {
            console.log('License activation response data:', data);
            
            if (data.success) {
                showStatusMessage('✅ <?php echo esc_js(__('License activated successfully! Refreshing page...', 'sica-content-ai')); ?>', 'success');
                
                setTimeout(() => {
                    window.location.reload();
                }, 2000);
            } else {
                showStatusMessage('❌ <?php echo esc_js(__('License activation failed: ', 'sica-content-ai')); ?>' + (data.data || '<?php echo esc_js(__('Invalid license key', 'sica-content-ai')); ?>'), 'error');
                
                // Re-enable buttons
                buttons.forEach(btn => {
                    btn.disabled = false;
                    btn.style.opacity = '1';
                });
            }
        })
        .catch(error => {
            console.error('License activation error:', error);
            showStatusMessage('❌ <?php echo esc_js(__('Network error: ', 'sica-content-ai')); ?>' + error.message, 'error');
            
            // Re-enable buttons
            buttons.forEach(btn => {
                btn.disabled = false;
                btn.style.opacity = '1';
            });
        });
    }
    
    function forceRefreshSubscription() {
        // Show confirmation with nice UI
        showStatusMessage('🔄 <?php echo esc_js(__('Force refresh subscription status from server? This will clear all caches and fetch fresh data.', 'sica-content-ai')); ?><br><div style="margin-top:8px;"><button onclick="confirmRefresh()" style="margin-right:8px; padding:6px 12px; background:#0073aa; color:white; border:none; border-radius:4px; cursor:pointer;">🔄 Refresh</button><button onclick="cancelAction()" style="padding:6px 12px; background:#6c757d; color:white; border:none; border-radius:4px; cursor:pointer;">❌ Cancel</button></div>', 'warning');
    }
    
    function confirmRefresh() {
        showStatusMessage('⏳ <?php echo esc_js(__('Refreshing subscription status...', 'sica-content-ai')); ?>', 'loading');
        
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: new URLSearchParams({
                action: 'force_refresh_subscription',
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(r => r.json())
        .then(resp => {
            if (resp.success) {
                showStatusMessage('✅ <?php echo esc_js(__('Subscription status refreshed successfully! Reloading...', 'sica-content-ai')); ?>', 'success');
                setTimeout(() => {
                    window.location.reload();
                }, 2000);
            } else {
                showStatusMessage('❌ <?php echo esc_js(__('Failed to refresh: ', 'sica-content-ai')); ?>' + (resp.data?.message || '<?php echo esc_js(__('Unknown error', 'sica-content-ai')); ?>'), 'error');
                console.error('Refresh error:', resp);
            }
        })
        .catch(err => {
            showStatusMessage('❌ <?php echo esc_js(__('Network error during refresh. Check console for details.', 'sica-content-ai')); ?>', 'error');
            console.error('Refresh network error:', err);
        });
    }

    function changeEmail() {
        const currentEmail = '<?php echo esc_js($customer_email); ?>';
        // Find the email management card specifically
        const emailCard = Array.from(document.querySelectorAll('.sica-account-card')).find(card => 
            card.querySelector('h3') && card.querySelector('h3').textContent.includes('Email Management')
        );
        
        showStatusMessage('✏️ <?php echo esc_js(__('Change your account email address? You will need to verify the new email.', 'sica-content-ai')); ?><br><div style="margin-top:8px;"><input type="email" id="new-email-input" placeholder="Enter new email address" autocomplete="off" autocapitalize="none" spellcheck="false" data-lpignore="true" data-form-type="other" style="width:100%; padding:8px; border:1px solid #e5e7eb; border-radius:4px; margin-bottom:8px;" value="' + currentEmail + '"><br><button onclick="confirmChangeEmail()" style="margin-right:8px; padding:6px 12px; background:#0073aa; color:white; border:none; border-radius:4px; cursor:pointer;">✏️ Change Email</button><button onclick="cancelAction()" style="padding:6px 12px; background:#6c757d; color:white; border:none; border-radius:4px; cursor:pointer;">❌ Cancel</button></div>', 'warning', emailCard);
        
        // Focus the input after a brief delay
        setTimeout(() => {
            const input = document.getElementById('new-email-input');
            if (input) input.focus();
        }, 100);
    }

    function confirmChangeEmail() {
        const newEmail = document.getElementById('new-email-input')?.value?.trim();
        const emailCard = Array.from(document.querySelectorAll('.sica-account-card')).find(card => 
            card.querySelector('h3') && card.querySelector('h3').textContent.includes('Email Management')
        );
        
        if (!newEmail) {
            showStatusMessage('❌ <?php echo esc_js(__('Please enter a valid email address.', 'sica-content-ai')); ?>', 'error', emailCard);
            return;
        }
        
        // Simple email validation
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(newEmail)) {
            showStatusMessage('❌ <?php echo esc_js(__('Please enter a valid email address.', 'sica-content-ai')); ?>', 'error', emailCard);
            return;
        }
        
        showStatusMessage('⏳ <?php echo esc_js(__('Updating email address...', 'sica-content-ai')); ?>', 'loading', emailCard);
        
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: new URLSearchParams({
                action: 'change_customer_email',
                new_email: newEmail,
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                showStatusMessage('✅ <?php echo esc_js(__('Email updated successfully! Please check your new email for verification. Reloading...', 'sica-content-ai')); ?>', 'success', emailCard);
                setTimeout(() => {
                    window.location.reload();
                }, 3000);
            } else {
                showStatusMessage('❌ <?php echo esc_js(__('Failed to update email: ', 'sica-content-ai')); ?>' + (data.data || '<?php echo esc_js(__('Unknown error', 'sica-content-ai')); ?>'), 'error', emailCard);
            }
        })
        .catch(e => {
            showStatusMessage('❌ <?php echo esc_js(__('Network error: ', 'sica-content-ai')); ?>' + e.message, 'error', emailCard);
        });
    }

    function removeEmailRegistration() {
        // Find the email management card specifically
        const emailCard = Array.from(document.querySelectorAll('.sica-account-card')).find(card => 
            card.querySelector('h3') && card.querySelector('h3').textContent.includes('Email Management')
        );
        
        showStatusMessage('🗑️ <?php echo esc_js(__('Remove your email registration? This will reset you to unverified status and you will need to re-verify to post content.', 'sica-content-ai')); ?><br><div style="margin-top:8px;"><button onclick="confirmRemoveEmail()" style="margin-right:8px; padding:6px 12px; background:#dc3545; color:white; border:none; border-radius:4px; cursor:pointer;">🗑️ Remove Registration</button><button onclick="cancelAction()" style="padding:6px 12px; background:#6c757d; color:white; border:none; border-radius:4px; cursor:pointer;">❌ Cancel</button></div>', 'warning', emailCard);
    }

    function confirmRemoveEmail() {
        const emailCard = Array.from(document.querySelectorAll('.sica-account-card')).find(card => 
            card.querySelector('h3') && card.querySelector('h3').textContent.includes('Email Management')
        );
        
        showStatusMessage('⏳ <?php echo esc_js(__('Removing email registration...', 'sica-content-ai')); ?>', 'loading', emailCard);
        
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: new URLSearchParams({
                action: 'remove_email_registration',
                _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
            })
        })
        .then(r => r.json())
        .then(data => {
            if (data.success) {
                showStatusMessage('✅ <?php echo esc_js(__('Email registration removed successfully! Reloading...', 'sica-content-ai')); ?>', 'success', emailCard);
                setTimeout(() => {
                    window.location.reload();
                }, 2000);
            } else {
                showStatusMessage('❌ <?php echo esc_js(__('Failed to remove email: ', 'sica-content-ai')); ?>' + (data.data || '<?php echo esc_js(__('Unknown error', 'sica-content-ai')); ?>'), 'error', emailCard);
            }
        })
        .catch(e => {
            showStatusMessage('❌ <?php echo esc_js(__('Network error: ', 'sica-content-ai')); ?>' + e.message, 'error', emailCard);
        });
    }

    // Account Settings: Email notification chips functionality
    (function initAccountNotificationChips() {
        const stored = '<?php echo esc_js($notify_emails); ?>';
        const initialEmails = stored ? stored.split(',').map(e => e.trim()).filter(Boolean) : [];
        const chipsContainer = document.getElementById('account-notification-chips');
        const input = document.getElementById('account-notification-input');
        const errorEl = document.getElementById('account-notification-error');
        const statusEl = document.getElementById('account-notification-save-status');

        if (!chipsContainer || !input) return;

        function isValidEmail(email) {
            return /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
        }

        function renderChip(email) {
            const chip = document.createElement('span');
            chip.className = 'chip';
            chip.setAttribute('data-email', email);
            chip.style.cssText = 'display:inline-flex; align-items:center; gap:6px; padding:4px 8px; background:#f3f4f6; color:#374151; border:1px solid #e5e7eb; border-radius:999px; font-size:12px;';
            chip.innerHTML = `<span>${email}</span><button type="button" aria-label="Remove" style="border:none;background:transparent;color:#6b7280;cursor:pointer;padding:0;line-height:1;">×</button>`;
            chip.querySelector('button').addEventListener('click', () => {
                chipsContainer.removeChild(chip);
                saveAccountNotificationEmails();
                input.focus();
            });
            chipsContainer.insertBefore(chip, input);
        }

        function addEmail(email) {
            const normalized = email.trim();
            if (!normalized) return;
            if (!isValidEmail(normalized)) {
                errorEl.textContent = `Invalid email: ${normalized}`;
                errorEl.style.display = 'block';
                return;
            }
            if (Array.from(chipsContainer.querySelectorAll('.chip')).some(c => c.getAttribute('data-email') === normalized)) {
                errorEl.textContent = `Email already added: ${normalized}`;
                errorEl.style.display = 'block';
                return;
            }
            errorEl.style.display = 'none';
            renderChip(normalized);
            saveAccountNotificationEmails();
        }

        function saveAccountNotificationEmails() {
            const emails = Array.from(chipsContainer.querySelectorAll('.chip'))
                .map(chip => chip.getAttribute('data-email'))
                .join(',');
            
            if (statusEl) {
                statusEl.textContent = 'Saving…';
            }
            
            fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: new URLSearchParams({
                    action: 'save_notification_emails',
                    emails: emails,
                    _ajax_nonce: '<?php echo wp_create_nonce('my_auto_blog_nonce'); ?>'
                })
            })
            .then(r => r.json())
            .then(resp => {
                if (statusEl) {
                    statusEl.textContent = resp.success ? resp.data : 'Save failed';
                    setTimeout(() => { statusEl.textContent = ''; }, 2000);
                }
            })
            .catch(err => {
                if (statusEl) {
                    statusEl.textContent = 'Network error';
                    setTimeout(() => { statusEl.textContent = ''; }, 2000);
                }
            });
        }

        // Event listeners
        input.addEventListener('keydown', (e) => {
            if (e.key === 'Enter' || e.key === ',') {
                e.preventDefault();
                addEmail(input.value);
                input.value = '';
            }
        });

        input.addEventListener('paste', (e) => {
            e.preventDefault();
            const pastedText = (e.clipboardData || window.clipboardData).getData('text');
            const emails = pastedText.split(/[,;\n\t\s]+/).filter(Boolean);
            emails.forEach(addEmail);
            input.value = '';
        });

        // Initialize with stored emails
        initialEmails.forEach(renderChip);
    })();

    // Email delivery method management
    (function initEmailDeliveryMethod() {
        const select = document.getElementById('email-delivery-method');
        if (!select) return;
        
        // Load current setting
        const currentMethod = '<?php echo esc_js(get_option('sica_email_delivery_method', 'auto')); ?>';
        select.value = currentMethod;
        
        // Save on change
        select.addEventListener('change', function() {
            const method = this.value;
            
            fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: new URLSearchParams({
                    action: 'sica_save_email_delivery_method',
                    method: method,
                    _ajax_nonce: '<?php echo wp_create_nonce('sica_email_method_nonce'); ?>'
                })
            })
            .then(r => r.json())
            .then(resp => {
                if (resp.success) {
                    // Show brief success indicator
                    const originalBorder = this.style.borderColor;
                    this.style.borderColor = '#00a32a';
                    this.style.background = '#f0f9ff';
                    
                    setTimeout(() => {
                        this.style.borderColor = originalBorder;
                        this.style.background = '';
                    }, 1000);
                    
                    console.log('Email delivery method updated to:', method);
                } else {
                    alert('Failed to save email delivery method');
                }
            })
            .catch(err => {
                console.error('Error saving email delivery method:', err);
                alert('Network error while saving email delivery method');
            });
        });
    })();

    // Test email notifications function
    function sicaTestEmailNotifications() {
        const button = event.target;
        const originalText = button.textContent;
        button.disabled = true;
        button.textContent = 'Testing...';
        
        const formData = new FormData();
        formData.append('action', 'sica_test_email_notifications');
        formData.append('_ajax_nonce', '<?php echo wp_create_nonce('sica_test_email_nonce'); ?>');
        
        fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
            method: 'POST',
            body: formData
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                button.textContent = '✅ Sent!';
                button.style.background = '#00a32a';
                button.style.color = '#fff';
                
                // Show success message in the email card
                const emailCard = button.closest('.sica-account-card');
                showStatusMessage('✅ Test email sent successfully! Check your inbox and error logs for delivery details.', 'success', emailCard);
                
                // Reset button after 3 seconds
                setTimeout(() => {
                    button.disabled = false;
                    button.textContent = originalText;
                    button.style.background = '';
                    button.style.color = '';
                }, 3000);
            } else {
                button.textContent = '❌ Failed';
                button.style.background = '#d63638';
                button.style.color = '#fff';
                
                // Show error message in the email card
                const emailCard = button.closest('.sica-account-card');
                showStatusMessage('❌ Test email failed: ' + (data.data || 'Unknown error. Check error logs for details.'), 'error', emailCard);
                
                // Reset button after 3 seconds
                setTimeout(() => {
                    button.disabled = false;
                    button.textContent = originalText;
                    button.style.background = '';
                    button.style.color = '';
                }, 3000);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            button.disabled = false;
            button.textContent = originalText;
            
            // Show network error message
            const emailCard = button.closest('.sica-account-card');
            showStatusMessage('❌ Network error occurred while testing email', 'error', emailCard);
        });
    }
    
    // Debug function to clear cache and test fixes
    function clearCacheDebug() {
        const button = event.target;
        const originalText = button.textContent;
        
        button.disabled = true;
        button.textContent = '🔄 Clearing...';
        button.style.background = '#f59e0b';
        
        // Call the AJAX handler to clear cache
        fetch(ajaxurl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'sica_clear_cache_debug',
                _ajax_nonce: '<?php echo wp_create_nonce('sica_clear_cache_debug'); ?>'
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                button.textContent = '✅ Cleared!';
                button.style.background = '#10b981';
                
                // Show detailed results
                showStatusMessage(
                    '🧪 Cache cleared successfully!<br/>' +
                    '<strong>Cleared:</strong> API token, subscription cache, month cache<br/>' +
                    '<strong>Next:</strong> This page will reload to test fixes...',
                    'success'
                );
                
                // Reload page after 2 seconds to test the fixes
                setTimeout(() => {
                    window.location.reload();
                }, 2000);
            } else {
                button.textContent = '❌ Failed';
                button.style.background = '#dc2626';
                showStatusMessage('❌ Failed to clear cache: ' + (data.data || 'Unknown error'), 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            button.textContent = '❌ Error';
            button.style.background = '#dc2626';
            showStatusMessage('❌ Network error occurred while clearing cache', 'error');
        })
        .finally(() => {
            // Reset button after 3 seconds
            setTimeout(() => {
                button.disabled = false;
                button.textContent = originalText;
                button.style.background = '#dc2626';
            }, 3000);
        });
    }
    </script>
    
    <?php
}

// Function to save setup settings
function save_setup_section_settings() {
    if (!current_user_can('manage_options')) {
        wp_die('Unauthorized user');
    }

    check_admin_referer('save_setup_settings_nonce', 'setup_nonce');

    // Save publishing settings
    $publish_immediately = isset($_POST['publish_immediately']) ? 1 : 0;
    update_option('publish_immediately', $publish_immediately);

    // Save author
    if (isset($_POST['my_auto_blog_author'])) {
        update_option('my_auto_blog_author', intval($_POST['my_auto_blog_author']));
    }

    // Save notification emails
    if (isset($_POST['my_auto_blog_notify_emails'])) {
        $emails = sanitize_text_field($_POST['my_auto_blog_notify_emails']);
        update_option('my_auto_blog_notify_emails', $emails);
    }



    // Set success transient
    set_transient('my_auto_blog_settings_updated', true, 30);

    // Redirect back with success parameter
    wp_redirect(add_query_arg([
        'page' => 'ai-blogger',
        'tab' => 'setup',
        'settings-updated' => 'true',
        'message' => 'setup'
    ], admin_url('admin.php')));
    exit;
}
add_action('admin_post_save_setup_settings', 'save_setup_section_settings');

// AJAX handler for saving email delivery method
function sica_ajax_save_email_delivery_method() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    check_ajax_referer('sica_email_method_nonce', '_ajax_nonce');
    
    $method = sanitize_text_field($_POST['method']);
    $valid_methods = array('auto', 'server', 'local', 'wordpress');
    
    if (!in_array($method, $valid_methods)) {
        wp_send_json_error('Invalid email delivery method');
        return;
    }
    
    update_option('sica_email_delivery_method', $method);
    error_log('[SICA Email] Email delivery method updated to: ' . $method);
    
    wp_send_json_success('Email delivery method updated to: ' . $method);
}
add_action('wp_ajax_sica_save_email_delivery_method', 'sica_ajax_save_email_delivery_method');

/**
 * AJAX handler to clear cache for debugging (dev mode only)
 */
function sica_ajax_clear_cache_debug() {
    // Security check
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions');
        return;
    }
    
    // Verify nonce
    if (!check_ajax_referer('sica_clear_cache_debug', '_ajax_nonce', false)) {
        wp_send_json_error('Security check failed');
        return;
    }
    
    // Only allow in debug mode
    if (!(defined('WP_DEBUG') && WP_DEBUG)) {
        wp_send_json_error('Debug mode not enabled');
        return;
    }
    
    try {
        // Clear all caches and force token refresh
        $cleared = [];
        
        if (get_option('sica_api_token')) {
            delete_option('sica_api_token');
            $cleared[] = 'API token';
        }
        
        if (get_transient('sica_subscription_status')) {
            delete_transient('sica_subscription_status');
            $cleared[] = 'Subscription cache';
        }
        
        if (get_option('sica_cached_month')) {
            update_option('sica_cached_month', '');
            $cleared[] = 'Month cache';
        }
        
        // Log for debugging
        error_log('SICA DEBUG: Cache cleared by user - ' . implode(', ', $cleared));
        
        wp_send_json_success([
            'cleared' => $cleared,
            'message' => 'Cache cleared successfully. Page will reload to test fixes.'
        ]);
        
    } catch (Exception $e) {
        error_log('SICA DEBUG: Error clearing cache - ' . $e->getMessage());
        wp_send_json_error('Error clearing cache: ' . $e->getMessage());
    }
}
add_action('wp_ajax_sica_clear_cache_debug', 'sica_ajax_clear_cache_debug');

// Note: Heartbeat functions removed as they were not implemented

/**
 * Check production readiness and log debug status
 */
function sica_check_production_readiness() {
    $wp_debug = defined('WP_DEBUG') && WP_DEBUG;
    $is_production_ready = !$wp_debug;
    
    error_log('SICA Production Readiness Check:');
    error_log('- WP_DEBUG: ' . ($wp_debug ? 'ENABLED (should be false for production)' : 'disabled ✓'));
    error_log('- Production Ready: ' . ($is_production_ready ? 'YES ✓' : 'NO - debug features visible'));
    
    return $is_production_ready;
}

// Add admin notice if not production ready (only for admins)
add_action('admin_notices', function() {
    if (!current_user_can('manage_options')) return;
    
    $wp_debug = defined('WP_DEBUG') && WP_DEBUG;
    
    if ($wp_debug) {
        echo '<div class="notice notice-warning is-dismissible">';
        echo '<p><strong>SICA Content AI:</strong> Debug mode is active. Debug information and developer tools are visible. ';
        echo 'Set <code>WP_DEBUG</code> to <code>false</code> for production.</p></div>';
    }
});

// Redirect to plugin Overview tab immediately after activation
add_action('admin_init', function() {
    if (!is_admin()) return;
    if (!current_user_can('activate_plugins')) return;
    if (get_transient('sica_activation_redirect_overview')) {
        delete_transient('sica_activation_redirect_overview');
        if (!isset($_GET['activate-multi'])) {
            wp_safe_redirect(admin_url('admin.php?page=ai-blogger&tab=overview'));
            exit;
        }
    }
});
